import http.client
import logging

from mongoengine import NotUniqueError, ValidationError

from walle.errors import BadRequestError
from walle.physical_location_tree import LocationNamesMap
from walle.util.api import api_handler, api_response, admin_request

log = logging.getLogger(__name__)


@api_handler(
    "/shortnames",
    "POST",
    {
        "type": "object",
        "properties": {
            "shortnames": {
                "type": "array",
                "items": {
                    "type": "object",
                    "properties": {
                        "path": {"type": "string"},
                        "name": {"type": "string", "pattern": "^[a-zA-Z0-9-]+$"},
                    },
                    "description": "Object to add by path and name",
                    "required": ["name", "path"],
                },
            }
        },
        "required": ["shortnames"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
)
@admin_request
def add_with_name_and_path(request, reason, issuer):
    for item in request["shortnames"]:
        try:
            LocationNamesMap(**item).save()
        except NotUniqueError:
            raise BadRequestError(
                "Failed to add name for path '{}': '{}' is not unique name.", item["path"], item["name"]
            )
        except ValidationError as e:
            raise BadRequestError(e.errors)
    return api_response(request)


@api_handler(
    "/shortnames/<path>",
    "PUT",
    {
        "name": {"type": "string", "description": "Add custom location path for name", "pattern": "^[a-zA-Z0-9-]+$"},
        "required": ["name"],
    },
    authenticate=True,
)
@admin_request
def add_with_path(path, request, issuer):
    try:
        LocationNamesMap(path=path, name=request.get("name")).save()
    except NotUniqueError:
        raise BadRequestError("Failed to add name for path '{}': '{}' is not unique name.", path, request["name"])
    except ValidationError as e:
        raise BadRequestError(e.errors)
    return api_response({"path": path, "name": request["name"]})


@api_handler("/shortnames/<path>", "DELETE", authenticate=True)
@admin_request
def delete_with_name(path, issuer):
    LocationNamesMap.objects(path=path).delete()
    return "", http.client.NO_CONTENT


@api_handler("/shortnames/<path>", "GET", authenticate=True)
@admin_request
def get_one_shortname(path, issuer):
    name = LocationNamesMap.objects(path=path).only("name")[0].name
    return api_response(name)


@api_handler("/shortnames", "GET")
def get_shortnames():
    location_map = LocationNamesMap.get_map()
    location_map = [{"path": path, "name": location_map[path]} for path in sorted(location_map.keys())]
    return api_response(location_map)
