from walle import constants
from walle.authorization import iam
from walle.errors import NoSchemaForCheckError
from walle.expert import juggler
from walle.expert.types import ALL_STATUSES
from walle.host_health import HealthCheck
from walle.models import DocumentPostprocessor
from walle.util.api import api_handler, get_query_result, api_response, FilterQueryParser
from walle.util.gevent_tools import gevent_idle_iter


class HealthCheckPostprocessor(DocumentPostprocessor):
    def __init__(self):
        extra_db_fields = {HealthCheck.metadata.db_field}
        extra_fields = {"metadata"}
        super().__init__(extra_db_fields, extra_fields)

    def process(self, iterable, requested_fields):
        checks = []

        for check in gevent_idle_iter(iterable):
            object = check.to_api_object_shallow(requested_fields)
            object['metadata'] = juggler.decode_metadata(check.metadata)
            checks.append(object)

        return checks


def get_health_checks_scheme():
    return {
        "fqdn": {"type": "array", "items": {"type": "string", "description": "Filter by check fqdn substring"}},
        "type": {"type": "array", "items": {"type": "string", "description": "Filter by check type substring"}},
        "status": {"type": "array", "items": {"type": "string", "description": "Filter by check status substring"}},
        "timestamp": {
            "type": "array",
            "items": {"type": "integer", "description": "Filter by check timestamp substring"},
        },
    }


@api_handler(
    "/health-checks",
    "GET",
    params=get_health_checks_scheme(),
    with_fields=HealthCheck,
    with_paging={
        "offset": {"type": "integer", "minimum": 0, "description": "Offset to return $limit entries"},
        "limit": {"type": "integer", "description": "Max number of entries per page"},
        "max_limit": 10000,
    },
    iam_permissions=iam.GetHostApiIamPermission("fqdn"),
)
def get_health_checks(query_args):
    query_parser = FilterQueryParser(HealthCheck, enum_fields=("fqdn", "type", "status", "timestamp"))
    query = query_parser.parse_query_filter(query_args)
    postprocessor = HealthCheckPostprocessor()
    return api_response(get_query_result(HealthCheck, query, HealthCheck.id, query_args, postprocessor=postprocessor))


@api_handler(
    "/scheme-validators",
    "POST",
    {
        "type": "object",
        "properties": dict(
            {
                "status": {"enum": list(ALL_STATUSES.keys())},
                "metadata": {"type": "object", "properties": dict()},
                "check_type": {"type": "string"},
            }
        ),
        "required": ["status", "metadata", "check_type"],
        "additionalProperties": False,
    },
)
def check_metadata(request):
    cur_type = request["check_type"]
    cur_status = ALL_STATUSES[request["status"]]
    try:
        validator, preprocessor = juggler.get_metadata_validator(cur_type, cur_status)
        if validator is None:
            raise NoSchemaForCheckError

        juggler.validate_metadata_common(
            host_name="",
            check_type=cur_type,
            metadata=request["metadata"],
            validator=validator,
            preprocessor=preprocessor,
        )
    except juggler.InvalidCheck as e:
        return api_response({"message": str(e), "result": False})

    return api_response({"message": constants.SUCCESS_MESSAGE, "result": True})
