"""Various helpers for view modules."""

import logging

import mongoengine

from walle.clients.cms import CmsTaskAction
from walle.errors import HostNotFoundError
from walle.hosts import Host, HostOperationState, HostStatus

log = logging.getLogger(__name__)


def get_maintenance_params(
    is_new_maintenance=True, with_power_off=True, with_cms_action=True, with_default_operation_state=True
):
    properties = {
        "timeout_time": {
            "type": "integer",
            "description": "Time until which the specified status will be active for the host."
            "When the time comes, the host will be automatically switched to the status specified by "
            "timeout_status parameter. It's guaranteed that status will be switched not earlier than the specified "
            "time, but it's not guaranteed when exactly it will happen. In practice, it will be in minutes since "
            "the specified time.",
        },
        "timeout_status": {
            "enum": HostStatus.ALL_TIMEOUT,
            "description": "Status the host will be switched to at timeout_time",
        },
        "ticket_key": {"type": "string", "description": "Attach ticket to host's manual status."},
        "operation_state": {
            "enum": HostOperationState.ALL,
            "description": "Operation state of the host after switching to maintenance",
        },
    }

    if with_default_operation_state:
        properties["operation_state"]["description"] += ", default is {}".format(HostOperationState.OPERATION)

    if is_new_maintenance:
        if with_cms_action:
            properties["cms_task_action"] = {
                "enum": [CmsTaskAction.PROFILE, CmsTaskAction.REBOOT, CmsTaskAction.TEMPORARY_UNREACHABLE],
                "description": "CMS action name for switching to maintenance. "
                "Default is {}".format(CmsTaskAction.PROFILE),
            }
        if with_power_off:
            properties["power_off"] = {"type": "boolean", "description": "Power off the host, default is false"}
    else:
        properties.update(
            {
                "remove_timeout": {
                    "type": "boolean",
                    "description": "Clear maintenance state timeout to allow using ticket for leaving maintenance.",
                }
            }
        )

    return properties


def get_authorized_host(issuer, inv_or_name, query_args, allowed_host_types=None):
    try:
        host = Host.objects(**inv_or_name.kwargs()).get()
    except mongoengine.DoesNotExist:
        raise HostNotFoundError()

    host.authorize(issuer, query_args.get("ignore_maintenance", False), allowed_host_types=allowed_host_types)

    return host
