import json
import logging
import typing as tp

from walle import network
from walle._tasks.task_args import FooBarTaskArgs
from walle._tasks.task_creator import get_foobar_stages
from walle._tasks.task_provider import schedule_task_from_api
from walle.authorization import iam
from walle.clients import racktables, bot, dns as dns_clients
from walle.constants import PROVISIONERS, HOST_TYPES_WITH_MANUAL_OPERATION
from walle.dns import dns_lib
from walle.errors import NoInformationError, InvalidHostConfiguration, ResourceConflictError, ApiError, BadRequestError
from walle.expert import decisionmakers, dmc
from walle.hosts import Host, HostPlatform
from walle.util.api import host_id_handler, api_response, admin_request
from walle.util.deploy_config import DeployConfigPolicies
from walle.views.api.host_api.common import get_authorized_host
from walle.util.misc import drop_none

log = logging.getLogger(__name__)


@host_id_handler(
    "/hosts/<host_id>/update-bot-platform",
    "POST",
    authenticate=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
)
@admin_request
def update_bot_platform(issuer, host_id_query):
    log.debug('got update-bot-platform request for host: {}, by: {}'.format(host_id_query._value, issuer))

    host = Host.get_by_host_id_query(host_id_query)

    host_platform = bot._get_host_platform(host.inv)
    host_platform = HostPlatform(system=host_platform.system, board=host_platform.board)

    updated_host = Host.objects(inv=host.inv).modify(new=True, set__platform=host_platform)

    if updated_host is None:
        raise ApiError(503, 'error updating bot platform for: {}'.format(host.inv))

    return api_response(updated_host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/dns-status",
    "GET",
    params={
        "clear": {
            "type": "boolean",
            "description": "Show dns operations required to cleanup host records. Default is false",
        }
    },
    iam_permissions=iam.GetHostApiIamPermission("host_id"),
)
def get_dns_status(host_id_query, query_args):

    host = Host.get_by_host_id_query(host_id_query)
    project = _get_project(host)

    host_vlans = _get_host_vlans(host, project)
    location_info = _get_host_current_switch(host)
    active_mac_info = _get_active_mac(host)

    expected_networks = {vlan: racktables.get_vlan_networks(location_info.switch, vlan) or [] for vlan in host_vlans}

    fqdns = network.get_host_fqdns(host)
    expected_records = _get_expected_host_dns_records(host, project, location_info.switch, active_mac_info.mac)

    if query_args.get("clear"):
        operations = _get_clear_operations(fqdns)
    else:
        operations = _get_fix_operations(expected_records)

    return api_response(
        [
            # dict has unordered keys which is very inconvenient in this case.
            ("inv", host.inv),
            ("name", host.name),
            ("project", host.project),
            ("vlan_scheme", project.vlan_scheme),
            ("hbf_project_id", hex(project.hbf_project_id)[2:] if project.hbf_project_id else None),
            ("native_vlan", project.native_vlan),
            ("project_extra_vlans", project.extra_vlans),
            ("host_extra_vlans", host.extra_vlans),
            ("project_dns_domain", project.dns_domain),
            ("active_mac", active_mac_info.mac),
            ("active_mac_source", active_mac_info.source),
            ("active_mac_timestamp", active_mac_info.timestamp),
            ("switch", location_info.switch),
            ("port", location_info.port),
            ("location_source", location_info.source),
            ("location_timestamp", location_info.timestamp),
            ("host_expected_vlans", ", ".join(map(str, host_vlans))),
            ("host_expected_networks", expected_networks),
            ("fqdns", fqdns),
            (
                "expected_records",
                [
                    [{"type": rtype, "name": rname, "value": rdata} for rdata in rdata_set]
                    for rtype, rname, rdata_set in expected_records
                ],
            ),
            ("required_operations", [operation.to_dict() for operation in operations]),
        ]
    )


@host_id_handler(
    "/hosts/<host_id>/decision-maker",
    "POST",
    schema={
        "type": "object",
        "properties": {
            "alternate_decision_args": {
                "type": "object",
                "description": "args for decision_maker.make_alternate_decision",
                "properties": {
                    "include": {
                        "type": "boolean",
                        "default": False,
                        "description": "include decision_maker.make_alternate_decision to result",
                    },
                    "checks": {"type": "array", "items": {"type": "string"}},
                    "checks_for_use": {"type": "array", "items": {"type": "string"}},
                },
            },
            "decision_args": {
                "type": "object",
                "description": "args for decision_maker.make_decision",
                "properties": {
                    "include": {
                        "type": "boolean",
                        "default": False,
                        "description": "include decision_maker.make_decision to result",
                    },
                    "checks": {"type": "array", "items": {"type": "string"}},
                },
            },
            "trace_args": {
                "type": "object",
                "description": "args for decision_maker.make_decision_trace",
                "properties": {
                    "include": {
                        "type": "boolean",
                        "default": False,
                        "description": "include decision_maker.make_decision_trace to result",
                    },
                    "checks": {"type": "array", "items": {"type": "string"}},
                },
            },
        },
        "additionalProperties": False,
    },
    iam_permissions=iam.GetHostApiIamPermission("host_id"),
)
def trace_decision_maker(request, host_id_query):
    """
    Result:
    1. Get decision_maker.make_decision(*args, **kwargs)
    2. Get decision_maker.make_alternate_decision(*args, **kwargs)
    3. Get decision_maker.make_decision_trace(*args, **kwargs)
    """

    def _replace_sets_with_lists(decision_dict: dict[tp.Any, tp.Any]):
        for key, val in decision_dict.items():
            if isinstance(val, set):
                decision_dict[key] = list(val)

    host = Host.get_by_host_id_query(host_id_query)
    decision_maker = decisionmakers.get_decision_maker(host.get_project())

    decision_trace = decision = alternate_decision = None

    if decision_args := request.get("decision_args"):
        reasons = dmc.get_host_reasons(host, decision_maker, checks=set(decision_args.get("checks", [])))
        decision = decision_maker.make_decision(host, reasons).to_dict()
        _replace_sets_with_lists(decision)

    if trace_args := request.get("trace_args"):
        host_reasons = dmc.get_host_reasons(host, decision_maker, checks=set(trace_args.get("checks", [])))
        if host_reasons:
            decisions = decision_maker.make_decision_trace(host, host_reasons)
        else:
            decisions = []
        decision_trace = {"reasons": host_reasons, "decisions": [decision.to_dict() for decision in decisions]}

    if alternate_decision_args := request.get("alternate_decision_args"):
        host_reasons = dmc.get_host_reasons(host, decision_maker)
        alternate_decision = decision_maker.make_alternate_decision(
            host,
            host_reasons,
            checks=set(alternate_decision_args.get("checks", [])),
            checks_for_use=set(alternate_decision_args.get("checks_for_use", [])),
        ).to_dict()
        _replace_sets_with_lists(alternate_decision)

    return api_response(
        drop_none(dict(decision_trace=decision_trace, decision=decision, alternate_decision=alternate_decision))
    )


@host_id_handler(
    "/hosts/<host_id>/render-deploy-config",
    "POST",
    schema={
        "type": "object",
        "properties": {
            "provisioner": {"enum": PROVISIONERS, "description": "Provisioning system"},
            "deploy_config_name": {"type": "string", "description": "Deploy config name"},
            "deploy_config_policy": {
                "enum": DeployConfigPolicies.get_all_names(),
                "description": "Deploy config policy",
            },
        },
        "additionalProperties": False,
    },
    authenticate=False,
    iam_permissions=iam.GetHostApiIamPermission("host_id"),
)
def render_deploy_config(request, host_id_query):
    """Render host's deploy config
    Provisioner, config name and config policy can be passed explicitly, otherwise they will be taken from
    host settings with fallback to project settings
    """
    requested_provisioner = request.get("provisioner")
    requested_config_name = request.get("deploy_config_name")
    requested_config_policy = request.get("deploy_config_policy")

    if not _all_or_nothing(
        lambda e: e is None, [requested_provisioner, requested_config_name, requested_config_policy]
    ):
        raise BadRequestError("provisioner, deploy_config_name and deploy_config_policy can be used only together")

    host = Host.get_by_host_id_query(host_id_query)
    _, _, _, _, _, deploy_configuration = host.deduce_deploy_configuration(
        requested_provisioner=requested_provisioner,
        requested_config=requested_config_name,
        requested_deploy_config_policy=requested_config_policy,
    )

    policy_cls = DeployConfigPolicies.get_policy_class(deploy_configuration.deploy_config_policy)
    results = policy_cls().generate(host=host, deploy_config_name=deploy_configuration.config)

    # decode generated config
    config_content_json = results.pop("config_content_json", None)
    if config_content_json:
        results["config_content"] = json.loads(config_content_json)

    results.update(
        {
            "deduced_provisioner": deploy_configuration.provisioner,
            "deduced_deploy_config": deploy_configuration.config,
            "deduced_deploy_config_policy": deploy_configuration.deploy_config_policy,
        }
    )

    return api_response(results)


@host_id_handler(
    "/hosts/<host_id>/foobar",
    "POST",
    {
        "type": "object",
        "properties": dict(
            {
                "cycles": {"type": "integer", "minimum": 1, "default": 1, "description": "Run stages for N times"},
                "foo_args": {
                    "type": "object",
                    "description": "Foo stage config",
                    "properties": {
                        "repeat": {
                            "type": "integer",
                            "minimum": 1,
                            "default": 1,
                            "description": "Repeat each stage run N times",
                        },
                        "label": {"type": "string", "maxLength": 20, "description": "Label for the stage"},
                        "period": {"type": "integer", "default": 5, "description": "Repeat period"},
                    },
                },
                "bar_args": {
                    "type": "object",
                    "description": "Bar stage config",
                    "properties": {
                        "repeat": {
                            "type": "integer",
                            "minimum": 1,
                            "default": 1,
                            "description": "Repeat each stage run N times",
                        },
                        "label": {"type": "string", "maxLength": 20, "description": "Label for the stage"},
                        "period": {"type": "integer", "default": 5, "description": "Repeat period"},
                    },
                },
            }
        ),
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.GetHostApiIamPermission("host_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
@admin_request
def foobar_host(issuer, host_id_query, allowed_host_types, request, reason):
    """Foobar task to make fsm execute host task without doing anything to host."""
    host = get_authorized_host(issuer, host_id_query, {}, allowed_host_types=allowed_host_types)

    reboot_task_args = FooBarTaskArgs(
        issuer=issuer,
        project=host.project,
        host_inv=host.inv,
        host_name=host.name,
        host_uuid=host.uuid,
        ignore_cms=True,
        disable_admin_requests=True,
        monitor_on_completion=False,
        with_auto_healing=False,
        ignore_maintenance=True,
        reason=reason,
        **request,
    )
    schedule_task_from_api(host, reboot_task_args, get_foobar_stages)

    return api_response(host.to_api_obj())


def _all_or_nothing(condition, seq):
    matched = list(filter(condition, seq))
    return not len(matched) or len(matched) == len(seq)


def _get_project(host):
    return host.get_project(fields=network.DNS_REQUIRED_PROJECT_FIELDS)


def _get_host_vlans(host, project):
    try:
        expected_vlans = network.get_host_expected_vlans(host, project).vlans
    except (NoInformationError, InvalidHostConfiguration) as e:
        raise InvalidHostConfiguration("Can not get host expected vlans: {}", e)

    if not expected_vlans:
        if project.vlan_scheme:
            raise InvalidHostConfiguration("DNS manipulation is not supported for vlan scheme: {}", project.vlan_scheme)
        else:
            raise InvalidHostConfiguration("DNS manipulation is not supported for projects without vlan scheme.")

    return expected_vlans


def _get_host_current_switch(host):
    try:
        return network.get_current_host_switch_port(host)
    except (NoInformationError, InvalidHostConfiguration) as e:
        raise InvalidHostConfiguration("Can not get host current switch information: {}", e)


def _get_active_mac(host):
    active_mac_info = host.get_active_mac()
    if active_mac_info is not None:
        return active_mac_info
    else:
        raise ResourceConflictError("Unable to determine active MAC address of the host.")


def _get_expected_host_dns_records(host, project, switch, active_mac):
    try:
        return network.get_host_dns_records(host, project, host.name, switch, active_mac)
    except (InvalidHostConfiguration, network.NoNetworkOnSwitch) as e:
        raise ResourceConflictError("Can not build expected dns records: {}".format(e))


def _get_clear_operations(fqdns):
    try:
        return dns_lib.get_delete_operations_for_fqdns(fqdns, MockStatboxLogger)
    except dns_clients.DnsError as e:
        raise ResourceConflictError("Error communication with dns api: {}".format(e))


def _get_fix_operations(dns_records):
    try:
        return dns_lib.get_operations_for_dns_records(dns_records, MockStatboxLogger)
    except dns_clients.DnsError as e:
        raise ResourceConflictError("Error communication with dns api: {}".format(e))


class MockStatboxLogger:
    @staticmethod
    def get_child(*args, **kwargs):
        return MockStatboxLogger

    @staticmethod
    def log(*args, **kwargs):
        return None
