"""Hosts"""  # Will be used as category name in API reference

import logging

import mongoengine
from flask import Response
from gevent.timeout import Timeout

import walle.expert.types
import walle.host_health
import walle.network
import walle.util.host_health
import walle.util.misc
from sepelib.core.exceptions import Error, LogicalError
from walle import constants as walle_constants, restrictions, util
from walle.authorization import iam
from walle.clients import racktables, deploy, eine, staff
from walle.errors import HostNotFoundError, BadRequestError, InvalidHostConfiguration, TooManyRequestsError
from walle.expert import automation_plot, constants as expert_constants
from walle.expert.types import CheckType, CheckStatus
from walle.hosts import (
    Host,
    HostState,
    HostStatus,
    HostLocation,
    get_host_human_name,
    HostNetwork,
    parse_physical_location_query,
)
from walle.models import DocumentPostprocessor, get_requested_fields
from walle.projects import Project, get_project_owners
from walle.util.api import (
    api_handler,
    host_id_handler,
    api_response,
    FilterQueryParser,
    SortQueryParser,
    get_query_result,
    get_object_result,
    expand_query_params,
)
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.juggler import get_aggregate_name
from walle.util.misc import drop_none, parallelize_execution
from walle.util.mongo import MongoDocument
from walle.util.patterns import parse_pattern
from walle.views.api.common import validated_tags

log = logging.getLogger(__name__)

RECEIVE_TIMEOUT = 5


class HostPostprocessor(DocumentPostprocessor):
    def __init__(
        self, resolve_owners=False, resolve_tags=False, resolve_deploy_configuration=False, juggler_aggregate_name=False
    ):
        self.resolve_owners = resolve_owners
        self.resolve_tags = resolve_tags
        self.resolve_deploy_configuration = resolve_deploy_configuration
        self.juggler_aggregate_name = juggler_aggregate_name

        extra_db_fields = set()
        extra_fields = set()

        if self.resolve_owners:
            extra_db_fields.add(Host.project.db_field)
            extra_fields.add("owners")

        if self.resolve_tags:
            extra_db_fields.add(Host.project.db_field)
            extra_fields.add("tags")

        if self.resolve_deploy_configuration:
            extra_db_fields.update((Host.state.db_field, Host.project.db_field, Host.provisioner.db_field))
            extra_fields.update(
                ("provisioner", "config", "deploy_tags", "custom_deploy_configuration", "deploy_config_policy")
            )

        if self.juggler_aggregate_name:
            extra_db_fields.update(
                [
                    Host.location.db_field + "." + HostLocation.short_queue_name.db_field,
                    Host.location.db_field + "." + HostLocation.rack.db_field,
                ]
            )
            extra_fields.add("juggler_aggregate_name")

        super().__init__(extra_db_fields, extra_fields)

    def strip_fields(self, requested_fields):
        strip_fields = {field for field in (self.extra_db_fields | {"inv"}) if [field] not in requested_fields}
        return strip_fields

    def process(self, iterable, requested_fields):
        hosts = []
        project_ids = set()

        resolve_owners = self.resolve_owners
        resolve_tags = self.resolve_tags
        resolve_deploy_configuration = self.resolve_deploy_configuration
        juggler_aggregate_name = self.juggler_aggregate_name

        for host in gevent_idle_iter(iterable):
            no_deploy_config = host.state in HostState.ALL_ASSIGNED and host.provisioner is None
            needs_project = resolve_owners or resolve_tags or (resolve_deploy_configuration and no_deploy_config)
            if needs_project:
                project_ids.add(host.project)

            hosts.append((host, needs_project))

        if project_ids:
            fields = []
            if resolve_tags:
                fields.append("tags")
            if resolve_deploy_configuration:
                fields.extend(("provisioner", "deploy_config", "deploy_config_policy", "deploy_tags"))

            project_doc = MongoDocument.for_model(Project)
            projects = {prj.id: prj for prj in project_doc.find({"_id": {"$in": list(project_ids)}}, fields)}
        else:
            projects = {}

        objects = []
        owners = {}
        if resolve_owners:
            forced_owners = [walle_constants.ROBOT_WALLE_OWNER]
            for prj_id, project in projects.items():
                owners[prj_id] = staff.resolve_owners(tuple(sorted(get_project_owners(project)))) + forced_owners

        for host, needs_project in gevent_idle_iter(hosts):
            # Warning:
            #
            # We skip hosts for which we can't find the project.
            #
            # This may be due to:
            # * The host has been removed and project has been deleted.
            # * The host has been moved to another project and the source project has been deleted.
            #
            # We can't gracefully handle the situation, but we can consider project switching operation as non-atomic
            # host remove/add operations from user's point of view and with this assumption we can skip the host in both
            # cases.

            extra_fields = {}

            project = None
            if needs_project:
                try:
                    project = projects[host.project]
                except KeyError:
                    log.critical("Failed to return a host from API: '%s' project is missing.", host.project)
                    continue

            if resolve_tags:
                extra_fields["tags"] = project.tags

            if resolve_owners:
                extra_fields["owners"] = owners[host.project]

            if resolve_deploy_configuration and host.state in HostState.ALL_ASSIGNED:
                if host.provisioner is None:
                    extra_fields.update(
                        {
                            "provisioner": project.provisioner,
                            "config": project.deploy_config,
                            "deploy_config_policy": project.deploy_config_policy,
                            "deploy_tags": project.deploy_tags,
                            "custom_deploy_configuration": False,
                        }
                    )
                else:
                    extra_fields["custom_deploy_configuration"] = True

            if juggler_aggregate_name:
                extra_fields["juggler_aggregate_name"] = get_aggregate_name(
                    host.location.short_queue_name, host.location.rack
                )

            objects.append(host.to_api_object_shallow(requested_fields, extra_fields=extra_fields))

        return objects


def _get_check_type_filters(check_type):
    """Returns all available failure reason filters for the specified check type."""

    return [
        walle.util.host_health.get_failure_reason(check_type, check_status)
        for check_status in CheckStatus.ALL
        if check_status != CheckStatus.PASSED
    ] + [
        walle.util.host_health.get_failure_reason_deprecated(check_type, check_status)
        for check_status in CheckStatus.ALL
        if check_status != CheckStatus.PASSED
    ]


def _get_health_status_filters(runtime=False):
    """Returns all available health status filters."""

    seen_ui_check_types = set()
    failure_reasons_filters = []

    if runtime:
        automation_plot_check_types = list(automation_plot.get_all_automation_plots_checks())
    else:
        automation_plot_check_types = []

    all_check_types = CheckType.ALL + automation_plot_check_types

    for check_type in all_check_types:
        ui_check_type = walle.expert.types.get_walle_check_type(check_type)

        if ui_check_type not in seen_ui_check_types:
            failure_reasons_filters.extend(_get_check_type_filters(check_type))
            seen_ui_check_types.add(ui_check_type)

    return (
        expert_constants.HEALTH_STATUS_FILTERS
        + CheckType.ALL_UI_TYPES
        + failure_reasons_filters
        + automation_plot_check_types
    )


def _get_host_sort_fields():
    return (
        "inv",
        "name",
        "project",
        "state",
        "status",
        "tier",
        "scenario_id",
        "ticket",
        "switch",
        "health",
        "datacenter",
        "type",
    )


def _get_host_filters_schema():
    """Returns host filters schema."""

    return {
        "name": {"type": "string", "description": "Filter by host name substring"},
        "tags": {"type": "array", "items": {"type": "string"}, "description": "Projects' tags"},
        "type": {
            "type": "array",
            "items": {"enum": walle_constants.HostType.get_choices()},
            "description": "Filter by type",
        },
        "include_shadow": {
            "type": "boolean",
            "description": "Add shadow hosts to result. If used if 'type' field in request - ignored. Default is false.",
        },
        "state": {"type": "array", "items": {"enum": HostState.ALL}, "description": "Filter by state"},
        "status": {
            "type": "array",
            "items": {"enum": HostStatus.ALL_FILTERS + HostStatus.ALL},
            "description": "Filter by status",
        },
        "health": {
            "type": "array",
            "items": {
                "anyOf": [{"enum": _get_health_status_filters()}, {"pattern": "[a-zA-Z0-9_-].+"}],
            },
            "description": "Filter by health status. This works like logical 'OR' or 'ANY', "
            "e.g. this will return hosts with any of the specified problems. "
            "There is currently no api providing logical 'AND' or 'ALL' functionality.",
        },
        "scenario_id": {"type": "array", "items": {"type": "integer"}, "description": "Filter by scenario id"},
        "ticket": {"type": "array", "items": {"type": "string"}, "description": "Filter hosts by associated ticket(s)"},
        "project": {"type": "array", "items": {"type": "string"}, "description": "Filter by project"},
        "task_owner": {
            "type": "array",
            "items": {"type": "string"},
            "description": "Filter by task owner (don't forget @ after user login)",
        },
        "provisioner": {
            "type": "array",
            "items": {"enum": walle_constants.PROVISIONERS},
            "description": "Filter by provisioner",
        },
        "config": {"type": "array", "items": {"type": "string"}, "description": "Filter by config"},
        "restrictions": {"type": "array", "items": {"enum": restrictions.ALL}, "description": "Filter by restriction"},
        "physical_location": {
            "type": "array",
            "items": {"type": "string"},
            "description": "Filter by physical location (format: country|city|datacenter|queue|rack)",
        },
        "switch": {"type": "array", "items": {"type": "string"}, "description": "Filter by switch name"},
        "port": {"type": "array", "items": {"type": "string"}, "description": "Filter by port name"},
    }


def _get_host_postprocessors_schema():
    """Returns a schema for host postprocessing filters."""

    return {
        "resolve_owners": {
            "type": "boolean",
            "description": "Lookup host owners from it's project for every host (this option adds `owners` field to result). "
            "Default is false",
        },
        "resolve_tags": {
            "type": "boolean",
            "description": "Lookup host tags from it's project for every host (this option adds `tags` field to result)."
            "Default is false",
        },
        "resolve_deploy_configuration": {
            "type": "boolean",
            "description": "Show project's deploy configuration if host doesn't have custom config "
            "(this option adds `custom_deploy_configuration` boolean field)."
            "Default is false",
        },
        "juggler_aggregate_name": {"type": "boolean", "description": "Show juggler aggregate name. Default is false"},
    }


def _get_host_query_schema():
    schema = expand_query_params(_get_host_filters_schema())
    schema.update(_get_host_postprocessors_schema())
    return schema


def _expand_shadow_hosts_filters(query_args: dict):
    """
    https://st.yandex-team.ru/WALLE-4572
    if "type" in request - return what visibly requested.
    else if "include_shadow" in request - return all hosts' types.
    else - just return "HOST_TYPES_WITH_MANUAL_OPERATION".
    """
    if "type" in query_args:
        return
    elif query_args.get("include_shadow", False):
        query_args["type"] = walle_constants.HOST_TYPES_ALL
    else:
        query_args["type"] = walle_constants.HOST_TYPES_WITH_MANUAL_OPERATION


def _get_fields_description():
    return "Object fields to return. Available fields: {}.".format(", ".join(Host.api_fields))


@api_handler(
    "/hosts",
    "GET",
    params=_get_host_query_schema(),
    with_fields=Host,
    with_paging={
        "cursor": {"type": "integer", "minimum": 0, "description": "Inventory number to start the search from"},
        "max_limit": 10000,
    },
    with_sort={"fields": _get_host_sort_fields(), "description": "Host fields to sort"},
    max_concurrent_requests=10,
    concurrent_requests_timeout=10,
)
def get_hosts(query_args):
    """Returns hosts that match the specified query string filter."""
    if query_args.get("offset", 0) > 1000:
        raise BadRequestError("Big offsets are forbidden")
    _expand_shadow_hosts_filters(query_args)
    return _get_hosts(query_args, iam_public_handler=True)


@api_handler(
    "/get-hosts",
    "POST",
    {
        "type": "object",
        "properties": {
            "invs": {"type": "array", "items": {"type": "integer", "minimum": 0, "description": "Inventory numbers"}},
            "names": {"type": "array", "items": {"type": "string", "description": "FQDNs"}},
            "uuids": {"type": "array", "items": {"type": "string", "description": "UUIDs"}},
            "patterns": {"type": "array", "items": {"type": "string", "description": "FQDN patterns with globs"}},
        },
        "additionalProperties": False,
    },
    params=_get_host_query_schema(),
    with_fields=Host,
    with_paging={
        "cursor": {"type": "integer", "minimum": 0, "description": "Inventory number to start the search from"},
        "max_limit": 10000,
    },
    with_sort={"fields": _get_host_sort_fields(), "description": "Host fields to sort"},
    rps=30,
    iam_permissions=iam.GetHostsApiIamPermission(
        hosts_arg_name="names",
        hosts_arg_inv="invs",
        hosts_arg_uuid="uuids",
        host_patterns="patterns",
    ),
)
def get_specified_hosts(query_args, request):
    """Returns only the specified hosts that match the specified query string filter."""
    if query_args.get("offset", 0) > 1000:
        raise BadRequestError("Big offsets are forbidden")

    _expand_shadow_hosts_filters(query_args)

    return _get_hosts(
        query_args, host_spec=(request.get("invs"), request.get("uuids"), request.get("names"), request.get("patterns"))
    )


@host_id_handler(
    "/hosts/<host_id>",
    "GET",
    params=_get_host_postprocessors_schema(),
    with_fields=Host,
    iam_permissions=iam.GetHostApiIamPermission("host_id"),
)
def get_host(host_id_query, query_args):
    """Returns the specified host."""
    requested_fields = get_requested_fields(Host, query_args.get("fields"))

    postprocessor = HostPostprocessor(
        resolve_owners=query_args.get("resolve_owners", False),
        resolve_tags=query_args.get("resolve_tags", False),
        resolve_deploy_configuration=query_args.get("resolve_deploy_configuration", False),
        juggler_aggregate_name="juggler_aggregate_name" in requested_fields,
    )

    query = {getattr(Host, field).db_field: value for field, value in host_id_query.kwargs().items()}
    host = get_object_result(Host, query, query_args, postprocessor=postprocessor)
    if host is None:
        raise HostNotFoundError()

    return api_response(host)


@api_handler(
    "/hosts/<host_uuid>/network",
    "GET",
    with_fields=HostNetwork,
    iam_permissions=iam.GetHostApiIamPermission("host_uuid"),
)
def get_host_network(host_uuid, query_args):
    """Returns network dynamic information of specified host."""

    fields = query_args.get("fields")
    host_network = HostNetwork.get_by_uuid(host_uuid, fields=fields)
    return api_response(host_network.to_api_obj(fields))


@api_handler(
    "/get-hosts/network",
    "POST",
    {
        "type": "object",
        "properties": {
            "uuids": {"type": "array", "items": {"type": "string", "description": "UUIDs"}},
        },
        "additionalProperties": False,
    },
    params=_get_host_query_schema(),
    with_fields=HostNetwork,
    with_paging={
        "cursor": {"type": "string", "description": "UUID to start the search from"},
        "cursor_only": True,
        "max_limit": 10000,
    },
    rps=30,
    iam_permissions=iam.GetHostsApiIamPermission(hosts_arg_uuid="uuids"),
)
def get_hosts_network(query_args, request):
    """Returns hosts that match the specified query string filter."""
    _expand_shadow_hosts_filters(query_args)
    return _get_hosts_network(query_args, request.get("uuids"))


@host_id_handler(
    "/hosts/<host_id>/current-configuration",
    "GET",
    iam_permissions=iam.GetHostApiIamPermission("host_id"),
)
def get_current_configuration(host_id_query):
    """Returns current host configuration with best effort up-to-date status."""
    host = Host.get_by_host_id_query(host_id_query)

    def get_expected_vlans():
        try:
            vlan_config = walle.network.get_host_expected_vlans(host)
        except InvalidHostConfiguration as e:
            log.warning("Failed to determine expected VLANs for %s: %s", host.human_name(), e)
            return

        if vlan_config is not None:
            return {
                "expected_hbf_project_id": vlan_config.mtn_project_id,
                "expected_vlans": vlan_config.vlans,
                "expected_native_vlan": vlan_config.native_vlan,
            }

    def get_netmap():
        if host.location is None or host.location.switch is None or host.location.port is None:
            return

        vlans, native_vlan, synced = racktables.get_port_vlan_status(host.location.switch, host.location.port)

        return {"vlans": vlans, "native_vlan": native_vlan, "vlans_synced": synced}

    def get_port_project():
        if host.location is None or host.location.switch is None or host.location.port is None:
            return

        try:
            real_hbf_project_id, synced = racktables.get_port_project_status(host.location.switch, host.location.port)
        except racktables.MtnNotSupportedForSwitchError:
            return

        return {"hbf_project_id": real_hbf_project_id, "hbf_project_id_synced": synced}

    info_funcs = [get_netmap, get_port_project]
    if host.state in HostState.ALL_ASSIGNED:
        info_funcs.append(get_expected_vlans)

    info = {}
    for result in parallelize_execution(*info_funcs):
        if result:
            info.update(result)

    return api_response(drop_none(info))


@host_id_handler(
    "/hosts/<host_id>/profile-log",
    "GET",
    iam_permissions=iam.GetHostApiIamPermission("host_id"),
)
def get_profile_log(host_id_query):
    """Returns host's profile log."""

    try:
        host = Host.objects.only("inv", "name", "project").get(**host_id_query.kwargs())
    except mongoengine.DoesNotExist:
        raise HostNotFoundError()

    try:
        client = eine.get_client(eine.get_eine_provider(host.get_eine_box()))
        log_data = client.get_profile_log(host.inv)
    except Exception as e:
        raise Error("Unable to get {}'s profile log: {}", get_host_human_name(host.inv, host.name), e)
    return Response(log_data, mimetype="text/plain")


@host_id_handler(
    "/hosts/<host_id>/ban-status",
    "GET",
    rps=300,
    iam_permissions=iam.GetHostApiIamPermission("host_id"),
)
def get_iss_ban_status(host_id_query):
    """Returns host's ban status."""

    fields = (
        "cms_task_id",
        "task",
        "ticket",
        "status",
        "status_reason",
        "state",
        "state_reason",
    )
    with Timeout(RECEIVE_TIMEOUT, TooManyRequestsError):
        try:
            host = Host.objects.only(*fields).get(**host_id_query.kwargs())
            if host.cms_task_id:
                banned = True
                cms_task_id = host.cms_task_id
            elif host.get_iss_ban_flag():
                banned = True
                cms_task_id = host.task.cms_task_id
            else:
                banned = False
                cms_task_id = None
        except mongoengine.DoesNotExist:
            raise HostNotFoundError()

        if banned and host.state == HostState.MAINTENANCE:
            reason = host.state_reason
        else:
            reason = host.status_reason

        return api_response(
            drop_none({"result": banned, "reason": reason, "cms_task_id": cms_task_id, "ticket": host.ticket})
        )


@host_id_handler(
    "/hosts/<host_id>/deploy-log",
    "GET",
    params={
        "provisioner": {"enum": walle_constants.PROVISIONERS, "description": "Provisioner to get the log from"},
        "tail_bytes": {
            "type": "integer",
            "minimum": 0,
            "description": "Return only the specified number of bytes from the tail. "
            "If provisioner doesn't support this feature the whole log will be returned.",
        },
    },
    iam_permissions=iam.GetHostApiIamPermission("host_id"),
)
def get_deploy_log(host_id_query, query_args):
    """Returns host's deploy log."""

    host_fields = ("project", "inv", "name", "provisioner", "config", "state", "platform")
    try:
        host = Host.objects.only(*host_fields).get(**host_id_query.kwargs())
    except mongoengine.DoesNotExist:
        raise HostNotFoundError()

    if "provisioner" in query_args:
        provisioner = query_args["provisioner"]
    else:
        if host.state == HostState.FREE:
            provisioner = host.get_project_deploy_configuration().provisioner
        elif host.state in HostState.ALL_ASSIGNED:
            provisioner = host.get_deploy_configuration().provisioner
        else:
            raise LogicalError()

    log_data = _get_deploy_log(provisioner, host.get_eine_box(), host.inv, host.name, query_args.get("tail_bytes"))

    return Response(log_data, mimetype="text/plain")


def _get_deploy_log(provisioner, project_box, inv, name=None, tail_bytes=None):
    if provisioner not in [walle_constants.PROVISIONER_LUI, walle_constants.PROVISIONER_EINE]:
        raise LogicalError()

    try:
        if provisioner == walle_constants.PROVISIONER_LUI:
            if not name:
                return ""
            lui_client = deploy.get_lui_client(deploy.get_deploy_provider(project_box))
            return lui_client.get_deploy_log(name, tail_bytes=tail_bytes)
        elif provisioner == walle_constants.PROVISIONER_EINE:
            eine_client = eine.get_client(eine.get_eine_provider(project_box))
            return eine_client.get_profile_log(inv)
    except Exception as e:
        raise Error("Unable to get {}'s deploy log from {}: {}", get_host_human_name(inv, name), provisioner, e)


def _validate_host_health_query(query_args):
    all_check_types = _get_health_status_filters(runtime=True)
    expanded_args = expand_query_params({"health": all_check_types})

    for key in expanded_args.keys():
        if key in query_args:
            query_set = set(query_args[key])
            validation_set = set(expanded_args[key])

            if not query_set.issubset(validation_set):
                bad_health = ','.join(query_set.difference(validation_set))
                raise BadRequestError("Health args: {} are not supported".format(bad_health))


def _get_hosts(query_args, host_spec=None, iam_public_handler=False):
    # runtime validations
    util.api.validate_project_filter(query_args)
    _validate_host_health_query(query_args)

    query_parser = FilterQueryParser(
        Host,
        enum_fields=("project", "state", "switch", "port", "ticket", "task_owner", "scenario_id", "type"),
        substring_fields=("name",),
        field_aliases={"switch": "location.switch", "port": "location.port", "task_owner": "task.owner"},
    )
    filter_query = query_parser.parse_query_filter(query_args)

    sort_parser = SortQueryParser(
        Host,
        aliases={
            "switch": "location.switch",
            "health": "health.status",
            # full sort by full location is very long
            "datacenter": ("location.country", "location.city", "location.datacenter"),
        },
    )
    order_by = sort_parser.parse_sort_query(query_args.get("sort-by", ()))

    # host-specific query filter - physical location
    try:
        physical_locations = query_args["physical_location"]
    except KeyError:
        physical_locations = None
    else:
        filter_query.update(parse_physical_location_query(physical_locations))

    _parse_project_tags_filter(filter_query, query_args)
    _parse_health_filter(filter_query, query_args)
    _parse_restrictions_filter(filter_query, query_args)

    queries = [filter_query]
    _parse_status_filter(queries, query_args)
    _parse_deploy_configuration_filter(queries, query_args)

    if host_spec is not None:
        invs, uuids, names, patterns = host_spec
        if any((invs, uuids, names, patterns)):
            host_spec_query = []

            if invs:
                host_spec_query.append({Host.inv.db_field: {"$in": invs}})
            if uuids:
                host_spec_query.append({Host.uuid.db_field: {"$in": uuids}})
            if names:
                host_spec_query.append({Host.name.db_field: {"$in": names}})
            if patterns:
                host_spec_query.append({Host.name.db_field: {"$in": [parse_pattern(p) for p in patterns if p]}})

            queries.append({"$or": host_spec_query})
        else:
            queries = None

    if physical_locations is not None and not physical_locations:
        queries = None

    requested_fields = get_requested_fields(Host, query_args.get("fields"), iam_public_handler)

    postprocessor = HostPostprocessor(
        resolve_owners=query_args.get("resolve_owners", False),
        resolve_tags=query_args.get("resolve_tags", False),
        resolve_deploy_configuration=query_args.get("resolve_deploy_configuration", False),
        juggler_aggregate_name="juggler_aggregate_name" in requested_fields,
    )
    return api_response(get_query_result(Host, queries, Host.inv, query_args, order_by, postprocessor=postprocessor))


def _get_hosts_network(query_args, uuids=None):
    query = {HostNetwork.uuid.db_field: {"$in": uuids}}
    return api_response(get_query_result(HostNetwork, query, HostNetwork.uuid, query_args, cursor_only=True))


def _parse_status_filter(filter_queries, query_args):
    keywords = {"status", "status__in", "status__nin", "task_owner"}  # task owner implicitly enables status filter
    if not query_args.keys() & keywords:
        return

    if len(query_args.keys() & keywords) > 1:
        raise BadRequestError("Use one status filter, not many, either status/status__in or status__nin")

    status_filters = []
    if "status" in query_args:
        status_query = set(query_args.pop("status"))
        status_filters_query = status_query & set(HostStatus.ALL_FILTERS)
        requested_statuses = status_query - set(HostStatus.ALL_FILTERS)

        if status_filters_query:
            pymongo_host = MongoDocument.for_model(Host)

            for group in status_filters_query:
                if group == HostStatus.FILTER_ERROR:
                    # this filters not exactly by the status field but we want user to be unaware of this
                    status_filters.append({pymongo_host.resolve_field("task.error"): {"$exists": True}})
                elif group == HostStatus.FILTER_STEADY:
                    requested_statuses.update(HostStatus.ALL_STEADY)
                elif group == HostStatus.FILTER_TASK:
                    requested_statuses.update(HostStatus.ALL_TASK)
                else:
                    # unsupported filter
                    raise LogicalError
        if requested_statuses:
            query_args["status"] = list(requested_statuses)
    elif "task_owner" in query_args:
        # Filtering by task owner implicitly searches by tasks, so search only by hosts having tasks to use indices
        query_args["status"] = HostStatus.ALL_TASK

    query_parser = FilterQueryParser(Host, enum_fields=["status"])

    status_filters.append(query_parser.parse_query_filter(query_args))
    status_filters = list(filter(None, status_filters))  # filter out empty clauses

    if not status_filters:
        return

    if len(status_filters) > 1:
        filter_queries.append({"$or": status_filters})
    else:
        filter_queries.append(status_filters[0])


def _parse_project_tags_filter(filter_query, query_args):
    if "tags" in query_args and query_args["tags"]:
        # this filter:
        # * does not support tag exclusion and tag intersections
        project_ids = Project.objects(tags__all=validated_tags(query_args["tags"])).get_ids()
        if "project" in query_args:
            project_filter_value = filter_query[Host.project.db_field]["$in"]
            intersection_of_previous_filter_and_current = set(project_filter_value).intersection(set(project_ids))
            project_ids = list(intersection_of_previous_filter_and_current)
        filter_query[Host.project.db_field] = {"$in": project_ids}


def _parse_health_filter(filter_query, query_args):
    if not query_args.keys() & ("health", "health__in", "health__nin"):
        return

    pymongo_host = MongoDocument.for_model(Host)
    health_filter = set(query_args.get("health", []))
    status_filter = health_filter & set(expert_constants.HEALTH_STATUS_FILTERS)

    if status_filter:
        health_status = tuple(status_filter)[0]
        if len(health_filter) != 1 or "health__in" in query_args or "health__nin" in query_args:
            raise BadRequestError(
                "Invalid health status filter: {} mustn't be used with other health status filters.", health_status
            )

        if health_status == expert_constants.HEALTH_STATUS_FILTER_MISSING:
            filter_query[Host.health.db_field] = {"$exists": False}
        else:
            filter_query[pymongo_host.resolve_field("health.status")] = health_status
    else:
        any_of = set(query_args.get("health", [])) | set(query_args.get("health__in", []))
        none_of = set(query_args.get("health__nin", []))

        for check_type in CheckType.ALL + list(automation_plot.get_all_automation_plots_checks()):
            walle_check_type = walle.expert.types.get_walle_check_type(check_type)

            if walle_check_type in any_of:
                any_of.discard(walle_check_type)
                any_of.update(_get_check_type_filters(check_type))

            if walle_check_type in none_of:
                none_of.discard(walle_check_type)
                none_of.update(_get_check_type_filters(check_type))

        query = {}
        if any_of:
            query["$in"] = list(any_of)
        if none_of:
            query["$nin"] = list(none_of)

        if query:
            filter_query[pymongo_host.resolve_field("health.reasons")] = query


def _parse_restrictions_filter(filter_query, query_args):
    if "restrictions" in query_args:
        user_restrictions = query_args["restrictions"]

        if user_restrictions:
            filter_query[Host.restrictions.db_field] = {"$in": restrictions.expand_restrictions(user_restrictions)}
        else:
            filter_query[Host.restrictions.db_field] = {"$exists": False}


def _parse_deploy_configuration_filter(queries, query_args):
    if "provisioner" not in query_args and "config" not in query_args:
        return

    # Find all projects with the specified deployment configuration
    project_ids = Project.objects(
        **drop_none(
            dict(
                id__in=query_args.get("project"),
                provisioner__in=query_args.get("provisioner"),
                deploy_config__in=query_args.get("config"),
            )
        )
    ).get_ids()

    host_provisioner_query = {}
    if query_args.get("provisioner"):
        host_provisioner_query[Host.provisioner.db_field] = {"$in": query_args["provisioner"]}
    if query_args.get("config"):
        host_provisioner_query[Host.config.db_field] = {"$in": query_args["config"]}

    provisioner_query = {
        Host.state.db_field: {"$in": HostState.ALL_ASSIGNED},
        "$or": [
            # Host with the requested custom configuration
            host_provisioner_query,
            # Host without custom deploy configuration, but which inherits the requested one
            {Host.provisioner.db_field: {"$exists": False}, Host.project.db_field: {"$in": project_ids}},
        ],
    }
    queries.append(provisioner_query)
