"""Hosts"""  # Will be used as category name in API reference

import logging

import walle.host_status
import walle.projects
import walle.util.api
from sepelib.core.exceptions import Error, LogicalError
from walle import audit_log, constants as walle_constants, restrictions
from walle import authorization
from walle.authorization import iam
from walle.clients import deploy
from walle.constants import (
    NetworkTarget,
    HostType,
    HOST_TYPES_WITH_PARTIAL_AUTOMATION,
    HOST_TYPES_WITH_MANUAL_OPERATION,
)
from walle.errors import InvalidHostStateError, BadRequestError, RequestValidationError
from walle.hosts import Host, HostState, HostStatus, get_host_query, TaskType, HostLocation
from walle.restrictions import strip_restrictions
from walle.tasks import schedule_setting_assigned_state
from walle.util.api import host_id_handler, api_response
from walle.util.deploy_config import DeployConfigPolicies
from walle.util.misc import fix_mongo_set_kwargs
from walle.views.api.common import (
    API_ACTION_SET,
    API_ACTION_REMOVE,
    get_api_action_methods,
    get_vlan_id_schema,
    get_api_action,
    get_api_action_update_query,
    get_eine_tags_schema,
)
from walle.views.api.host_api.common import get_authorized_host, get_maintenance_params
from walle.views.api.host_api.operations import _get_new_task_params, _get_schedule_task_params
from walle.views.helpers.maintenance import set_host_maintenance_for_host, change_host_maintenance

log = logging.getLogger(__name__)


@host_id_handler(
    "/hosts/<host_id>/deploy_config",
    "PUT",
    {
        "type": "object",
        "properties": {
            "provisioner": {"enum": walle_constants.PROVISIONERS, "description": "Provisioning system"},
            "config": {"type": "string", "description": "Deploy config"},
            "deploy_config_policy": {
                "enum": DeployConfigPolicies.get_all_names(),
                "description": "Deploy config policy",
            },
            "deploy_tags": get_eine_tags_schema(for_profile=False),
            "deploy_network": {
                "enum": NetworkTarget.DEPLOYABLE,
                "description": "Deploy host either in service or in project VLAN's",
            },
        },
        "required": ["config"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=[HostType.SERVER],
)
def set_deploy_config(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    allowed_states = HostState.ALL_ASSIGNED

    if host.state not in allowed_states:
        raise InvalidHostStateError(host, allowed_states=allowed_states)

    if not reason:
        raise BadRequestError("Please specify a ticket or human-readable reason")

    provisioner, deploy_config, deploy_tags, deploy_network, deploy_config_policy, _ = host.deduce_deploy_configuration(
        request.get("provisioner"),
        request["config"],
        request.get("deploy_tags"),
        request.get("deploy_network"),
        request.get("deploy_config_policy"),
    )

    with audit_log.on_update_host(
        issuer, host.project, host.inv, host.name, host.uuid, request, reason, scenario_id=host.scenario_id
    ):
        if provisioner == walle_constants.PROVISIONER_LUI:
            try:
                lui_client = deploy.get_lui_client(deploy.get_deploy_provider(host.get_eine_box()))
                lui_client.set_host_config(host.name, deploy_config)
            except deploy.DeployPersistentError as e:
                # Might be caused by IPv6-only host, invalid hostname or absence of hostname in the system
                log.warning("Failed to set deploy config in LUI/DHCP for %s: %s.", host.name, e)
            except deploy.DeployInternalError as e:
                raise Error("Failed to update host's config in LUI: {}", e)

        query = get_host_query(
            issuer, query_args.get("ignore_maintenance", False), allowed_states, **host_id_query.kwargs(inv=host.inv)
        )

        updated_host = Host.objects(**query).modify(
            new=True,
            **fix_mongo_set_kwargs(
                set__provisioner=provisioner,
                set__config=deploy_config,
                set__deploy_config_policy=deploy_config_policy,
                set__deploy_tags=deploy_tags,
                set__deploy_network=deploy_network,
            )
        )

        if updated_host is None:
            raise InvalidHostStateError(host, allowed_states=allowed_states)

        return api_response(updated_host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/deploy_config",
    "DELETE",
    {
        "type": "object",
        "properties": {},
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=[HostType.SERVER],
)
def delete_deploy_config(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    allowed_states = HostState.ALL_ASSIGNED

    if host.state not in allowed_states:
        raise InvalidHostStateError(host, allowed_states=allowed_states)

    with audit_log.on_update_host(
        issuer,
        host.project,
        host.inv,
        host.name,
        host.uuid,
        {
            "provisioner": None,
            "config": None,
            "deploy_config_policy": None,
            "deploy_tags": None,
            "deploy_network": None,
        },
        reason,
        scenario_id=host.scenario_id,
    ):
        query = get_host_query(
            issuer, query_args.get("ignore_maintenance", False), allowed_states, **host_id_query.kwargs(inv=host.inv)
        )

        updated_host = Host.objects(**query).modify(
            new=True,
            unset__provisioner=True,
            unset__config=True,
            unset__deploy_config_policy=True,
            unset__deploy_tags=True,
            unset__deploy_network=True,
        )

        if updated_host is None:
            raise InvalidHostStateError(host, allowed_states=allowed_states)

        return api_response(updated_host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>",
    ("PATCH", "POST"),
    {
        "type": "object",
        "properties": {
            "restrictions": {"type": "array", "items": {"enum": restrictions.ALL}, "description": "Host restrictions"}
        },
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=[HostType.SERVER],
)
def update_host(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    """Modify the specified host information."""

    if len(request) != 1:
        raise BadRequestError("Exactly one parameter must be specified for update.")

    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    update_field, update_value = next(iter(request.items()))

    if not reason:
        raise BadRequestError("Please specify a ticket or human-readable reason")

    if update_field == "restrictions":
        allowed_states = HostState.ALL_ASSIGNED
        host_restrictions = strip_restrictions(update_value, strip_to_none=True)
        update_kwargs = fix_mongo_set_kwargs(set__restrictions=host_restrictions)
    else:
        raise LogicalError()

    host_query = get_host_query(
        issuer, query_args.get("ignore_maintenance", False), allowed_states, **host_id_query.kwargs(inv=host.inv)
    )
    with audit_log.on_update_host(
        issuer, host.project, host.inv, host.name, host.uuid, request, reason, scenario_id=host.scenario_id
    ):
        if host.state in allowed_states:
            updated_host = Host.objects(**host_query).modify(new=True, **update_kwargs)
            if updated_host is not None:
                return api_response(updated_host.to_api_obj())

        raise InvalidHostStateError(host, allowed_states=allowed_states)


@host_id_handler(
    "/hosts/<host_id>/extra_vlans",
    get_api_action_methods(),
    {
        "type": "object",
        "properties": {"vlans": {"type": "array", "items": get_vlan_id_schema(), "description": "VLAN IDs"}},
        "required": ["vlans"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=[HostType.SERVER],
)
def modify_extra_vlans(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    """Modify extra VLANs that should by assigned to the host."""

    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    action = get_api_action()
    if action == API_ACTION_REMOVE and request is None:
        vlans = None
    elif action == API_ACTION_SET and not request["vlans"]:
        vlans = None
    else:
        vlans = sorted(set(request["vlans"]))

    update_kwargs = get_api_action_update_query("extra_vlans", action, vlans)

    with audit_log.on_update_host(
        issuer,
        host.project,
        host.inv,
        host.name,
        host.uuid,
        {"extra_vlans": {action: vlans}},
        reason,
        scenario_id=host.scenario_id,
    ):
        if action != API_ACTION_REMOVE and vlans is not None:
            walle.projects.authorize_vlans(host.project, vlans)

        allowed_states = HostState.ALL_ASSIGNED
        allowed_statuses = HostStatus.ALL_STEADY
        host_query = get_host_query(
            issuer,
            query_args.get("ignore_maintenance", False),
            allowed_states,
            allowed_statuses,
            **host_id_query.kwargs(inv=host.inv)
        )

        if host.state in allowed_states and host.status in allowed_statuses:
            updated_host = Host.objects(**host_query).only("extra_vlans").modify(new=True, **update_kwargs)

            if updated_host is not None:
                return api_response({"extra_vlans": updated_host.extra_vlans or []})

        raise InvalidHostStateError(host, allowed_states=allowed_states, allowed_statuses=allowed_statuses)


@host_id_handler(
    "/hosts/<host_id>/force-status",
    "POST",
    {
        "properties": {
            "status": {
                "enum": HostStatus.ALL_STEADY,
                "description": "Status that will be forced for the specified host",
            },
            "ticket_key": {"type": "string", "description": "Attach ticket to host's manual status."},
        },
        "required": ["status"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def force_host_status(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    """Forces status for the specified host (cancels any running tasks)."""
    # TODO Deprecate all functionality except task cancellation
    status = request.pop("status")

    if "ticket_key" in request and status not in {HostStatus.MANUAL, HostStatus.DEAD}:
        raise RequestValidationError(
            "Ticket is only accepted for '{}' and '{}' statuses.", HostStatus.MANUAL, HostStatus.DEAD
        )

    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    if status == HostStatus.MANUAL and host.status != HostStatus.INVALID:
        raise RequestValidationError(
            "Setting maintenance by forcing status \"manual\" is deprecated. "
            "Please use the \"set-maintenance\" method instead."
        )

    if (
        host.is_maintenance() == (status == HostStatus.default(HostState.MAINTENANCE))
        or host.status == HostStatus.INVALID
        and issuer != authorization.ISSUER_WALLE
    ):
        # not changing state, just status
        # ~ /cancel-task + un-invalidating
        walle.host_status.force_status(
            issuer,
            host,
            status,
            ignore_maintenance=query_args.get("ignore_maintenance", False),
            reason=reason,
            ticket_key=request.get("ticket_key"),
        )
    elif host.is_maintenance():
        # from maintenance
        # ~ /set-assigned
        schedule_setting_assigned_state(
            issuer,
            TaskType.MANUAL,
            host,
            status,
            ignore_maintenance=query_args.get("ignore_maintenance", False),
            reason=reason,
        )
    else:
        raise LogicalError()

    return api_response(host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/cancel-task",
    "POST",
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def cancel_host_task_api(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    if host.task:
        walle.host_status.cancel_task(
            issuer, host, reason, ignore_maintenance=query_args.get("ignore_maintenance", False)
        )

    # If there's no task or it's already completed don't bother the user. Don't change the status either.
    return api_response(host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/set-maintenance",
    "POST",
    {
        "properties": dict(get_maintenance_params(), **_get_new_task_params(with_check=False)),
        "additionalProperties": False,
        "required": ["ticket_key"],
    },
    authenticate=True,
    with_ignore_maintenance=False,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=HOST_TYPES_WITH_PARTIAL_AUTOMATION,
)
def set_host_maintenance_api(issuer, host_id_query, allowed_host_types, request, reason):
    host = get_authorized_host(
        issuer, host_id_query, {"ignore_maintenance": True}, allowed_host_types=allowed_host_types
    )

    set_host_maintenance_for_host(issuer, host, reason=reason, **request)
    return api_response(host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/change-maintenance",
    "POST",
    {
        "properties": get_maintenance_params(is_new_maintenance=False, with_default_operation_state=False),
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=False,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=[HostType.SERVER, HostType.VM],
)
def change_host_maintenance_api(issuer, host_id_query, allowed_host_types, request, reason):
    host = get_authorized_host(
        issuer, host_id_query, {"ignore_maintenance": True}, allowed_host_types=allowed_host_types
    )

    if host.state != HostState.MAINTENANCE:
        raise InvalidHostStateError(host, allowed_states=[HostState.MAINTENANCE])

    with audit_log.on_change_host_maintenance(
        issuer, host.project, host.inv, host.name, host.uuid, reason=reason, scenario_id=host.scenario_id, **request
    ):
        change_host_maintenance(issuer, host, reason=reason, **request)
    return api_response(host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/set-assigned",
    "POST",
    {
        "properties": dict(
            {
                "status": {
                    "enum": HostStatus.ALL_ASSIGNED,
                    "description": "Status that will be assigned for the specified host",
                },
                "power_on": {"type": "boolean", "description": "Power on the host. Default is false"},
            },
            **_get_new_task_params(with_cms=False)
        ),
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=HOST_TYPES_WITH_PARTIAL_AUTOMATION,
)
def set_host_assigned_state(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    schedule_setting_assigned_state(
        issuer,
        task_type=TaskType.MANUAL,
        host=host,
        status=request.get("status"),
        power_on=request.get("power_on", False),
        reason=reason,
        **_get_schedule_task_params(request, query_args, with_cms=False)
    )
    return api_response(host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/location",
    "PUT",
    {
        "type": "object",
        "properties": {
            "switch": {"type": "string", "description": "Name of switch"},
            "port": {"type": "string", "description": "Name of port"},
            "network_source": {"enum": walle_constants.NETWORK_SOURCES, "description": "Source of information"},
            "country": {"type": "string", "description": "Country"},
            "city": {"type": "string", "description": "City"},
            "datacenter": {"type": "string", "description": "Datacenter"},
            "queue": {"type": "string", "description": "Queue"},
            "rack": {"type": "string", "description": "Rack"},
            "unit": {"type": "string", "description": "Unit"},
            "physical_timestamp": {"type": "string", "description": "Physical location updated at"},
            "short_datacenter_name": {"type": "string", "description": "Short datacenter name"},
            "short_queue_name": {"type": "string", "description": "Short datacenter's queue name"},
        },
        "required": ["short_datacenter_name", "short_queue_name", "switch"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
    allowed_host_types=[HostType.VM],
)
def set_vm_location(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    location = HostLocation(**request)

    with audit_log.on_update_host(
        issuer, host.project, host.inv, host.name, host.uuid, request, reason, scenario_id=host.scenario_id
    ):

        allowed_states = HostState.ALL
        update_kwargs = fix_mongo_set_kwargs(set__location=location)
        host_query = get_host_query(
            issuer, query_args.get("ignore_maintenance", False), allowed_states, **host_id_query.kwargs(inv=host.inv)
        )
        updated_host = Host.objects(**host_query).only("location").modify(new=True, **update_kwargs)
        return api_response(updated_host.to_api_obj())
