"""Hosts"""  # Will be used as category name in API reference

import http.client
import logging

import walle.constants
import walle.host_fsm.control
import walle.network
import walle.projects
import walle.tasks
from walle import constants as constant, network, audit_log, restrictions, constants, host_operations
from walle._tasks.task_args import (
    DeleteHostTaskArgs,
    PowerOnTaskArgs,
    PowerOffTaskArgs,
    RebootTaskArgs,
    VlanSwitchTaskArgs,
    KexecRebootTaskArgs,
)
from walle._tasks.task_creator import (
    get_delete_host_task_stages,
    get_power_on_task_stages,
    get_power_off_task_stages,
    get_reboot_task_stages,
    get_kexec_reboot_task_stages,
    get_vlan_switch_task_stages,
)
from walle._tasks.task_provider import schedule_task_from_api
from walle.admin_requests.severity import get_eine_tag_by_repair_request_severity
from walle.authorization import blackbox, iam, has_iam
from walle.clients.eine import ProfileMode
from walle.constants import PROVISIONERS, SshOperation, NetworkTarget, HostType, HOST_TYPES_WITH_PARTIAL_AUTOMATION
from walle.errors import (
    ResourceConflictError,
    RequestValidationError,
    InvalidHostConfiguration,
    BadRequestError,
    UnprocessableRequest,
)
from walle.expert import dmc, juggler
from walle.expert.types import CheckType, CheckGroup
from walle.host_operations import check_host_not_missing_from_preorder
from walle.hosts import TaskType, HostState, HostStatus, Host
from walle.projects import get_by_id
from walle.util.api import host_id_handler, api_response, api_handler, admin_request
from walle.util.deploy_config import DeployConfigPolicies
from walle.util.host_health import get_human_reasons
from walle.util.misc import drop_none
from walle.util.tasks import check_post_code_allowed, _network_target_project
from walle.util.text import enumeration_join
from walle.views.api.common import get_vlan_id_schema, get_eine_tags_schema
from walle.views.api.host_api.common import get_authorized_host, get_maintenance_params
from walle.views.helpers.maintenance import set_host_maintenance_for_host, change_host_maintenance

log = logging.getLogger(__name__)


def _get_new_task_params(with_cms=True, with_check=True, with_auto_healing=True):
    params = {
        "disable_admin_requests": {
            "type": "boolean",
            "description": "Don't issue any admin requests if something is broken - just fail the task. "
            "Default is false",
        },
    }

    if with_cms:
        params["ignore_cms"] = {"type": "boolean", "description": "Don't acquire permission from CMS. Default is false"}

    if with_check:
        params["check"] = {
            "type": "boolean",
            "description": "Check host after task completion and consider it as failed if host is not healthy "
            "Default is true",
        }

        if with_auto_healing:
            params["with_auto_healing"] = {
                "type": "boolean",
                "description": "Try to automatically repair the host if task will fail. Default is false",
            }

    return params


def _get_vlan_switch_params(host, vlans, network_target):
    # Here the preference is given to the network_target, vlans are used if they're
    # custom (not 999, 542 or from vlan_scheme).

    vlans_from_scheme = network.get_host_expected_vlans(host).vlans
    result_vlans = None

    if network_target is None and vlans is not None:
        if not vlans:
            network_target = _network_target_project(host)

        elif vlans == [constant.PARKING_VLAN]:
            network_target = NetworkTarget.PARKING

        elif vlans == [constant.PROFILING_VLAN]:
            network_target = NetworkTarget.SERVICE

        elif vlans == vlans_from_scheme:
            network_target = _network_target_project(host)

        else:
            result_vlans = vlans

    elif network_target is None and vlans is None:
        network_target = _network_target_project(host)

    if vlans:
        walle.projects.authorize_vlans(host.project, vlans)

    return result_vlans, network_target


@api_handler(
    "/hosts",
    "POST",
    {
        "type": "object",
        "properties": dict(
            {
                "inv": {"type": "integer", "minimum": 0, "description": "Inventory number"},
                "name": {
                    "type": "string",
                    "pattern": r"^([a-zA-Z0-9]+(-[a-zA-Z0-9]+)*\.)+[a-z]{2,}$",
                    "description": "FQDN",
                },
                "project": {"type": "string", "description": "Project ID"},
                "state": {
                    "enum": [HostState.FREE, HostState.ASSIGNED, HostState.MAINTENANCE],
                    "description": "State that will be set for the host. Default is {}".format(HostState.ASSIGNED),
                },
                "status": {
                    "enum": HostStatus.ALL_STEADY,
                    "description": "Status that will be set for the host. Default is {}".format(HostStatus.READY),
                },
                "provisioner": {"enum": PROVISIONERS, "description": "Provisioning system"},
                "config": {"type": "string", "description": "Deploy config"},
                "deploy_config_policy": {
                    "enum": DeployConfigPolicies.get_all_names(),
                    "description": "Deploy config policy",
                },
                "deploy_tags": get_eine_tags_schema(for_profile=False),
                "deploy_network": {
                    "enum": NetworkTarget.DEPLOYABLE,
                    "description": "Deploy host either in service or in project VLANs",
                },
                "restrictions": {
                    "type": "array",
                    "items": {"enum": restrictions.ALL},
                    "description": "Host restrictions",
                },
                "dns": {"type": "boolean", "description": "Check and setup DNS records for the host. Default is false"},
                "instant": {
                    "type": "boolean",
                    "description": "instant operation, don't schedule deletion task, don't ask CMS, etc. "
                    "Default is false",
                },
                "maintenance_properties": {
                    "type": "object",
                    "properties": get_maintenance_params(with_cms_action=False),
                    "additionalProperties": False,
                },
            },
            **_get_new_task_params()
        ),
        "required": ["project"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project"),
)
def add_host(issuer, request, reason):
    """Add a new host to the system."""
    host = host_operations.add_host(issuer, reason=reason, **request)
    return api_response(host.to_api_obj(), code=http.client.CREATED)


@host_id_handler(
    "/hosts/<host_id>",
    "DELETE",
    params=dict(
        {
            "lui": {
                "type": "boolean",
                "description": "delete the host also from LUI (for hosts provisioned by LUI). Default is false",
            },
            "instant": {
                "type": "boolean",
                "description": "instant operation, don't schedule deletion task, don't ask CMS, etc. Default is false",
            },
        },
        **_get_new_task_params(with_check=False, with_auto_healing=False)
    ),
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.DeleteHostApiIamPermission("host_id"),
    allowed_host_types=constant.HOST_TYPES_WITH_MANUAL_OPERATION,
)
def delete_host(issuer, host_id_query, allowed_host_types, query_args, request, reason):
    """Deletes the specified host from the system."""

    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    lui = query_args.pop("lui", False)
    ignore_maintenance = query_args.get("ignore_maintenance", False)
    host_query = host_id_query.kwargs(inv=host.inv)

    if lui:
        if host.state == HostState.FREE and host.name is None:
            raise BadRequestError("Unable to delete the host from LUI: it has no name assigned to it.")
        elif host.state != HostState.FREE and host.get_deploy_configuration().provisioner != constants.PROVISIONER_LUI:
            raise BadRequestError("Unable to delete the host from LUI: it's not provisioned by LUI.")

    if query_args.get("instant") or host.state == HostState.FREE:
        removed_host = host_operations.instant_delete_host(
            issuer, host, host_query, lui=lui, ignore_maintenance=ignore_maintenance, reason=reason
        )
        if removed_host is not None:
            return "", http.client.NO_CONTENT

    else:
        task_args = DeleteHostTaskArgs(
            issuer=issuer,
            task_type=TaskType.MANUAL,
            project=host.project,
            host_inv=host.inv,
            host_name=host.name,
            host_uuid=host.uuid,
            scenario_id=host.scenario_id,
            cms_task_id=host.cms_task_id,
            lui=lui,
            reason=reason,
            **_get_schedule_task_params(query_args, query_args, with_check=False, with_auto_healing=False)
        )
        schedule_task_from_api(host, task_args, get_delete_host_task_stages)
        return "", http.client.NO_CONTENT


@host_id_handler(
    "/hosts/<host_id>/switch-vlans",
    "POST",
    {
        "type": "object",
        "properties": {
            "vlans": {"type": "array", "items": get_vlan_id_schema(), "description": "Allowed VLAN IDs"},
            "native_vlan": get_vlan_id_schema("Native VLAN ID"),
            "network_target": {"enum": NetworkTarget.ALL, "description": "Target network"},
            "update_network_location": {
                "type": "boolean",
                "description": "Update host's network location using swp-up profile mode. Default is false",
            },
        },
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=[constant.HostType.SERVER],
)
def switch_host_vlan(issuer, query_args, allowed_host_types, host_id_query, request, reason):
    """Switches VLANs of the specified host."""
    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    native_vlan = request.get("native_vlan")
    vlans = request.get("vlans")
    update_network_location = request.get("update_network_location")

    if vlans is not None and native_vlan is not None:
        vlans.append(native_vlan)
    vlans = sorted(set(vlans)) if vlans is not None else None

    network_target = request.get("network_target")

    if vlans is not None and network_target is not None:
        return api_response(
            {"message": "Use only one parameter: network_target or vlans"},
            code=http.client.BAD_REQUEST,
        )

    vlans, network_target = _get_vlan_switch_params(host, vlans, network_target)
    network_update_args = host.deduce_profile_configuration(profile_mode=ProfileMode.SWP_UP)
    task_args = VlanSwitchTaskArgs(
        issuer=issuer,
        task_type=TaskType.MANUAL,
        project=host.project,
        host_inv=host.inv,
        host_name=host.name,
        host_uuid=host.uuid,
        scenario_id=host.scenario_id,
        vlans=vlans,
        native_vlan=native_vlan,
        network_target=network_target,
        ignore_maintenance=query_args.get("ignore_maintenance", False),
        reason=reason,
        network_update_args=network_update_args,
        update_network_location=update_network_location,
    )
    schedule_task_from_api(host, task_args, get_vlan_switch_task_stages)

    return "", http.client.NO_CONTENT


@host_id_handler(
    "/hosts/<host_id>/power-on",
    "POST",
    {
        "type": "object",
        "properties": _get_new_task_params(with_cms=False),
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=[constant.HostType.SERVER],
)
def power_on_host(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    """Powers on the specified host."""

    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    schedule_task_params = _get_schedule_task_params(request, query_args, with_cms=False)
    check_post_code, check_post_code_reason = check_post_code_allowed(
        host, TaskType.MANUAL, schedule_task_params.get('with_auto_healing', False)
    )
    checks_to_monitor = sorted(set(CheckGroup.NETWORK_AVAILABILITY))

    task_args = PowerOnTaskArgs(
        issuer=issuer,
        task_type=TaskType.MANUAL,
        project=host.project,
        host_inv=host.inv,
        host_name=host.name,
        host_uuid=host.uuid,
        scenario_id=host.scenario_id,
        reason=reason,
        check_post_code=check_post_code,
        check_post_code_reason=check_post_code_reason,
        operation_restrictions=restrictions.REBOOT,
        checks_to_monitor=checks_to_monitor,
        **schedule_task_params
    )
    schedule_task_from_api(host, task_args, get_power_on_task_stages)

    return api_response(host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/power-off",
    "POST",
    {
        "type": "object",
        "properties": dict(
            get_maintenance_params(with_power_off=False, with_default_operation_state=False),
            **_get_new_task_params(with_check=False)
        ),
        "additionalProperties": False,
        "required": ["ticket_key"],
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=[constant.HostType.SERVER],
)
def power_off_host(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    """Powers off and switches to maintenance state the specified host."""
    params = request
    params["power_off"] = True

    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    if host.is_maintenance():
        task_args = PowerOffTaskArgs(
            issuer=issuer,
            task_type=TaskType.MANUAL,
            project=host.project,
            host_inv=host.inv,
            host_name=host.name,
            host_uuid=host.uuid,
            scenario_id=host.scenario_id,
            reason=reason,
            soft=True,
            operation_restrictions=restrictions.REBOOT,
            **_get_schedule_task_params(request, query_args, with_check=False)
        )
        schedule_task_from_api(host, task_args, get_power_off_task_stages)

        change_maintenance_params = dict(
            operation_state=params.get("operation_state"), ticket_key=params.get("ticket_key")
        )
        with audit_log.on_change_host_maintenance(
            issuer,
            host.project,
            host.inv,
            host.name,
            host.uuid,
            reason=reason,
            scenario_id=host.scenario_id,
            **change_maintenance_params
        ):

            # A little hacky but it is ok since we in fact work with a flag in db
            change_host_maintenance(issuer, host, **change_maintenance_params)
    else:
        task_params = _get_schedule_task_params(request, query_args, with_check=False, with_ignore_maintenance=False)
        params.update(task_params)

        set_host_maintenance_for_host(issuer, host, reason=reason, **params)

    return api_response(host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/reboot",
    "POST",
    {
        "type": "object",
        "properties": dict(
            {
                "ssh": {
                    "enum": SshOperation.ALL,
                    "default": SshOperation.FORBID,
                    "description": "Allow using ssh for the operation.",
                },
            },
            **_get_new_task_params()
        ),
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=constant.HOST_TYPES_WITH_PARTIAL_AUTOMATION,
)
def reboot_host(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    """Reboots the specified host."""

    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    ssh = request.get("ssh", SshOperation.FORBID)

    if ssh != SshOperation.ONLY and host.type not in HOST_TYPES_WITH_PARTIAL_AUTOMATION:
        raise BadRequestError("This host support SSH rebooting only")

    if ssh != SshOperation.FORBID:
        if host.type != HostType.SERVER:
            raise BadRequestError("SSH rebooting is not possible for this host")
        project = walle.projects.get_by_id(host.project)
        if not walle.projects.is_reboot_via_ssh_enabled(project):
            raise BadRequestError("SSH rebooting is disabled in project")

    schedule_task_params = _get_schedule_task_params(request, query_args)
    check_post_code, check_post_code_reason = check_post_code_allowed(
        host, TaskType.MANUAL, schedule_task_params.get('with_auto_healing', False)
    )
    reboot_task_args = RebootTaskArgs(
        issuer=issuer,
        task_type=TaskType.MANUAL,
        project=host.project,
        host_inv=host.inv,
        host_name=host.name,
        host_uuid=host.uuid,
        scenario_id=host.scenario_id,
        ssh=ssh,
        reason=reason,
        check_post_code=check_post_code,
        check_post_code_reason=check_post_code_reason,
        from_current_task=False,
        checks_to_monitor=set(CheckGroup.NETWORK_AVAILABILITY) | {CheckType.W_META},
        operation_restrictions=restrictions.REBOOT,
        without_ipmi=host.type != HostType.SERVER,
        **schedule_task_params
    )
    schedule_task_from_api(host, reboot_task_args, get_reboot_task_stages)

    return api_response(host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/kexec_reboot",
    "POST",
    {
        "type": "object",
        "properties": _get_new_task_params(),
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=[constant.HostType.SERVER],
)
def kexec_reboot_host(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    """Reboots the specified host with kexec. Just for api only at least for now"""

    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)
    schedule_task_params = _get_schedule_task_params(request, query_args)

    project = walle.projects.get_by_id(host.project)
    if host.type != HostType.SERVER or not walle.projects.is_reboot_via_ssh_enabled(project):
        ssh = SshOperation.FORBID
    else:
        ssh = SshOperation.FALLBACK

    kexec_reboot_task_args = KexecRebootTaskArgs(
        issuer=issuer,
        task_type=TaskType.MANUAL,
        project=host.project,
        host_inv=host.inv,
        host_name=host.name,
        host_uuid=host.uuid,
        scenario_id=host.scenario_id,
        reason=reason,
        from_current_task=False,
        operation_restrictions=restrictions.REBOOT,
        checks_to_monitor=set(CheckGroup.NETWORK_AVAILABILITY) | {CheckType.W_META},
        ssh=ssh,
        without_ipmi=host.type != HostType.SERVER,
        **schedule_task_params
    )
    schedule_task_from_api(host, kexec_reboot_task_args, get_kexec_reboot_task_stages)

    return api_response(host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/prepare",
    "POST",
    {
        "type": "object",
        "properties": dict(
            {
                "profile": {"type": "string", "description": "Einstellung profile"},
                "profile_tags": get_eine_tags_schema(),
                "skip_profile": {"type": "boolean", "description": "Skip profiling. Default is false"},
                "provisioner": {"enum": PROVISIONERS, "description": "Provisioning system"},
                "config": {"type": "string", "description": "Deploy config"},
                "deploy_tags": get_eine_tags_schema(for_profile=False),
                "deploy_config_policy": {
                    "enum": DeployConfigPolicies.get_all_names(),
                    "description": "Deploy config policy",
                },
                "deploy_network": {
                    "enum": NetworkTarget.DEPLOYABLE,
                    "description": "Deploy host either in service or in project VLAN's",
                },
                "restrictions": {
                    "type": "array",
                    "items": {"enum": restrictions.ALL},
                    "description": "Host restrictions",
                },
                "extra_vlans": {
                    "type": "array",
                    "items": get_vlan_id_schema(),
                    "description": "Extra VLAN IDs to be assigned to the host",
                },
                "keep_name": {"type": "boolean", "description": "Keep host's current name. Default is false"},
                "update_firmware": {
                    "type": "boolean",
                    "description": "Update host's disk/bios/hba/nic firmware. Default is false",
                },
            },
            **_get_new_task_params()
        ),
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=[constant.HostType.SERVER],
)
def prepare_host(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    """Prepares the specified host - i.e. profiles it, sets up DNS, etc"""

    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    check_host_not_missing_from_preorder(host.inv)
    walle.tasks.schedule_prepare(
        issuer,
        TaskType.MANUAL,
        host,
        reason=reason,
        host_restrictions=request.pop("restrictions", None),  # rename parameter
        ignore_maintenance=query_args.get("ignore_maintenance", False),
        **request
    )
    return api_response(host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/profile",
    "POST",
    {
        "type": "object",
        "properties": dict(
            {
                "profile": {"type": "string", "description": "Einstellung profile"},
                "profile_tags": get_eine_tags_schema(),
                "redeploy": {"type": "boolean", "description": "Redeploy host after profiling. Default is false"},
                "provisioner": {"enum": PROVISIONERS, "description": "Provisioning system"},
                "config": {"type": "string", "description": "Deploy config"},
                "deploy_config_policy": {
                    "enum": DeployConfigPolicies.get_all_names(),
                    "description": "Deploy config policy",
                },
                "deploy_tags": get_eine_tags_schema(for_profile=False),
                "deploy_network": {
                    "enum": NetworkTarget.DEPLOYABLE,
                    "description": "Deploy host either in service or in project VLAN's",
                },
                "profile_mode": {"enum": ProfileMode.ALL + [None], "description": "Profile mode"},
                "repair_request_severity": {
                    "enum": walle.projects.RepairRequestSeverity.ALL + [None],
                    "description": "Repair Request Severity for Eine",
                },
            },
            **_get_new_task_params()
        ),
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=[constant.HostType.SERVER],
)
def profile_host(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    """Profile the specified host with the specified Einstellung profile and optionally redeploy host after that."""

    if "provisioner" in request and "config" not in request:
        raise RequestValidationError("Deploy config is required if provisioner is specified.")
    if "tags" in request and "config" not in request:
        raise RequestValidationError("Deploy config is required if deploy tags are specified.")
    if request.get("deploy_network") and request.get("redeploy") is not True and not request.get("config"):
        raise RequestValidationError(
            "Will not implicitly trigger host redeploy by deploy network option. "
            "If you want to redeploy host, add redeploy option explicitly."
        )
    if request.get("config") and request.get("redeploy") is False:
        raise RequestValidationError("Can't redeploy host with given config since redeploy is explicitly forbidden.")

    repair_request_severity = request.get("repair_request_severity")
    if repair_request_severity == walle.projects.RepairRequestSeverity.HIGH and not has_iam():
        blackbox.authorize(
            issuer,
            "'HIGH' Repair Request Severity can use only admins",
            authorize_admins=True,
            authorize_by_group=False,
        )
    repair_request_severity_tag = (
        get_eine_tag_by_repair_request_severity(repair_request_severity) if repair_request_severity else None
    )

    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    walle.tasks.schedule_profile(
        issuer,
        TaskType.MANUAL,
        host,
        reason=reason,
        profile=request.get("profile"),
        profile_tags=request.get("profile_tags"),
        redeploy=request.get("redeploy"),
        provisioner=request.get("provisioner"),
        deploy_config=request.get("config"),
        deploy_config_policy=request.get("deploy_config_policy"),
        deploy_tags=request.get("deploy_tags"),
        deploy_network=request.get("deploy_network"),
        profile_mode=request.get("profile_mode"),
        repair_request_severity=repair_request_severity_tag,
        **_get_schedule_task_params(request, query_args)
    )
    return api_response(host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/redeploy",
    "POST",
    {
        "type": "object",
        "properties": dict(
            {
                "provisioner": {"enum": PROVISIONERS, "description": "Provisioning system"},
                "tags": get_eine_tags_schema(for_profile=False),
                "config": {"type": "string", "description": "Deploy config"},
                "network": {
                    "enum": NetworkTarget.DEPLOYABLE,
                    "description": "Deploy host either in service or in project VLAN's",
                },
                "deploy_config_policy": {
                    "enum": DeployConfigPolicies.get_all_names(),
                    "description": "Deploy config policy",
                },
            },
            **_get_new_task_params()
        ),
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=[constant.HostType.SERVER],
)
def redeploy_host(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    """Redeploys the specified host."""

    if "provisioner" in request and "config" not in request:
        raise RequestValidationError("Config is required if provisioner is specified.")
    if "tags" in request and "config" not in request:
        raise RequestValidationError("Config is required if tags are specified.")

    provisioner = request.get("provisioner")
    deploy_config = request.get("config")
    deploy_tags = request.get("tags")
    deploy_network = request.get("network")
    deploy_config_policy = request.get("deploy_config_policy")

    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    walle.tasks.schedule_redeploy(
        issuer,
        TaskType.MANUAL,
        host,
        provisioner=provisioner,
        deploy_config=deploy_config,
        deploy_tags=deploy_tags,
        deploy_network=deploy_network,
        deploy_config_policy=deploy_config_policy,
        reason=reason,
        **_get_schedule_task_params(request, query_args)
    )

    return api_response(host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/switch-project",
    "POST",
    {
        "type": "object",
        "properties": dict(
            {
                "project": {"type": "string", "description": "Project ID to switch the host to"},
                "release": {"type": "boolean", "description": "Release the host before switching to the project"},
                "erase_disks": {"type": "boolean", "description": "Erase disks while releasing host. Default is true"},
                "restrictions": {
                    "type": "array",
                    "items": {"enum": restrictions.ALL},
                    "description": "Restrictions to assign to the host after switching",
                },
                "force": {
                    "type": "boolean",
                    "description": "Force switching without checking the projects for compatibility. Default is false",
                },
            },
            **_get_new_task_params(with_cms=True, with_check=False, with_auto_healing=False)
        ),
        "required": ["project", "release"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=constant.HOST_TYPES_WITH_MANUAL_OPERATION,
)
def switch_project(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    """Moves a host from one project to another."""

    release = request["release"]
    if "restrictions" in request and release:
        raise RequestValidationError("Restrictions can be set only when host is switching without releasing.")

    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    target_project = get_by_id(request["project"])
    if host.project == target_project.id:
        raise ResourceConflictError("The host is already in '{}' project.", target_project.id)
    if host.type != target_project.type:
        raise ResourceConflictError(
            "Host can only be switched to a project with the same type. Host type '{}', target project type '{}'",
            host.type,
            target_project.type,
        )

    walle.tasks.schedule_project_switching(
        issuer,
        TaskType.MANUAL,
        host,
        target_project.id,
        request.get("restrictions"),
        release,
        erase_disks=request.get("erase_disks", True),
        force=request.get("force", False),
        reason=reason,
        **_get_schedule_task_params(request, query_args, with_check=False, with_auto_healing=False)
    )
    return api_response(host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/release-host",
    "POST",
    {
        "type": "object",
        "properties": dict(
            {
                "erase_disks": {"type": "boolean", "description": "Erase disks while releasing host. Default is true"},
            },
            **_get_new_task_params(with_cms=True, with_check=False, with_auto_healing=False)
        ),
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=[constant.HostType.SERVER],
)
def release_host(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    """Release host (clear disks, set as free)."""
    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    walle.tasks.schedule_host_release(
        issuer,
        TaskType.MANUAL,
        host,
        erase_disks=request.get("erase_disks", True),
        reason=reason,
        **_get_schedule_task_params(request, query_args, with_check=False, with_auto_healing=False)
    )

    return api_response(host.to_api_obj())


def _get_schedule_task_params(
    request, query, with_cms=True, with_check=True, with_auto_healing=True, with_ignore_maintenance=True
):
    # TODO: How to understand where to get params from: query or request :)
    return drop_none(
        dict(
            ignore_maintenance=query.get("ignore_maintenance", False) if with_ignore_maintenance else None,
            ignore_cms=request.get("ignore_cms", False) if with_cms else None,
            disable_admin_requests=request.get("disable_admin_requests", False),
            monitor_on_completion=request.get("check", True) if with_check else None,
            with_auto_healing=request.get("with_auto_healing") if with_check and with_auto_healing else None,
        )
    )


@host_id_handler(
    "/hosts/<host_id>/check-dns",
    "POST",
    {
        "type": "object",
        "properties": _get_new_task_params(with_cms=False),
        "additionalProperties": False,
    },
    authenticate=True,
    with_ignore_maintenance=True,
    with_reason=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=[constant.HostType.SERVER],
)
def check_host_dns(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    """Check DNS records for the host and fix errors if any."""

    # NB: we don't ask CMS for permission for DNS fix operations

    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    try:
        walle.tasks.schedule_dns_check(
            issuer,
            TaskType.MANUAL,
            host,
            reason=reason,
            **_get_schedule_task_params(request, query_args, with_cms=False)
        )
    except InvalidHostConfiguration as e:
        raise ResourceConflictError(str(e))

    return api_response(host.to_api_obj())


@host_id_handler("/hosts/<host_id>/power-status", "GET", iam_permissions=iam.GetHostApiIamPermission("host_id"))
def get_power_status(host_id_query):
    host = Host.get_by_host_id_query(host_id_query)

    if host.type != HostType.SERVER or host.status == HostStatus.INVALID:
        return api_response({"is_powered_on": False})

    try:
        client = host.get_ipmi_client()
        return api_response({"is_powered_on": client.is_power_on()})
    except Exception as e:
        log.error("Can't get power status for %s: %s", host.inv, e)
        raise UnprocessableRequest(e)


@host_id_handler(
    "/hosts/<host_id>/fqdn-deinvalidation",
    "POST",
    {
        "type": "object",
        "properties": dict(
            {
                "release": {
                    "type": "boolean",
                    "description": "Release the host (erase disks, power-off, set state as free). Default is false",
                },
                "clear_old_fqdn_records": {
                    "type": "boolean",
                    "description": "Clear DNS records, remove host from LUI. Default is false",
                },
            },
            **_get_new_task_params(with_cms=True, with_check=False, with_auto_healing=False)
        ),
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    with_ignore_maintenance=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=constant.HOST_TYPES_WITH_PARTIAL_AUTOMATION,
)
def fqdn_deinvalidation(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    ignore_maintenance = query_args.get("ignore_maintenance")

    walle.tasks.schedule_fqdn_deinvalidation(
        issuer, TaskType.MANUAL, host, reason=reason, ignore_maintenance=ignore_maintenance, **request
    )

    return api_response(host.to_api_obj())


@host_id_handler(
    "/hosts/<host_id>/handle-failure",
    "POST",
    {
        "type": "object",
        "properties": _get_new_task_params(),
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    with_ignore_maintenance=True,
    iam_permissions=iam.UpdateHostApiIamPermission("host_id"),
    allowed_host_types=constant.HOST_TYPES_WITH_PARTIAL_AUTOMATION,
)
@admin_request
def handle_host_failure(issuer, query_args, host_id_query, allowed_host_types, request, reason):
    host = get_authorized_host(issuer, host_id_query, query_args, allowed_host_types=allowed_host_types)

    if host.state not in HostState.ALL_ASSIGNED:
        raise ResourceConflictError(
            "The host is in '{}' state. This operation only allowed in {} states.",
            host.state,
            enumeration_join(sorted(HostState.ALL_ASSIGNED)),
        )

    if host.status not in HostStatus.ALL_STEADY:
        raise ResourceConflictError(
            "The has a '{}' task running. This operation only allowed in steady statuses.", host.status
        )

    decision_maker = dmc.get_decision_maker(host.get_project())

    reasons = juggler.get_host_health_reasons(host, decision_maker)
    human_reasons = get_human_reasons(reasons)

    decision = decision_maker.make_decision(host, reasons)
    message = dmc.handle_user_executed(
        host,
        decision,
        human_reasons,
        decision_maker,
        reason=reason,
        issuer=issuer,
        ignore_maintenance=query_args.get("ignore_maintenance"),
    )

    return api_response({"host": host.to_api_obj(), "dmc_message": message})
