import functools
import json
import logging

from werkzeug.exceptions import BadRequest

from sepelib.core import config
from walle.authorization import iam
from walle.clients import staff
from walle.clients.juggler import send_event, JugglerCheckStatus
from walle.constants import LOGIN_RE
from walle.idm.common import idm_string_path_to_list
from walle.idm.role_tree import ChildNotFound
from walle.idm.traversal import get_node
from walle.idm.tree_views import RoleTreeView, RoleMembersView
from walle.util.api import api_handler, api_response

logger = logging.getLogger(__name__)

INTEGER_STR_RE = r"^\d+$"


def fields_json_decoder(form):
    form_data = dict(form.items())  # form is ImmutableMultiDict, dict(form) would make dict of with lists in values
    fields_in_json = {"role", "fields"}
    for field in fields_in_json:
        if field in form_data:
            form_data[field] = json.loads(form_data[field])
    return form_data


def idm_api_handler(path, method, *args, **kwargs):
    if method == "POST":
        kwargs["allowed_mime_type"] = "application/x-www-form-urlencoded"
        kwargs["form_processor"] = fields_json_decoder
    return api_handler(
        path,
        method,
        authenticate=False,
        allowed_tvm_sources_aliases_fn=lambda: list(config.get_value("idm.tvm_aliases").keys()),
        *args,
        **kwargs
    )


def idm_response(handle_name):
    def decorator(func):
        @functools.wraps(func)
        def decorated_function(*args, **kwargs):
            response_body = {}
            idm_status_code = 0
            juggler_status = JugglerCheckStatus.OK
            juggler_message = "OK"

            try:
                response_body = func(*args, **kwargs)
            except Exception as e:
                logger.exception("Got exception while rendering {}".format(handle_name))
                response_body["error"] = str(e)

                idm_status_code = 1
                juggler_status = JugglerCheckStatus.CRIT
                juggler_message = "Got exception while rendering {}: {}".format(handle_name, e)
            finally:
                response_body["code"] = idm_status_code
                send_event(
                    "wall-e{}-handle".format(handle_name.replace("/", "-")),
                    status=juggler_status,
                    message=juggler_message,
                )

                return api_response(response_body)

        return decorated_function

    return decorator


@idm_api_handler(
    "/idm/info/",
    "GET",
    iam_permissions=iam.NoOneApiIamPermission(),
)
@idm_response("/idm/info/")
def get_role_tree():
    return RoleTreeView().to_dict()


@idm_api_handler(
    "/idm/get-all-roles/",
    "GET",
    iam_permissions=iam.NoOneApiIamPermission(),
)
@idm_response("/idm/get-all-roles/")
def get_role_members():
    return RoleMembersView().to_dict()


def get_validated_role_path(path_str):
    role_path = idm_string_path_to_list(path_str)
    if not role_path:
        raise BadRequest("Role path is empty")
    return role_path


def _get_common_push_handle_properties():
    return {
        "login": {"type": "string", "pattern": LOGIN_RE, "description": "User login"},
        "group": {"type": "string", "pattern": INTEGER_STR_RE, "description": "Staff group ID"},
        "role": {"type": "object", "description": "Role path in dict form"},
        "path": {"type": "string", "description": "Role path in file-path-like form"},
        "fields": {"type": ["object", "null"], "description": "Additional fields"},
    }


def _get_group_param(request):
    group = request.get("group")
    if group is not None:
        return int(group)


@idm_api_handler(
    "/idm/add-role/",
    "POST",
    schema={
        "type": "object",
        "properties": _get_common_push_handle_properties(),
        "oneOf": [
            {"required": ["role", "path", "group"]},
            {"required": ["role", "path", "login"]},
        ],
        "additionalProperties": False,
    },
    iam_permissions=iam.NoOneApiIamPermission(),
)
@idm_response("/idm/add-role/")
def add_role_member(request):
    role_path = get_validated_role_path(request["path"])
    role_node = get_node(role_path)

    group_id = _get_group_param(request)
    if group_id is not None:
        member = staff.id_to_group(group_id)
        logger.info("Adding group member %s (%s) into role %s", member, group_id, request["path"])
    else:
        member = request["login"]
        logger.info("Adding member %s into role %s", member, request["path"])

    role_node.add_role_member(member)

    return {}


def _get_fired_param(request):
    return bool(int(request.get("fired", "0")))


def _get_deleted_param(request):
    return bool(int(request.get("deleted", "0")))


@idm_api_handler(
    "/idm/remove-role/",
    "POST",
    schema={
        "type": "object",
        "properties": dict(
            _get_common_push_handle_properties(),
            **{
                "fired": {
                    "type": "string",
                    "pattern": INTEGER_STR_RE,
                    "description": "Integer encoded as string marking if employee was fired",
                },
                "deleted": {
                    "type": "string",
                    "pattern": INTEGER_STR_RE,
                    "description": "Integer encoded as string marking if group was deleted",
                },
            }
        ),
        "oneOf": [
            {"required": ["role", "path", "group"]},
            {"required": ["role", "path", "login"]},
        ],
        "additionalProperties": False,
    },
    iam_permissions=iam.NoOneApiIamPermission(),
)
@idm_response("/idm/remove-role/")
def remove_role_member(request):
    request["fired"] = _get_fired_param(request)

    role_path = get_validated_role_path(request["path"])

    try:
        role_node = get_node(role_path)
    except ChildNotFound:
        # IDM can come to revoke project roles after project was deleted, let's not break here
        logger.warning("IDM node %s was not found", role_path)
        return {}

    group_id = _get_group_param(request)
    if group_id is not None:
        member = staff.id_to_group(group_id, is_deleted=_get_deleted_param(request))
        logger.info("Removing group member %s (%s) from role %s", member, group_id, request["path"])
    else:
        member = request["login"]
        logger.info("Removing member %s from role %s", member, request["path"])

    role_node.remove_role_member(member)

    return {}
