import typing as tp

from walle import authorization
from walle.clients import abc, staff
from walle.errors import UnauthorizedError, BadRequestError
from walle.maintenance_plot.model import MaintenancePlotModel

TMPL_LINK_TO_SERVICE = "https://abc.yandex-team.ru/services/{}"


def check_if_issuer_authorized(issuer: str, abc_service_slug: str):
    login = authorization.get_issuer_login(issuer)

    if authorization.is_admin(login):
        return

    if login not in {item["person"]["login"] for item in abc.get_service_members(service_slug=abc_service_slug)}:
        msg_link_to_service = TMPL_LINK_TO_SERVICE.format(abc_service_slug)
        raise UnauthorizedError(
            "You must have any role in ABC service with slug {}. {}".format(abc_service_slug, msg_link_to_service)
        )


def check_if_all_staff_logins_exists(raw_logins: list[str]):
    validated_logins = staff.check_logins(raw_logins)
    unexisting_logins = set(raw_logins) - set(validated_logins)
    if unexisting_logins:
        raise BadRequestError("Please check that these logins exist on Staff: {}".format(", ".join(unexisting_logins)))


def check_if_people_in_abc_service_have_role_codes_and_role_scope_slugs(
    abc_service_slug: str, role_codes: tp.Optional[list[str]], role_scope_slugs: tp.Optional[list[str]]
):
    if role_codes:
        if (
            len(
                [
                    item["person"]["login"]
                    for item in abc.get_service_members(service_slug=abc_service_slug, role_codes=role_codes)
                ]
            )
            == 0
        ):
            raise BadRequestError("Please give us a valid role code some people have")
    if role_scope_slugs:
        if (
            len(
                [
                    item["person"]["login"]
                    for item in abc.get_service_members(
                        service_slug=abc_service_slug, role_scope_slugs=role_scope_slugs
                    )
                ]
            )
            == 0
        ):
            raise BadRequestError("Please give us a valid role scope slug some people have")


def check_if_scenarios_settings_in_plot_allowed_to_change(plot: MaintenancePlotModel):
    if plot.gc_enabled:
        raise BadRequestError(
            "It's automatically created maintenance plot, you can't change scenarios settings. Please create your own maintenance plot"
        )
