import http.client as httplib

import mongoengine

from walle import audit_log
from walle.authorization import iam
from walle.errors import ResourceConflictError
from walle.locks import MaintenancePlotInterruptableLock
from walle.maintenance_plot.crud import get_all_maintenance_plots, get_maintenance_plot
from walle.maintenance_plot.exceptions import MaintenancePlotAlreadyExistError
from walle.maintenance_plot.model import (
    MaintenancePlotModel,
    MaintenancePlotMetaInfo,
    MaintenancePlotCommonSettings,
    get_scenarios_settings_json_schema,
    MaintenancePlotScenarioSettings,
    MaintenancePlot,
)
from walle.projects import Project
from walle.util.api import api_handler, api_response, get_query_result
from walle.views.api.maintenance_plot_api.common import (
    check_if_issuer_authorized,
    check_if_all_staff_logins_exists,
    check_if_people_in_abc_service_have_role_codes_and_role_scope_slugs,
    check_if_scenarios_settings_in_plot_allowed_to_change,
)


@api_handler(
    "/maintenance-plots",
    "GET",
    with_fields=MaintenancePlotModel,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def list_all_maintenance_plots(query_args):
    """List all maintenance plots IDs and meta info."""
    fields = query_args.get("fields")
    plots = get_all_maintenance_plots(fields)
    return api_response([p.to_api_obj(fields) for p in plots])


@api_handler(
    "/maintenance-plots/<plot_id>",
    "GET",
    with_fields=MaintenancePlotModel,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def get_maintenance_plot_by_id(plot_id, query_args):
    """Return maintenance plot."""
    fields = query_args.get("fields")
    plot = get_maintenance_plot(plot_id, fields=fields)
    return api_response(plot.to_api_obj(fields))


@api_handler(
    "/maintenance-plots/<plot_id>",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def delete_maintenance_plot(issuer, plot_id, request, reason):
    """Delete specified maintenance plot."""
    with MaintenancePlotInterruptableLock(plot_id):
        plot = get_maintenance_plot(plot_id)
        plot_class = plot.as_dataclass()

        check_if_issuer_authorized(issuer, plot_class.meta_info.abc_service_slug)

        projects_with_maintenance_plot = Project.objects(maintenance_plot_id=plot_id).only("id", "maintenance_plot_id")
        if projects_with_maintenance_plot.count() > 0:
            project_ids = ", ".join([project.id for project in projects_with_maintenance_plot])
            raise ResourceConflictError("There are projects that use this maintenance plot: {}".format(project_ids))

        with audit_log.on_delete_maintenance_plot(issuer, plot_id, reason):
            plot.delete()

    return "", httplib.NO_CONTENT


@api_handler(
    "/maintenance-plots/<plot_id>/meta_info",
    "PUT",
    MaintenancePlotMetaInfo.get_json_schema(),
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def set_maintenance_plot_meta_info(issuer, plot_id, request, reason):
    plot_document = get_maintenance_plot(plot_id)
    plot_class = plot_document.as_dataclass()

    check_if_issuer_authorized(issuer, plot_class.meta_info.abc_service_slug)

    updated_meta_info = MaintenancePlotMetaInfo.from_dict(request)

    with audit_log.on_update_maintenance_plot(issuer, plot_document.id, request, reason=reason):
        with MaintenancePlotInterruptableLock(plot_document.id):
            plot_document.set_meta_info(updated_meta_info)
            return api_response(plot_document.to_api_obj(("id", "meta_info")))


@api_handler(
    "/maintenance-plots/<plot_id>/common_settings",
    "PUT",
    MaintenancePlotCommonSettings.get_json_schema(),
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def set_maintenance_plot_common_settings(issuer, plot_id, request, reason):
    plot_document = get_maintenance_plot(plot_id)
    plot_class = plot_document.as_dataclass()

    check_if_issuer_authorized(issuer, plot_class.meta_info.abc_service_slug)

    updated_common_settings = MaintenancePlotCommonSettings.from_dict(request)

    check_if_all_staff_logins_exists(updated_common_settings.maintenance_approvers.logins)
    check_if_people_in_abc_service_have_role_codes_and_role_scope_slugs(
        plot_class.meta_info.abc_service_slug,
        updated_common_settings.maintenance_approvers.abc_roles_codes,
        updated_common_settings.maintenance_approvers.abc_role_scope_slugs,
    )

    with audit_log.on_update_maintenance_plot(issuer, plot_document.id, request, reason=reason):
        with MaintenancePlotInterruptableLock(plot_document.id):
            plot_document.set_common_settings(updated_common_settings)
            return plot_document.to_api_obj(("id", "common_settings"))


@api_handler(
    "/maintenance-plots/<plot_id>/scenarios_settings",
    "PUT",
    get_scenarios_settings_json_schema(),
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def set_maintenance_plot_scenarios_settings(issuer, plot_id, request, reason):

    plot_document = get_maintenance_plot(plot_id)
    plot_class = plot_document.as_dataclass()

    check_if_issuer_authorized(issuer, plot_class.meta_info.abc_service_slug)
    check_if_scenarios_settings_in_plot_allowed_to_change(plot_document)

    updated_scenario_settings = []
    for settings_dict in request.get("scenarios_settings"):
        settings_obj = MaintenancePlotScenarioSettings.from_dict(settings_dict)
        updated_scenario_settings.append(settings_obj)

    with audit_log.on_update_maintenance_plot(issuer, plot_document.id, request, reason=reason):
        with MaintenancePlotInterruptableLock(plot_document.id):
            plot_document.set_scenario_settings(updated_scenario_settings)
            return plot_document.to_api_obj(("id", "scenarios_settings"))


@api_handler(
    "/maintenance-plots/<plot_id>/projects",
    "GET",
    with_paging={
        "cursor": {"type": "string", "minLength": 1, "description": "Event time to start the search from"},
        "cursor_only": True,
    },
    iam_permissions=iam.NoOneApiIamPermission(),
)
def get_projects_of_maintenance_plot_by_id(plot_id, query_args):
    return api_response(
        get_query_result(
            Project,
            dict(maintenance_plot_id=plot_id),
            Project.id,
            query_args,
            cursor_only=True,
        )
    )


@api_handler(
    "/maintenance-plots",
    "POST",
    MaintenancePlot.get_json_schema(),
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def create_new_maintenance_plot(issuer, request, reason):

    maintenance_plot_obj = MaintenancePlot.from_dict(request)
    maintenance_plot_document = MaintenancePlotModel.from_dataclass(maintenance_plot_obj)

    check_if_issuer_authorized(issuer, maintenance_plot_obj.meta_info.abc_service_slug)
    check_if_all_staff_logins_exists(maintenance_plot_obj.common_settings.maintenance_approvers.logins)
    check_if_people_in_abc_service_have_role_codes_and_role_scope_slugs(
        maintenance_plot_obj.meta_info.abc_service_slug,
        maintenance_plot_obj.common_settings.maintenance_approvers.abc_roles_codes,
        maintenance_plot_obj.common_settings.maintenance_approvers.abc_role_scope_slugs,
    )

    with audit_log.on_add_maintenance_plot(issuer, maintenance_plot_document.id, request, reason):
        try:
            maintenance_plot_document.save(force_insert=True)
        except mongoengine.NotUniqueError:
            raise MaintenancePlotAlreadyExistError

    return api_response(maintenance_plot_document.to_api_obj(), code=httplib.CREATED)
