import logging

import mongoengine

from walle.authorization import iam
from walle.errors import ResourceNotFoundError
from walle.operations_log.constants import OPERATION_CHOICES
from walle.operations_log.operations import OperationLog
from walle.util.api import api_handler, api_response, get_query_result, FilterQueryParser

log = logging.getLogger(__name__)


class OperationLogEntryNotFoundError(ResourceNotFoundError):
    def __init__(self):
        super().__init__("The specified audit log entry doesn't exist.")


@api_handler(
    "/operation-log",
    "GET",
    params={
        "id": {"type": "array", "items": {"type": "string", "minLength": 1, "description": "Filter by id"}},
        "audit_log_id": {
            "type": "array",
            "items": {"type": "string", "minLength": 1, "description": "Filter by audit_log id"},
        },
        "host_inv": {
            "type": "array",
            "items": {"type": "integer", "minimum": 0, "description": "Filter by inventory number"},
        },
        "host_name": {
            "type": "array",
            "items": {"type": "string", "minLength": 1, "description": "Filter by host name"},
        },
        "host_uuid": {
            "type": "array",
            "items": {"type": "string", "minLength": 1, "description": "Filter by host uuid"},
        },
        "project": {
            "type": "array",
            "items": {"type": "string", "minLength": 1, "description": "Filter by host's project"},
        },
        "scenario_id": {"type": "array", "items": {"type": "integer", "description": "Filter by scenario id"}},
        "type": {"type": "array", "items": {"enum": OPERATION_CHOICES, "description": "Filter by operation type"}},
        "start_time": {"type": "number", "minimum": 0, "description": "Time lower bound"},
        "end_time": {"type": "number", "minimum": 0, "description": "Time upper bound"},
        "reverse": {"type": "boolean", "description": "Log entries output order. Default is true"},
    },
    with_fields=OperationLog,
    with_paging={
        "cursor": {"type": "number", "minimum": 0, "description": "Event time to start the search from"},
        "cursor_only": True,
    },
    iam_permissions=iam.GetHostsApiIamPermission(
        projects_arg_name="project",
        hosts_arg_name="host_name",
        hosts_arg_inv="host_inv",
        hosts_arg_uuid="host_uuid",
    ),
)
def get_operation_log(query_args):
    """Returns operation log entries."""
    query_parser = FilterQueryParser(
        OperationLog,
        enum_fields=("id", "audit_log_id", "scenario_id", "type", "host_inv", "host_uuid", "project", "host_name"),
        or_fields=("host_name", "host_inv", "host_uuid"),
    )
    query = query_parser.parse_query_filter(query_args)

    time_db_field = OperationLog.time.db_field
    if "start_time" in query_args:
        query.setdefault(time_db_field, {})["$gte"] = query_args["start_time"]

    if "end_time" in query_args:
        query.setdefault(time_db_field, {})["$lt"] = query_args["end_time"]

    return api_response(
        get_query_result(
            OperationLog,
            query,
            OperationLog.time,
            query_args,
            cursor_only=True,
            reverse=query_args.get("reverse", True),
        )
    )


@api_handler(
    "/operation-log/<entry_id>",
    "GET",
    with_fields=OperationLog,
    iam_permissions=iam.LogEntryApiIamPermission("entry_id"),
)
def get_operation_log_entry(entry_id, query_args):
    """Returns the specified operation log entry."""
    fields = query_args.get("fields")

    try:
        oplog_entry = OperationLog.objects.only(*OperationLog.api_query_fields(fields)).get(id=entry_id)
    except mongoengine.DoesNotExist:
        raise OperationLogEntryNotFoundError

    return api_response(oplog_entry.to_api_obj(fields))
