"""Preorders"""  # Will be used as category name in API reference

import http.client
import logging

import mongoengine

import walle.projects
import walle.util.limits
import walle.util.notifications
from walle import audit_log, constants as walle_constants, preorders, restrictions, projects
from walle.authorization import iam
from walle.clients import bot, inventory, staff
from walle.errors import (
    BadRequestError,
    ResourceAlreadyExistsError,
    RequestValidationError,
    ResourceNotFoundError,
    ResourceConflictError,
)
from walle.locks import ProjectInterruptableLock
from walle.preorders import Preorder, PreorderNotFoundError
from walle.util.api import api_handler, api_response, get_simple_query_result

log = logging.getLogger(__name__)


@api_handler(
    "/preorders",
    "POST",
    {
        "type": "object",
        "properties": {
            "id": {"type": "integer", "minimum": 1, "description": "Preorder ID"},
            "project": {"type": "string", "description": "Project ID to add the hosts to"},
            "prepare": {"type": "boolean", "description": "Prepare taken hosts. Default is false"},
            "provisioner": {"enum": walle_constants.PROVISIONERS, "description": "Provisioning system"},
            "deploy_config": {"type": "string", "description": "Config to deploy the hosts with"},
            "restrictions": {
                "type": "array",
                "items": {"enum": restrictions.ALL},
                "description": "Restrictions to set for the hosts",
            },
        },
        "required": ["id", "project"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_sudo=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def add_preorder(issuer, query_args, request, reason):
    """Registers the specified preorder in Wall-E."""

    preorder = Preorder(
        id=request["id"],
        issuer=issuer,
        project=request["project"],
        prepare=request.get("prepare", False),
        processed=False,
    )

    try:
        preorders.check_id(preorder.id)
    except PreorderNotFoundError:
        pass
    else:
        raise ResourceAlreadyExistsError("#{} preorder is already registered in Wall-E.", preorder.id)

    try:
        project = walle.projects.get_by_id(preorder.project, fields=("id", "owners", "provisioner"))
    except walle.projects.ProjectNotFoundError as e:
        raise BadRequestError(str(e))

    project.authorize(issuer)

    try:
        preorder_info = bot.get_preorder_info(preorder.id)
    except bot.InvalidPreorderIdError:
        raise ResourceNotFoundError("Preorder #{} doesn't exist in BOT.", preorder.id)

    preorder.bot_project = preorder_info["bot_project_id"]

    try:
        preorder.owner = staff.check_login(preorder_info["owner"], allow_dismissed=True)
    except staff.InvalidLoginError as e:
        raise ResourceConflictError("Can't add the preorder to Wall-E: it has an invalid owner '{}'.", e.login)

    preorder.authorize(issuer, query_args.get("sudo", False))

    if preorder.prepare:
        if "provisioner" in request and "deploy_config" not in request:
            raise RequestValidationError("Provisioner must be specified with deploy config.")

        if "deploy_config" in request:
            if "provisioner" in request:
                preorder.provisioner = request["provisioner"]
            else:
                preorder.provisioner = project.provisioner

            preorder.deploy_config = request["deploy_config"]
            inventory.check_deploy_configuration(
                preorder.provisioner, preorder.deploy_config, projects.get_eine_box(preorder.project)
            )

        if "restrictions" in request:
            preorder.restrictions = restrictions.strip_restrictions(request["restrictions"])
    else:
        invalid_params = {"provisioner", "deploy_config", "restrictions"}.intersection(request.keys())
        if invalid_params:
            raise RequestValidationError(
                "'{}' parameter may be set only when hosts are assigned to preparing.", invalid_params.pop()
            )

    with ProjectInterruptableLock(preorder.project):
        try:
            walle.projects.check_id(preorder.project)
        except walle.projects.ProjectNotFoundError as e:
            raise BadRequestError(str(e))

        with audit_log.on_add_preorder(issuer, preorder.project, preorder.to_api_obj(Preorder.api_fields), reason):
            try:
                preorder.save(force_insert=True)
            except mongoengine.NotUniqueError:
                raise ResourceAlreadyExistsError("#{} preorder is already registered in Wall-E.", preorder.id)

    return api_response(preorder.to_api_obj(), code=http.client.CREATED)


@api_handler(
    "/preorders",
    "GET",
    with_fields=Preorder,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def get_preorders(query_args):
    """Returns all registered preorders."""

    return get_simple_query_result(Preorder, query_args)


@api_handler(
    "/preorders/<preorder_id>",
    "GET",
    with_fields=Preorder,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def get_preorder(preorder_id, query_args):
    """Returns the specified preorder."""

    preorder_id = _validate_id(preorder_id)
    fields = query_args.get("fields")
    preorder = preorders.get_by_id(preorder_id, fields=Preorder.api_query_fields(fields))
    return api_response(preorder.to_api_obj(fields))


@api_handler(
    "/preorders/<preorder_id>/restart",
    "POST",
    authenticate=True,
    with_sudo=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def restart_preorder_processing(issuer, preorder_id, query_args, request, reason):
    """Restarts processing of the specified preorder."""

    preorder = preorders.get_by_id(_validate_id(preorder_id))
    preorder.authorize(issuer, query_args.get("sudo", False))

    preorders.restart_processing(issuer, preorder, reason)
    return api_response(preorder.to_api_obj())


@api_handler(
    "/preorders/<preorder_id>",
    "DELETE",
    authenticate=True,
    with_sudo=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def delete_preorder(issuer, preorder_id, query_args, request, reason):
    """Deletes the specified preorder from Wall-E."""

    preorder = preorders.get_by_id(_validate_id(preorder_id))
    preorder.authorize(issuer, query_args.get("sudo", False))

    with audit_log.on_delete_preorder(issuer, preorder.project, preorder.to_api_obj(Preorder.api_fields), reason):
        removed_preorder = Preorder.objects(id=preorder.id).modify(remove=True)

    if removed_preorder is not None and not removed_preorder.processed and removed_preorder.audit_log_id is not None:
        audit_log.cancel_request(removed_preorder.audit_log_id, "The preorder has been removed from Wall-E.")

    return "", http.client.NO_CONTENT


def _validate_id(preorder_id):
    try:
        return int(preorder_id)
    except ValueError:
        raise PreorderNotFoundError()
