"""Some common methods for use in project api handlers."""
import re
from contextlib import contextmanager

from sepelib.core import config
from sepelib.core.exceptions import LogicalError
from walle import authorization
from walle.clients import staff, abc, bot
from walle.clients.network.racktables_client import get_owned_vlans
from walle.constants import (
    VLAN_SCHEME_MTN,
    MTN_NATIVE_VLAN,
    MTN_EXTRA_VLANS,
    VLAN_SCHEME_MTN_HOSTID,
    VLAN_SCHEME_MTN_WITHOUT_FASTBONE,
    MTN_IP_METHOD_MAC,
    MTN_IP_METHOD_HOSTNAME,
    EINE_PROFILES_WITH_DC_SUPPORT,
)
from walle.errors import (
    RequestValidationError,
    DNSDomainNotAllowedInCertificator,
    UserNotInOwningABCService,
    BadRequestError,
)
from walle.errors import UnauthorizedError
from walle.expert import automation_plot
from walle.locks import AutomationPlotInterruptableLock


def set_project_for_mtn_kwargs(hbf_project_id, ip_method, use_fastbone=True):
    if ip_method == MTN_IP_METHOD_MAC:
        if use_fastbone:
            # default one
            vlan_scheme = VLAN_SCHEME_MTN
        else:
            vlan_scheme = VLAN_SCHEME_MTN_WITHOUT_FASTBONE
    elif ip_method == MTN_IP_METHOD_HOSTNAME:
        vlan_scheme = VLAN_SCHEME_MTN_HOSTID
    else:
        raise LogicalError()  # need to add new IP methods here

    return {
        "hbf_project_id": hbf_project_id,
        "vlan_scheme": vlan_scheme,
        "native_vlan": MTN_NATIVE_VLAN,
        "extra_vlans": MTN_EXTRA_VLANS,
    }


def check_dns_domain_allowed_in_certificator(dns_domain):
    if dns_domain is None:
        raise RequestValidationError("Cannot enable certificator without DNS domain.")

    allowed_domains = config.get_value("certificator.allowed_dns_domains_re", [])
    for regex in allowed_domains:
        if re.search(regex, dns_domain):
            return
    raise DNSDomainNotAllowedInCertificator(dns_domain)


def authenticate_user_by_bot_project_id(user, bot_project_id):
    abc_service_id = bot.get_planner_id_by_bot_project_id(bot_project_id)
    abc_service_slug = abc.get_service_slug(abc_service_id)
    staff_group_to_belong = "@svc_{}".format(abc_service_slug.lower())
    if staff_group_to_belong not in staff.get_user_groups(user):
        raise UserNotInOwningABCService(user, abc_service_slug, bot_project_id)


def get_validated_profile(profile):
    if profile not in EINE_PROFILES_WITH_DC_SUPPORT:
        raise BadRequestError(
            "At this time only '{}' profile can be set as default host profile.",
            ", ".join(EINE_PROFILES_WITH_DC_SUPPORT),
        )
    return profile


def get_sorted_profile_tags(profile_tags):
    return sorted(set(profile_tags))


@contextmanager
def locked_automation_plot_id(automation_plot_id=None):
    if automation_plot_id is not None:
        with AutomationPlotInterruptableLock(automation_plot_id):
            plot = automation_plot.get_automation_plot(automation_plot_id, fields=())
            yield plot.id
    else:
        yield None


def check_vlan_authorization(issuer, vlans):
    if authorization.is_admin(issuer):
        return

    if not isinstance(vlans, list):
        vlans = [vlans]

    vlans = [int(vlan) for vlan in vlans]

    owned_vlans = get_owned_vlans(issuer)

    if set(vlans).issubset(owned_vlans):
        return

    error_msg = "You don't own some of requested vlans: {}".format(vlans)
    raise UnauthorizedError(error_msg)
