"""Projects"""  # Will be used as category name in API reference

import http.client
import logging
import typing as tp
from collections import OrderedDict

import mongoengine

from sepelib.core import config
from sepelib.core.exceptions import LogicalError
from walle import audit_log, restrictions, projects, boxes, authorization
from walle.authorization import iam, blackbox, has_iam
from walle.clients import staff, dns as dns_clients
from walle.clients.idm import IDMInternalError
from walle.constants import PROVISIONERS, MTN_IP_METHOD_MAC, NetworkTarget, HostType, HOST_TYPES_WITH_MANUAL_OPERATION
from walle.errors import BadRequestError, ResourceConflictError, ResourceAlreadyExistsError
from walle.expert.decisionmakers import get_decision_maker
from walle.idm.project_role_managers import get_projects_roles_members, ProjectRole
from walle.locks import ProjectInterruptableLock
from walle.models import DocumentPostprocessor
from walle.preorders import Preorder
from walle.project_builder import ProjectBuilder
from walle.projects import Project, ProjectNotFoundError, get_by_id, CmsSettings, CauthSettingsDocument
from walle.util.api import api_handler, api_response, validate_user_string, get_simple_query_result, production_only
from walle.util.deploy_config import DeployConfigPolicies
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.misc import fix_mongo_set_kwargs, replace_group_with_its_members
from walle.views.api.common import get_vlan_id_schema, get_eine_tags_schema, validated_tags
from walle.views.api.project_api import change_handlers
from walle.views.api.project_api.common import (
    set_project_for_mtn_kwargs,
    check_dns_domain_allowed_in_certificator,
    locked_automation_plot_id,
)
from walle.views.helpers.validators import (
    validated_hbf_project_id,
    validated_bot_project_id,
    check_cms_tvm_app_id_requirements,
    validated_host_shortname_template,
    check_cauth_settings,
)
from . import schemas

log = logging.getLogger(__name__)


def _validate_disabling_checks(project: Project, manually_disabled_checks: tp.Optional[list[str]]):
    if not manually_disabled_checks:
        return
    available_checks = set(get_decision_maker(project).all_available_checks())
    for manually_disabled_check in manually_disabled_checks:
        if manually_disabled_check not in available_checks:
            raise BadRequestError(f"Unknown check type for manually disabling '{manually_disabled_check}'")


@api_handler(
    "/projects",
    "POST",
    {
        "type": "object",
        "properties": dict(
            {
                "id": {
                    "type": "string",
                    "pattern": Project.id.regex.pattern,
                    "maxLength": Project.id.max_length,
                    "description": "A unique ID",
                },
                "type": {"enum": HostType.get_choices(), "description": "Type of hosts in the project"},
                "owners": schemas.get_owners_schema("Project owners"),
                "owned_vlans": {
                    "type": "array",
                    "items": get_vlan_id_schema(),
                    "description": "VLAN IDs owned by the project",
                },
                "notifications": {
                    "type": "object",
                    "properties": {"recipients": schemas.get_recipients_schema("Recipients by severity")},
                    "required": ["recipients"],
                    "additionalProperties": False,
                },
                "enable_healing_automation": {
                    "type": "boolean",
                    "description": "Enable automated healing for the project. Default is false",
                },
                "enable_dns_automation": {
                    "type": "boolean",
                    "description": "Enable DNS automation for the project. Default is false",
                },
                "manually_disabled_checks": {
                    "type": "array",
                    "items": {"type": "string"},
                    "description": "Checks disabled by man",
                },
                "provisioner": {"enum": PROVISIONERS, "description": "Provisioning system"},
                "profile": {"type": "string", "description": "Einstellung profile"},
                "profile_tags": get_eine_tags_schema(for_profile=True),
                "deploy_config": {"type": "string", "description": "Deploy config"},
                "deploy_config_policy": {
                    "enum": DeployConfigPolicies.get_all_names(),
                    "description": "Deploy config policy",
                },
                "deploy_tags": get_eine_tags_schema(for_profile=False),
                "deploy_network": {
                    "enum": NetworkTarget.DEPLOYABLE,
                    "description": "Deploy host either in service or in project VLAN's",
                },
                "automation_plot_id": {"type": "string", "description": "Automation Plot unique identifier"},
                "reboot_via_ssh": {"type": "boolean", "description": "Enable/disable rebooting via SSH"},
                "yc_iam_folder_id": {"type": "string", "description": "YC Folder ID"},
            },
            **schemas.get_general_properties_schema(),
        ),
        "required": ["id", "name", "provisioner", "deploy_config"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=[
        iam.CreateProjectApiIamPermission(),
        iam.BindProjectFolderApiIamPermission("yc_iam_folder_id"),
        iam.OptionalGetAutomationPlotApiIamPermission("automation_plot_id"),
    ],
)
@production_only
def add_project(issuer, request, reason):
    """Register a new project."""
    builder = ProjectBuilder(issuer)

    builder.set_fields(
        id=request["id"],
        owners=[],  # owners will be added later via IDM
        automation_limits=projects.get_default_project_automation_limits(),
        host_limits=projects.get_default_host_limits(),
        validate_bot_project_id=True,
    )

    builder.set_name(request["name"])
    builder.set_type(request.get("type"))
    builder.set_project_tags(request.get("tags"))
    builder.set_dns_domain(request.get("dns_domain"))
    builder.set_shortname_template(request.get("host_shortname_template"))

    builder.set_certificate_deploy(request.get("certificate_deploy"))
    builder.set_deploy_configuration(
        provisioner=request["provisioner"],
        deploy_config=request["deploy_config"],
        deploy_config_policy=request.get("deploy_config_policy"),
        deploy_tags=request.get("deploy_tags"),
        deploy_network=request.get("deploy_network"),
    )
    builder.set_profile(request.get("profile"))
    if authorization.has_iam():
        builder.set_fields(yc_iam_folder_id=request["yc_iam_folder_id"])
    builder.set_profile_tags(request.get("profile_tags"))
    builder.set_default_host_restrictions(request.get("default_host_restrictions"))

    builder.set_manually_disabled_checks(request.get("manually_disabled_checks"))
    builder.set_hbf_project_id(request.get("hbf_project_id"), request.get("ip_method"))

    owners = builder.calculate_owners(request.get("owners"))
    builder.set_notifications(request.get("notifications", {}).get("recipients"), owners)

    builder.set_cms_settings(request.get("cms"), request.get("cms_settings"))
    builder.set_bot_project_id(request.get("bot_project_id"), request.get("cms_settings"))

    reboot_via_ssh = builder.set_reboot_via_ssh(request.get("reboot_via_ssh"))
    builder.set_enable_dns_automation(request.get("enable_dns_automation"))

    builder.set_enable_healing_automation(request.get("enable_healing_automation"))
    builder.set_owned_vlans(request.get("owned_vlans"))
    builder.set_cauth_settings(
        request.get("cauth_flow_type"),
        request.get("cauth_trusted_sources"),
        request.get("cauth_key_sources"),
        request.get("cauth_secure_ca_list_url"),
        request.get("cauth_insecure_ca_list_url"),
        request.get("cauth_krl_url"),
        request.get("cauth_sudo_ca_list_url"),
    )
    builder.set_logical_datacenter(request.get("logical_datacenter"))
    automation_plot_id = builder.set_automation_plot_id(request.get("automation_plot_id"))

    add_project_ctx_manager = audit_log.on_add_project(issuer, request["id"], request, reason)
    locked_plot_ctx_manager = locked_automation_plot_id(automation_plot_id)
    with add_project_ctx_manager, locked_plot_ctx_manager:
        project = builder.build()

        _validate_disabling_checks(project, project.manually_disabled_checks)

        try:
            project.save(force_insert=True)
        except mongoengine.NotUniqueError:
            raise ResourceAlreadyExistsError("Project with the specified ID or name already exists.")

        if config.get_value("authorization.enabled"):
            # this param is set indirectly:
            # here we request special role in IDM
            # when it is approved, corresponding role tree node will set the param

            try:
                issuer_login = authorization.get_issuer_login(issuer)
                change_handlers.on_project_add(project, issuer_login, owners, enable_reboot_via_ssh=reboot_via_ssh)
            except IDMInternalError:
                project.delete()
                raise

    return api_response(project.to_api_obj(), code=http.client.CREATED)


@api_handler(
    "/projects/clone/<orig_project_id>",
    "POST",
    {
        "type": "object",
        "properties": {
            "id": {
                "type": "string",
                "pattern": Project.id.regex.pattern,
                "maxLength": Project.id.max_length,
                "description": "A unique ID",
            },
            "name": {
                "type": "string",
                "minLength": Project.name.min_length,
                "maxLength": Project.name.max_length,
                "description": "Name that will identify the project in UI",
            },
            "yc_iam_folder_id": {"type": "string", "description": "YC Folder ID"},
        },
        "required": ["id", "name"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=[
        iam.CreateProjectApiIamPermission(),
        iam.GetProjectApiIamPermission("orig_project_id"),
        iam.BindProjectFolderApiIamPermission("yc_iam_folder_id"),
    ],
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
@production_only
def clone_project(issuer, orig_project_id, allowed_host_types, request, reason):
    """Clone original project into a new one"""

    project = get_by_id(orig_project_id)
    project.authorize(issuer, allowed_host_types=allowed_host_types)

    project.id = request['id']
    project.name = request['name']
    if authorization.has_iam():
        if "yc_iam_folder_id" not in request:
            raise BadRequestError("Request should have 'yc_iam_folder_id' parameter")
        project.yc_iam_folder_id = request['yc_iam_folder_id']

    with audit_log.on_clone_project(issuer, project.id, orig_project_id, project.to_api_obj(), reason):
        try:
            project.save(force_insert=True)
        except mongoengine.NotUniqueError:
            raise ResourceAlreadyExistsError("Project with the specified ID or name already exists.")

        try:
            change_handlers.on_project_clone(project, orig_project_id)
        except IDMInternalError:
            project.delete()
            raise

    return api_response(project.to_api_obj(), code=http.client.CREATED)


class ProjectPostprocessor(DocumentPostprocessor):
    YC_BOXES_CALCULATED_FIELD = "boxes"

    def __init__(
        self,
        add_project_roles=False,
        add_reboot_via_ssh=False,
        add_owners=False,
        expand_groups=False,
        iam_public_handler=False,
    ):
        self.add_project_roles = add_project_roles
        self.add_reboot_via_ssh = add_reboot_via_ssh
        self.add_owners = add_owners
        self.expand_groups = expand_groups
        extra_db_fields, extra_fields = set(), set()

        if self.add_project_roles:
            extra_fields.add("roles")

        if self.add_reboot_via_ssh:
            extra_fields.add("reboot_via_ssh")

        if self.add_owners:
            extra_fields.add("owners")

        self._iam_public_handler = iam_public_handler
        super().__init__(extra_db_fields=extra_db_fields, extra_fields=extra_fields)

    def process(self, iterable, requested_fields):
        id_to_project = OrderedDict((project.id, project) for project in iterable)

        project_id_to_roles = {}
        if self.add_project_roles or self.add_reboot_via_ssh or self.add_owners:
            project_id_to_roles = get_projects_roles_members(id_to_project.keys())
            if self.expand_groups:
                group_to_members = self._extract_group_members(project_id_to_roles)
                project_id_to_roles = self._replace_groups_with_members(project_id_to_roles, group_to_members)

        objects = []
        for project in gevent_idle_iter(id_to_project.values()):
            extra_fields = {}

            if self.add_project_roles:
                extra_fields["roles"] = project_id_to_roles[project.id]

            if self.add_reboot_via_ssh:
                extra_fields["reboot_via_ssh"] = (
                    True if project_id_to_roles[project.id][ProjectRole.SSH_REBOOTER] else None
                )

            if self.add_owners:
                extra_fields["owners"] = project_id_to_roles[project.id][ProjectRole.OWNER]

            # NOTE(rocco66): can't do it in project.to_api_obj cause of circular deps
            if not requested_fields or Project.ALL_AVAILABLE_PROJECT_CHECKS_FIELD in requested_fields:
                decision_maker = get_decision_maker(project)
                extra_fields[Project.ALL_AVAILABLE_PROJECT_CHECKS_FIELD] = list(decision_maker.all_available_checks())

            project_response = project.to_api_obj(
                requested_fields,
                extra_fields=extra_fields,
                iam_public_handler=self._iam_public_handler,
            )
            project_response.update(self._calculated_api_obj(requested_fields, project))
            objects.append(project_response)

        return objects

    @staticmethod
    def _calculated_boxes(project):
        return {box_type: (boxes.get_box(project.id, box_type) or "") for box_type in boxes.BoxType}

    @staticmethod
    def _calculated_dns_zone(project):
        dns_zone_domain = dns_zone_link = ""
        dns_box_config = project.get_dns_box_config()
        if dns_box_config and dns_box_config.should_use_rurikk_dns() and project.yc_dns_zone_id:
            dns_client = tp.cast(dns_clients.RurikkDnsClient, project.get_dns())
            try:
                zone_info = dns_client.get_dns_zone_info()
            except Exception:
                log.exception("Can't get DNS zone info %s", str(dns_box_config))
            else:
                dns_zone_domain = zone_info.zone
                dns_zone_link = dns_box_config.get_dns_zone_link(zone_info)
        return {
            "domain": dns_zone_domain,
            "link": dns_zone_link,
        }

    def _calculated_api_obj(self, requested_fields, project):
        requested_fields = requested_fields or []
        res = {}
        if Project.YC_DNS_ZONE_CALCULATED_FIELD in requested_fields:
            res[Project.YC_DNS_ZONE_CALCULATED_FIELD] = self._calculated_dns_zone(project)
        if self.YC_BOXES_CALCULATED_FIELD in requested_fields:
            res[self.YC_BOXES_CALCULATED_FIELD] = self._calculated_boxes(project)
        return res

    def _extract_group_members(self, id_to_roles):
        groups = set()
        for _, _, members in self._iter_project_id_to_roles(id_to_roles):
            for member in members:
                if staff.is_group(member):
                    groups.add(member)

        group_to_members = staff.batch_get_groups_members(tuple(groups))
        return group_to_members

    def _replace_groups_with_members(self, id_to_roles, group_to_members):
        for project_id, role, members in self._iter_project_id_to_roles(id_to_roles):
            id_to_roles[project_id][role] = replace_group_with_its_members(members, group_to_members)
        return id_to_roles

    @staticmethod
    def _iter_project_id_to_roles(id_to_roles):
        for project_id, roles in id_to_roles.items():
            for role, members in roles.items():
                yield project_id, role, members


@api_handler(
    "/projects",
    "GET",
    params={
        "tags": {"type": "array", "items": {"type": "string"}, "description": "Projects' tags"},
        "expand_groups": {"type": "boolean", "description": "Expand groups in project roles members. Default is false"},
    },
    with_fields=Project,
)
def get_projects(query_args):
    """Returns all registered projects."""
    filter_kwargs = {}
    if "tags" in query_args:
        filter_kwargs["tags__all"] = validated_tags(query_args["tags"])

    requested_fields = query_args.get("fields", [])
    postprocessor = ProjectPostprocessor(
        add_project_roles="roles" in requested_fields,
        add_reboot_via_ssh="reboot_via_ssh" in requested_fields,
        add_owners="owners" in requested_fields,
        expand_groups=query_args.get("expand_groups", False),
        iam_public_handler=True,
    )
    return get_simple_query_result(Project, query_args, filter_kwargs, postprocessor=postprocessor)


@api_handler(
    "/projects/<project_id>",
    "GET",
    params={
        "expand_groups": {"type": "boolean", "description": "Expand groups in project roles members. Default is false"}
    },
    with_fields=Project,
    iam_permissions=iam.GetProjectApiIamPermission("project_id"),
)
def get_project(project_id, query_args):
    """Returns the specified project."""
    fields = query_args.get("fields")
    project = projects.get_by_id(project_id, fields=Project.api_query_fields(fields))

    requested_fields = query_args.get("fields", [])
    postprocessor = ProjectPostprocessor(
        add_project_roles="roles" in requested_fields,
        add_reboot_via_ssh="reboot_via_ssh" in requested_fields,
        add_owners="owners" in requested_fields,
        expand_groups=query_args.get("expand_groups", False),
    )
    project_api_obj = postprocessor.process_one(project, fields)
    return api_response(project_api_obj)


@api_handler(
    "/projects/<project_id>",
    ("PATCH", "POST"),
    {
        "type": "object",
        "properties": dict(
            {
                "validate_bot_project_id": {
                    "type": "boolean",
                    "description": "Check host bot_project_id when adding into project",
                },
            },
            **schemas.get_general_properties_schema(),
        ),
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=[
        iam.UpdateProjectApiIamPermission("project_id"),
        iam.OptionalGetAutomationPlotApiIamPermission("automation_plot_id"),
    ],
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
@production_only
def modify_project(issuer, project_id, allowed_host_types, request, reason):
    """Modify the specified project."""
    if not request:
        raise BadRequestError("At least one parameter must be specified for update.")

    project = projects.get_by_id(project_id)
    project.authorize(issuer)

    authorization_enabled = config.get_value("authorization.enabled")

    audit_log_payload = request.copy()
    update_kwargs = {}
    condition_kwargs = {"id": project_id}

    if "name" in request:
        update_kwargs.update(set__name=validate_user_string(request.pop("name"), "project name"))

    if "tags" in request:
        update_kwargs.update(set__tags=validated_tags(request.pop("tags")))

    if "hbf_project_id" in request:
        hbf_project_id = validated_hbf_project_id(request.pop("hbf_project_id"))
        ip_method = request.pop("ip_method", MTN_IP_METHOD_MAC)
        use_fastbone = request.get("use_fastbone", True)
        update_kwargs.update(set_project_for_mtn_kwargs(hbf_project_id, ip_method, use_fastbone=use_fastbone))

    if "ip_method" in request:
        # "ip_method" is here, but "hbf_project_id" is not.
        update_kwargs.update(set_project_for_mtn_kwargs(project.hbf_project_id, request.pop("ip_method")))

    if "bot_project_id" in request and not has_iam():
        blackbox.authorize(
            issuer,
            authorize_admins=True,
            error_message="BOT project id modification has been disabled for non-admins because of "
            "potential unmeasurable harm caused by it",
        )

        update_kwargs.update(set__bot_project_id=validated_bot_project_id(request.pop("bot_project_id")))

    if "validate_bot_project_id" in request:
        if not request["validate_bot_project_id"] and not has_iam():
            blackbox.authorize(
                issuer,
                authorize_admins=True,
                error_message="Validate BOT project id modification to False "
                "has been disabled for non-admins because of "
                "potential unmeasurable harm caused by it",
            )
        update_kwargs.update(set__validate_bot_project_id=request.pop("validate_bot_project_id"))

    if "certificate_deploy" in request:
        enable = request.pop("certificate_deploy")

        if enable:
            check_dns_domain_allowed_in_certificator(project.dns_domain)
            # check if dns domain has changed during exec of this function
            condition_kwargs["dns_domain"] = project.dns_domain

        update_kwargs.update(set__certificate_deploy=enable)

    if "cms" in request:
        cms_settings = CmsSettings.from_request(request.pop("cms"))
        all_cms_settings = []

        if cms_settings != CmsSettings.from_project(project):

            update_kwargs.update(set__cms=cms_settings.url)

            if cms_settings.url == projects.DEFAULT_CMS_NAME:
                update_kwargs.update(
                    set__cms_max_busy_hosts=cms_settings.max_busy_hosts,
                    unset__cms_api_version=True,
                    unset__cms_tvm_app_id=True,
                )
                all_cms_settings.append(cms_settings.to_default_cms_kwargs())
            else:
                if authorization_enabled and not authorization.is_admin(authorization.get_issuer_login(issuer)):
                    check_cms_tvm_app_id_requirements(cms_settings, project.bot_project_id)

                update_kwargs.update(
                    set__cms_api_version=cms_settings.api_version,
                    set__cms_tvm_app_id=cms_settings.tvm_app_id,
                    unset__cms_max_busy_hosts=True,
                )
                all_cms_settings.append(cms_settings.to_user_cms_api_kwargs())

            update_kwargs.update(set__cms_settings=all_cms_settings)

    if "default_host_restrictions" in request:
        host_restrictions = restrictions.strip_restrictions(
            request.pop("default_host_restrictions"), strip_to_none=True
        )

        update_kwargs.update(set__default_host_restrictions=host_restrictions)

    if "manually_disabled_checks" in request:
        manually_disabled_checks = request.pop("manually_disabled_checks")
        _validate_disabling_checks(project, manually_disabled_checks)
        update_kwargs.update(set__manually_disabled_checks=manually_disabled_checks)

    if "host_shortname_template" in request:
        if project.dns_domain:
            template = validated_host_shortname_template(request.pop("host_shortname_template"))
            update_kwargs.update(set__host_shortname_template=template)
            condition_kwargs["dns_domain"] = project.dns_domain
        else:
            raise BadRequestError("Can not use custom host shortname template in project without dns settings.")

    if "logical_datacenter" in request:
        update_kwargs.update(set__logical_datacenter=request.pop("logical_datacenter"))

    cauth_settings = project.cauth_settings or CauthSettingsDocument()
    cauth_settings_changed = False

    if "cauth_flow_type" in request or "cauth_trusted_sources" in request:
        flow_type, trusted_sources = check_cauth_settings(
            request.pop("cauth_flow_type", cauth_settings.flow_type),
            request.pop("cauth_trusted_sources", cauth_settings.trusted_sources),
        )
        cauth_settings_changed = True
        cauth_settings.flow_type = flow_type
        cauth_settings.trusted_sources = trusted_sources

    if "cauth_key_sources" in request:
        cauth_settings_changed = True
        cauth_settings.key_sources = request.pop("cauth_key_sources")

    if "cauth_secure_ca_list_url" in request:
        cauth_settings_changed = True
        cauth_settings.secure_ca_list_url = request.pop("cauth_secure_ca_list_url")

    if "cauth_insecure_ca_list_url" in request:
        cauth_settings_changed = True
        cauth_settings.insecure_ca_list_url = request.pop("cauth_insecure_ca_list_url")

    if "cauth_krl_url" in request:
        cauth_settings_changed = True
        cauth_settings.krl_url = request.pop("cauth_krl_url")

    if "cauth_sudo_ca_list_url" in request:
        cauth_settings_changed = True
        cauth_settings.sudo_ca_list_url = request.pop("cauth_sudo_ca_list_url")

    if cauth_settings_changed:
        update_kwargs.update(set__cauth_settings=cauth_settings)

    if request:
        raise LogicalError()  # unknown fields, wow

    with audit_log.on_update_project(issuer, project_id, audit_log_payload, reason):
        try:
            updated_project = Project.objects(**condition_kwargs).modify(
                new=True, **fix_mongo_set_kwargs(**update_kwargs)
            )
        except mongoengine.NotUniqueError:
            raise ResourceAlreadyExistsError("A project with this name already exists.")

        if updated_project is None:
            raise ProjectNotFoundError()

        return api_response(updated_project.to_api_obj())


@api_handler(
    "/projects/<project_id>",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.DeleteProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
@production_only
def delete_project(issuer, project_id, allowed_host_types, request, reason):
    """Deletes the specified project."""

    from walle.hosts import Host

    project = projects.get_by_id(project_id)
    project.authorize(issuer, allowed_host_types=allowed_host_types)

    with audit_log.on_delete_project(issuer, project_id, reason):
        with ProjectInterruptableLock(project_id):
            host_count = Host.objects(project=project_id).count()
            if host_count:
                raise ResourceConflictError("Unable to delete the project: it has {} hosts.", host_count)

            preorders_count = Preorder.objects(project=project_id, processed=False).count()
            if preorders_count:
                raise ResourceConflictError(
                    "Unable to delete the project: it has {} preorders that are in process.", preorders_count
                )

            change_handlers.on_project_delete(project)

            project.delete()

        return "", http.client.NO_CONTENT
