from datetime import datetime

from sepelib.core.constants import DAY_SECONDS, WEEK_SECONDS, MINUTE_SECONDS
from walle import projects, audit_log
from walle.authorization import iam
from walle.constants import HOST_TYPES_WITH_MANUAL_OPERATION
from walle.errors import ResourceConflictError, BadRequestError
from walle.host_fsm.handbrake import unset_fsm_handbrake
from walle.models import timestamp
from walle.projects import FsmHandbrake
from walle.util.api import api_handler, api_response

# even this is insane, but we can image people just pushing on any date in the calendar without thinking
_MAX_FSM_HANDBRAKE_TIMEOUT = WEEK_SECONDS
_MIN_FSM_HANDBRAKE_TIMEOUT = 10 * MINUTE_SECONDS  # just sanity check
_DEFAULT_FSM_HANDBRAKE_TIMEOUT = DAY_SECONDS


@api_handler(
    "/projects/<project_id>/fsm-handbrake",
    ("PUT", "POST"),
    {
        "type": "object",
        "properties": {
            "timeout_time": {
                "type": "integer",
                "description": "Time until which the emergency brake will be active for the project. "
                "When the time comes, the emergency brake will be automatically turned off by wall-e. "
                "It's guaranteed that status will be switched not earlier than the specified time. "
                "This is an alternative way of setting timeout which has a priority over the timeout parameter. "
                "Default is enable for one day",
            },
            "timeout": {
                "type": "integer",
                "minimum": _MIN_FSM_HANDBRAKE_TIMEOUT,
                "maximum": _MAX_FSM_HANDBRAKE_TIMEOUT,
                "description": "Timeout for which the emergency brake will be active for the project. "
                "This is an alternative way of setting timeout time. "
                "The timeout_time parameter has a priority. "
                "Default is enable for one day",
            },
            "ticket_key": {"type": "string", "description": "Ticket for project's emergency."},
        },
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def enable_emergency_brake(issuer, project_id, allowed_host_types, request, reason):
    """Enable emergency brake for the project."""
    project = projects.get_authorized(issuer, project_id, ["fsm_handbrake"], allowed_host_types=allowed_host_types)

    timeout_time = _get_timeout_time(request)
    ticket_key = request.get("ticket_key", None)

    if ticket_key is None and project.fsm_handbrake:
        ticket_key = project.fsm_handbrake.ticket_key

    if project.fsm_handbrake:
        success = _extend_fsm_handbrake(issuer, project, timeout_time, ticket_key, reason)
    else:
        success = _set_fsm_handbrake(issuer, project, timeout_time, ticket_key, reason)

    if not success:
        raise ResourceConflictError("Project state changed during the operation.")

    return api_response(project.to_api_obj({"id", "fsm_handbrake"}))


@api_handler(
    "/projects/<project_id>/fsm-handbrake",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def disable_emergency_brake(issuer, project_id, allowed_host_types, request, reason):
    """Disable emergency brake for the project."""
    project = projects.get_authorized(issuer, project_id, ["fsm_handbrake"], allowed_host_types=allowed_host_types)
    if not project.fsm_handbrake:
        return api_response(project.to_api_obj())

    if not unset_fsm_handbrake(issuer, project, reason):
        raise ResourceConflictError("Project state changed during the operation.")

    return api_response(project.to_api_obj())


def _set_fsm_handbrake(issuer, project, timeout_time, ticket_key, reason):
    audit_log_data = {
        "timeout_time": _fmt_timestamp(timeout_time),
        "ticket_key": ticket_key,
    }

    with audit_log.on_fsm_handbrake(issuer, project.id, reason=reason, **audit_log_data) as audit_log_entry:
        fsm_handbrake = FsmHandbrake(
            issuer=issuer,
            timestamp=timestamp(),
            timeout_time=timeout_time,
            reason=reason,
            ticket_key=ticket_key,
            audit_log_id=audit_log_entry.id,
        )
        return project.modify({"fsm_handbrake__exists": False}, fsm_handbrake=fsm_handbrake)


def _extend_fsm_handbrake(issuer, project, timeout_time, ticket_key, reason):
    original_audit_log_id = project.fsm_handbrake.audit_log_id
    audit_log_data = {
        "timeout_time": _fmt_timestamp(timeout_time),
        "ticket_key": ticket_key,
        "audit_log_id": original_audit_log_id,
    }

    with audit_log.on_extend_fsm_handbrake(issuer, project.id, reason=reason, **audit_log_data):
        return project.modify(
            {"fsm_handbrake__audit_log_id": original_audit_log_id},
            fsm_handbrake__timeout_time=timeout_time,
            fsm_handbrake__ticket_key=project.fsm_handbrake.ticket_key or ticket_key,
            fsm_handbrake__reason=project.fsm_handbrake.reason or reason,  # keep original reason if exists
        )


def _fmt_timestamp(time_timestamp):
    return datetime.fromtimestamp(time_timestamp).isoformat()


def _get_timeout_time(request):
    if "timeout_time" in request:
        if not _validate_timeout(request["timeout_time"] - timestamp()):
            raise BadRequestError(_insane_timeout_message())

        return request["timeout_time"]

    elif "timeout" in request:
        if not _validate_timeout(request["timeout"]):
            raise BadRequestError(_insane_timeout_message())

        return timestamp() + request["timeout"]

    else:
        return timestamp() + _DEFAULT_FSM_HANDBRAKE_TIMEOUT


def _validate_timeout(timeout):
    return _MIN_FSM_HANDBRAKE_TIMEOUT <= timeout <= _MAX_FSM_HANDBRAKE_TIMEOUT


def _insane_timeout_message():
    return "Timeout should be between {} and {} seconds.".format(_MIN_FSM_HANDBRAKE_TIMEOUT, _MAX_FSM_HANDBRAKE_TIMEOUT)
