"""Some common for all project views methods and schemas."""

from walle import projects, restrictions
from walle.clients.cauth import CauthFlowType, CauthSource, CAuthKeySources
from walle.clients.cms import CmsApiVersion
from walle.constants import MTN_IP_METHOD_MAC, MTN_IP_METHOD_HOSTNAME, LOGICAL_DATACENTERS
from walle.projects import Project
from walle.util import notifications
from walle.util.misc import concat_dicts
from walle.views.api.common import get_action_limits_schema

HEXADECIMAL = r"^(0x)?[0-9A-Fa-f]+$"

OLD_AUTOMATION_LIMIT_NAMES = {
    "max_rebooted_hosts",
    "max_profiled_hosts",
    "max_redeployed_hosts",
    "max_memory_changes",
    "max_disk_changes",
    "max_link_repairs",
    "max_cpu_repairs",
    "max_reboots_repairs",
}


def get_owners_schema(description):
    return {
        "type": "array",
        "items": {"type": "string", "pattern": Project.owners.field.regex.pattern},
        "description": description,
    }


def get_recipients_schema(description):
    return {
        "type": "object",
        "properties": {
            severity: {
                "type": "array",
                "items": {
                    "type": "string",
                    # See login restriction here - https://wiki.yandex-team.ru/passport/corporate
                    # This is a very simple validation, but it brings us a basic security and fully works for us now.
                    "pattern": r"^[a-z0-9_][a-z0-9_-]{,38}[a-z0-9_]@yandex-team\.ru$",
                },
            }
            for severity in notifications.SEVERITIES
        },
        "description": description,
        "additionalProperties": False,
    }


def get_general_properties_schema():
    return {
        "name": {
            "type": "string",
            "minLength": Project.name.min_length,
            "maxLength": Project.name.max_length,
            "description": "Name that will identify the project in UI",
        },
        "tags": get_tags_schema(),
        "cms": {
            "type": "object",
            "oneOf": [
                {
                    "type": "object",
                    "properties": {
                        "url": {
                            "type": "string",
                            "enum": [projects.DEFAULT_CMS_NAME],
                            "description": "CMS type",
                        },
                        "max_busy_hosts": {
                            "type": "integer",
                            "minimum": 1,
                            "description": "Maximum number of hosts that default CMS will allow to "
                            "process concurrently",
                        },
                    },
                    "required": ["url", "max_busy_hosts"],
                    "additionalProperties": False,
                },
                {
                    "type": "object",
                    "properties": {
                        "url": {
                            "type": "string",
                            "pattern": projects.CMS_RE_CUSTOM_URL,
                            "description": "Custom CMS URL",
                        },
                        "api_version": {
                            "type": "string",
                            "enum": CmsApiVersion.ALL_CMS_API,
                            "description": "Supported version of CMS API protocol",
                        },
                        "tvm_app_id": {
                            "type": "integer",
                            "description": "TVM application id of CMS",
                        },
                    },
                    "required": ["url", "api_version"],
                    "additionalProperties": False,
                },
            ],
        },
        "cms_settings": {
            "type": "array",
            "items": {
                "type": "object",
                "oneOf": [
                    {
                        "type": "object",
                        "properties": {
                            "url": {
                                "type": "string",
                                "enum": [projects.DEFAULT_CMS_NAME],
                                "description": "CMS type",
                            },
                            "max_busy_hosts": {
                                "type": "integer",
                                "minimum": 1,
                                "description": "Maximum number of hosts that default CMS will allow to "
                                "process concurrently",
                            },
                        },
                        "required": ["url", "max_busy_hosts"],
                        "additionalProperties": False,
                    },
                    {
                        "type": "object",
                        "properties": {
                            "url": {
                                "type": "string",
                                "pattern": projects.CMS_RE_CUSTOM_URL,
                                "description": "Custom CMS URL",
                            },
                            "api_version": {
                                "type": "string",
                                "enum": CmsApiVersion.ALL_CMS_API,
                                "description": "Supported version of CMS API protocol",
                            },
                            "tvm_app_id": {
                                "type": "integer",
                                "description": "TVM application id of CMS",
                            },
                            "temporary_unreachable_enabled": {
                                "type": "boolean",
                                "description": "Have implemented temporary-unreachable. Default is false",
                            },
                        },
                        "required": ["url", "api_version"],
                        "additionalProperties": False,
                    },
                ],
            },
            "description": "A list of cms settings for project",
        },
        "manually_disabled_checks": {
            "type": "array",
            "items": {"type": "string"},
            "description": "Mannually disabled checks",
        },
        "default_host_restrictions": {
            "type": "array",
            "items": {"enum": restrictions.ALL},
            "description": "Default restrictions for new hosts",
        },
        "dns_domain": {
            "type": "string",
            "pattern": projects.DNS_DOMAIN_RE,
            "description": "Domain for DNS auto-configuration",
        },
        "host_shortname_template": {
            "type": "string",
            "description": "Template for short names for the project's hosts. "
            "Python format string with mandatory {index} and "
            "optional {location} and {bucket} placeholders.",
        },
        "hbf_project_id": {
            "type": "string",
            "pattern": HEXADECIMAL,
            "description": "Hexadecimal HBF project id for HBF-enabled projects",
        },
        "ip_method": {
            "type": "string",
            "enum": [MTN_IP_METHOD_MAC, MTN_IP_METHOD_HOSTNAME],
            "description": "Method for generating host-id part of IP-address for MTN. "
            "Default is {}".format(MTN_IP_METHOD_MAC),
        },
        "use_fastbone": {"type": "boolean", "description": "Use fastbone. Default is true"},
        "bot_project_id": {"type": "integer", "minimum": -1, "description": "BOT/OEBS project id"},
        "certificate_deploy": {"type": "boolean", "description": "Enable certificate deploy. Default is false"},
        "logical_datacenter": {
            "type": "string",
            "enum": LOGICAL_DATACENTERS,
            "description": "Apply logical datacenter for project hosts to modify generating short names for location",
        },
        "cauth_flow_type": {"enum": CauthFlowType.ALL, "description": "CAuth trusted sources management type"},
        "cauth_trusted_sources": {
            "type": "array",
            "items": {"enum": CauthSource.ALL},
            "description": "CAuth trusted sources list",
        },
        "cauth_key_sources": {
            "type": "array",
            "items": {"enum": [source.value for source in CAuthKeySources]},
            "description": "CAuth sources for keys",
        },
        "cauth_secure_ca_list_url": {"type": "string", "description": "Custom URL for secure CA list"},
        "cauth_insecure_ca_list_url": {"type": "string", "description": "Custom URL for insecure CA list"},
        "cauth_krl_url": {"type": "string", "description": "Custom URL for KRL"},
        "cauth_sudo_ca_list_url": {"type": "string", "description": "Custom URL for sudo CA list"},
    }


def get_tags_schema():
    return {
        "type": "array",
        "description": "Project tags",
        "items": {
            "type": "string",
            "pattern": r"^#?{}".format(Project.tags.field.regex.pattern),
            "minLength": Project.tags.field.min_length,
            "maxLength": Project.tags.field.max_length,
        },
    }


def _deprecated_limit_schema():
    return {
        "type": "array",
        "items": {"type": "object"},
        "description": "DEPRECATED, will not be accepted",
    }


def get_automation_limits_schema():
    return {
        "type": "object",
        "properties": concat_dicts(
            {name: get_action_limits_schema() for name in projects.AUTOMATION_LIMIT_NAMES},
            {name: _deprecated_limit_schema() for name in OLD_AUTOMATION_LIMIT_NAMES},
        ),
        "description": ""
        "Automation limits configuration. "
        "If the specified number of hosts got the specified failure during the specified amount of time "
        "then all automation will be turned off until administrator turn it on again.",
        "additionalProperties": get_action_limits_schema(),
    }


def get_host_limits_schema():
    return {
        "type": "object",
        "properties": {name: get_action_limits_schema() for name in projects.HOST_LIMITS_NAMES_WITH_DEFAULT_VALUES},
        "description": ""
        "Host limits configuration. "
        "If the limit for a specific action for a host is exhausted, it will be skipped for that host.",
        "additionalProperties": False,
    }


def _credit_item_schema(name=None):
    if name:
        description = "Credit value for '{}' automation limit".format(name)
    else:
        description = "Credit value for other automation limits"

    return {"type": "integer", "minimum": 0, "description": description}


def get_automation_credit_schema(automation=None, strict=False):
    if automation is None:
        names = projects.AUTOMATION_LIMIT_NAMES
        automation_label = "automated actions"
    else:
        names = automation.get_limit_names()
        automation_label = automation.get_automation_label()

    return {
        "type": "object",
        "properties": dict(
            {
                "time": {"type": "integer", "minimum": 0, "description": "Credit time"},
            },
            **{name: _credit_item_schema(name) for name in names}
        ),
        "description": "Credit for {} to give to the project. "
        "Automated actions spend credit when it exists and not expired. "
        "Automated actions started at the expense of a credit aren't checked against automation limits.".format(
            automation_label
        ),
        "required": ["time"],
        "additionalProperties": False if strict else _credit_item_schema(name=None),
    }


def get_cms_schema():
    return {
        "type": "object",
        "oneOf": [
            {
                "type": "object",
                "properties": {
                    "url": {
                        "type": "string",
                        "enum": [projects.DEFAULT_CMS_NAME],
                        "description": "CMS type",
                    },
                    "max_busy_hosts": {
                        "type": "integer",
                        "minimum": 1,
                        "description": "Maximum number of hosts that default CMS will allow to process concurrently",
                    },
                },
                "required": ["url", "max_busy_hosts"],
                "additionalProperties": False,
            },
            {
                "type": "object",
                "properties": {
                    "url": {
                        "type": "string",
                        "pattern": projects.CMS_RE_CUSTOM_URL,
                        "description": "Custom CMS URL",
                    },
                    "api_version": {
                        "type": "string",
                        "enum": CmsApiVersion.ALL_CMS_API,
                        "description": "Supported version of CMS API protocol",
                    },
                    "tvm_app_id": {
                        "type": "integer",
                        "description": "TVM application id of CMS",
                    },
                    "temporary_unreachable_enabled": {
                        "type": "boolean",
                        "description": "Shows if 'temporary-unreachable' is implemented, default is false",
                    },
                },
                "required": ["url", "api_version"],
                "additionalProperties": False,
            },
        ],
    }
