"""Projects"""  # Will be used as category name in API reference

import http.client as httplib

import flask

import walle.projects
import walle.util.limits
from sepelib.core import config
from sepelib.core.exceptions import Error
from walle import projects, util, audit_log, authorization
from walle.authorization import iam
from walle.clients import abc, bot, idm
from walle.clients import inventory, staff
from walle.constants import (
    PROVISIONERS,
    NetworkTarget,
    VLAN_SCHEMES,
    MTN_VLAN_SCHEMES,
    MTN_IP_METHOD_MAC,
    MTN_IP_METHOD_HOSTNAME,
    PROVISIONER_EINE,
    ROBOT_WALLE_OWNER,
    HostType,
    HOST_TYPES_WITH_MANUAL_OPERATION,
)
from walle.errors import BadRequestError, ResourceConflictError, UnauthorizedError
from walle.expert.automation import PROJECT_HEALING_AUTOMATION, PROJECT_DNS_AUTOMATION
from walle.idm import project_push
from walle.idm.project_role_managers import ProjectRole
from walle.locks import ProjectInterruptableLock, MaintenancePlotInterruptableLock
from walle.maintenance_plot.crud import get_maintenance_plot
from walle.projects import Project, ProjectNotFoundError, Notifications, get_by_id, RepairRequestSeverity, CmsSettings
from walle.util.api import api_handler, api_response, admin_request, production_only
from walle.util.deploy_config import DeployConfigPolicies
from walle.util.misc import drop_none, fix_mongo_set_kwargs, filter_dict_keys, get_existing_tiers
from walle.views.api.common import (
    get_eine_tags_schema,
    API_ACTION_SET,
    get_api_action_methods,
    get_api_action,
    API_ACTION_REMOVE,
    get_api_action_update_query,
    get_vlan_id_schema,
    API_ACTION_ADD,
    validated_tags,
)
from walle.views.api.project_api import change_handlers
from walle.views.api.project_api.common import (
    set_project_for_mtn_kwargs,
    check_dns_domain_allowed_in_certificator,
    get_validated_profile,
    get_sorted_profile_tags,
    locked_automation_plot_id,
    check_vlan_authorization,
)
from walle.views.helpers.validators import (
    validated_hbf_project_id,
    validated_bot_project_id,
    check_cms_tvm_app_id_requirements,
    validated_host_shortname_template,
)
from . import schemas


@api_handler(
    "/projects/<project_id>/host-profiling-config",
    "PUT",
    {
        "type": "object",
        "properties": {
            "profile": {"type": "string", "description": "Einstellung profile"},
            "profile_tags": get_eine_tags_schema(for_profile=True),
        },
        "required": ["profile", "profile_tags"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def host_profiling_configuration(issuer, project_id, allowed_host_types, request, reason):
    """Modify host profiling configuration."""

    profile = get_validated_profile(request["profile"])
    profile_tags = get_sorted_profile_tags(request["profile_tags"])

    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    with audit_log.on_update_project(
        issuer,
        project_id,
        {"profile": {API_ACTION_SET: profile}, "profile_tags": {API_ACTION_SET: profile_tags}},
        reason,
    ):
        updated = Project.objects(id=project_id).update(
            set__profile=profile, set__profile_tags=profile_tags, multi=False
        )

        if not updated:
            raise ProjectNotFoundError()

    return api_response({"profile": profile, "profile_tags": profile_tags})


@api_handler(
    "/projects/<project_id>/host-provisioner-config",
    "PUT",
    {
        "type": "object",
        "properties": {
            "provisioner": {"enum": PROVISIONERS, "description": "Provisioning system"},
            "deploy_config": {"type": "string", "description": "Deploy config"},
            "deploy_config_policy": {
                "enum": DeployConfigPolicies.get_all_names(),
                "description": "Deploy config policy",
            },
            "deploy_tags": get_eine_tags_schema(for_profile=False),
            "deploy_network": {
                "enum": NetworkTarget.DEPLOYABLE,
                "description": "Deploy host either in service or in project VLAN's",
            },
        },
        "anyOf": [
            {"required": ["provisioner", "deploy_config"]},
            {"required": ["deploy_config"]},
        ],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def host_provisioner_config(issuer, project_id, allowed_host_types, request, reason):
    """Modify host provisioner configuration."""

    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    selected_fields = ["provisioner", "deploy_config", "deploy_config_policy", "deploy_tags", "deploy_network"]

    project = projects.get_by_id(project_id, fields=selected_fields)
    provisioner = request.get("provisioner", project.provisioner)
    deploy_config = request.get("deploy_config", project.deploy_config)
    deploy_config_policy = request.get("deploy_config_policy", project.deploy_config_policy)
    deploy_tags = (
        request.get("deploy_tags", project.deploy_tags if provisioner == PROVISIONER_EINE else None) or None
    )  # turn [] tags to None
    deploy_network = request.get("deploy_network", project.deploy_network)

    inventory.check_deploy_configuration(
        provisioner,
        deploy_config,
        projects.get_eine_box(project_id),
        deploy_tags,
        project.certificate_deploy,
        deploy_network,
        deploy_config_policy,
    )

    with audit_log.on_update_project(
        issuer,
        project_id,
        {
            "provisioner": {API_ACTION_SET: provisioner},
            "deploy_config": {API_ACTION_SET: deploy_config},
            "deploy_config_policy": {API_ACTION_SET: deploy_config_policy},
            "deploy_tags": {API_ACTION_SET: deploy_tags},
            "deploy_network": {API_ACTION_SET: deploy_network},
        },
        reason,
    ):
        query = dict(
            id=project_id,
            provisioner=project.provisioner,
            deploy_config=project.deploy_config,
            deploy_config_policy=project.deploy_config_policy,
            deploy_tags=project.deploy_tags,
            deploy_network=project.deploy_network,
        )
        update = {}
        if deploy_config != project.deploy_config:
            update["set__deploy_config"] = deploy_config
        if deploy_config_policy != project.deploy_config_policy:
            update["set__deploy_config_policy"] = deploy_config_policy
        if provisioner != project.provisioner:
            update["set__provisioner"] = provisioner
        if deploy_tags != project.deploy_tags:
            update["set__deploy_tags"] = deploy_tags
        if deploy_network != project.deploy_network:
            update["set__deploy_network"] = deploy_network

        if not update:
            return api_response(
                drop_none(
                    {
                        "provisioner": provisioner,
                        "deploy_config": deploy_config,
                        "deploy_config_policy": deploy_config_policy,
                        "deploy_tags": deploy_tags,
                        "deploy_network": deploy_network,
                    }
                )
            )

        updated_project = Project.objects(**query).only(*selected_fields).modify(**fix_mongo_set_kwargs(**update))
        if updated_project is None:
            raise ResourceConflictError("The project {} has changed its state during modification.", project_id)
        return api_response(
            drop_none(
                {
                    "provisioner": updated_project.provisioner,
                    "deploy_config": updated_project.deploy_config,
                    "deploy_config_policy": updated_project.deploy_config_policy,
                    "deploy_tags": updated_project.deploy_tags,
                    "deploy_network": updated_project.deploy_network,
                }
            )
        )


@api_handler(
    "/projects/<project_id>/owners",
    get_api_action_methods(),
    {
        "type": "object",
        "properties": {"owners": schemas.get_owners_schema("Project owners to add, set or remove")},
        "required": ["owners"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def modify_owners(issuer, project_id, allowed_host_types, request, reason):
    """Modify project owners."""

    # for some strange reason, DELETE requests without mimetype are not validated against schema
    # thank you, mr. laconical Original Wall-e Creator, we'll handle your legacy
    if request is None:
        raise BadRequestError("Wrong request mimetype, must be 'application/json'")

    action = get_api_action()

    project = projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)
    orig_owners, owners_to_add, owners_to_remove = set(projects.get_project_owners(project)), set(), set()

    try:
        owners_update = set(request["owners"] if action == API_ACTION_REMOVE else staff.check_owners(request["owners"]))
    except staff.InvalidOwnerError as e:
        raise BadRequestError(str(e))

    if action == API_ACTION_REMOVE:
        owners_to_remove = owners_update.intersection(orig_owners)
    elif action == API_ACTION_ADD:
        owners_to_add = owners_update - orig_owners
    else:  # set to new value
        owners_to_remove = orig_owners - owners_update
        owners_to_add = owners_update - orig_owners

    with audit_log.on_project_owners_update_request(issuer, project_id, {"owners": {action: owners_update}}, reason):
        if owners_to_add or owners_to_remove:
            change_handlers.on_project_owners_update(
                project,
                authorization.get_issuer_login(issuer),
                owners_to_add,
                owners_to_remove,
            )

    resp = {}
    if owners_to_add:
        resp["ownership_requested"] = sorted(owners_to_add)
    if owners_to_remove:
        resp["ownership_revocation_requested"] = sorted(owners_to_remove)

    return api_response(resp)


@api_handler(
    "/projects/<project_id>/is_project_owner/<login>",
    "GET",
    iam_permissions=iam.GetProjectApiIamPermission("project_id"),
)
def is_project_owner(project_id, login):
    """Check if <login> is an owner/user/superuser of <project_id>"""
    issuer = "{}@".format(login)
    project = get_by_id(project_id, fields=("id",))
    try:
        project.authorize_user(issuer)
        is_owner = True
    except UnauthorizedError:
        is_owner = False

    return api_response({"is_owner": is_owner})


@api_handler(
    "/projects/<project_id>/owned_vlans",
    get_api_action_methods(),
    {
        "type": "object",
        "properties": {
            "vlans": {"type": "array", "items": get_vlan_id_schema(), "description": "VLAN IDs owned by the project"},
        },
        "required": ["vlans"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def modify_owned_vlans(issuer, project_id, allowed_host_types, request, reason):
    """Modify VLANs owned by the project."""
    if not authorization.is_admin(issuer):
        projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    projects.check_id(project_id)

    if flask.request.method == "DELETE" and request is None:
        action = API_ACTION_SET
        vlans = []
    else:
        action = get_api_action()
        vlans = sorted(set(request["vlans"]))

        if not authorization.is_admin(issuer):
            check_vlan_authorization(issuer[:-1], vlans)

    update = get_api_action_update_query("owned_vlans", action, vlans)

    with audit_log.on_update_project(issuer, project_id, {"owned_vlans": {action: vlans}}, reason):
        updated_project = Project.objects(id=project_id).only("owned_vlans").modify(new=True, **update)
        if updated_project is None:
            raise ProjectNotFoundError()

    return api_response({"owned_vlans": updated_project.owned_vlans})


@api_handler(
    "/projects/<project_id>/vlan_scheme",
    "PUT",
    {
        "type": "object",
        "properties": {
            "scheme": {"type": "string", "enum": VLAN_SCHEMES, "description": "VLAN scheme"},
            "native_vlan": get_vlan_id_schema("Native VLAN ID"),
            "extra_vlans": {
                "type": "array",
                "items": get_vlan_id_schema(),
                "description": "A static list of VLAN IDs that should be assigned to each project's host",
            },
        },
        "required": ["scheme", "native_vlan"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def set_vlan_scheme(issuer, project_id, allowed_host_types, request, reason):
    """
    Set VLAN scheme for the project. This method resets hbf project id.
    To switch project to MTN, use `set_hbf_project_id` method.
    """

    # NB: Deprecating old behaviour. This can be validated via request schema,
    # but I am checking parameter manually to give the user a normal error message.
    if request["scheme"] in MTN_VLAN_SCHEMES:
        raise ResourceConflictError(
            "Setting 'mtn' VLAN scheme is deprecated."
            "Just set HBF project id and it will enable mtn for your project."
        )

    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    vlan_scheme, native_vlan, extra_vlans = request["scheme"], request["native_vlan"], request.get("extra_vlans")
    if extra_vlans is not None:
        extra_vlans = sorted(set(extra_vlans) - {native_vlan}) or None

    used_vlans = [native_vlan]
    if extra_vlans:
        used_vlans.extend(extra_vlans)

    audit = dict(request, hbf_project_id=None)
    with audit_log.on_update_project(issuer, project_id, audit, reason):
        for try_num in range(10):
            required_owned_vlans = walle.projects.authorize_vlans(project_id, used_vlans, allow_not_found_error=True)

            query = dict(id=project_id)
            if required_owned_vlans:
                query.update(owned_vlans__all=required_owned_vlans)

            with ProjectInterruptableLock(project_id):
                if Project.objects(**query).update(
                    multi=False,
                    **fix_mongo_set_kwargs(
                        unset__hbf_project_id=True,
                        set__vlan_scheme=vlan_scheme,
                        set__native_vlan=native_vlan,
                        set__extra_vlans=extra_vlans,
                    ),
                ):
                    return api_response(request)

        raise Error("Failed to set VLAN scheme due to high request concurrency.")


@api_handler(
    "/projects/<project_id>/vlan_scheme",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def reset_vlan_scheme(issuer, project_id, allowed_host_types, request, reason):
    """Reset VLAN scheme for the project. This method resets hbf project id as well."""

    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    audit = {
        "hbf_project_id": None,
        "vlan_scheme": None,
        "native_vlan": None,
        "extra_vlans": None,
    }

    with audit_log.on_update_project(issuer, project_id, audit, reason):
        with ProjectInterruptableLock(project_id):
            updated = Project.objects(id=project_id).update(
                unset__hbf_project_id=True,
                unset__vlan_scheme=True,
                unset__native_vlan=True,
                unset__extra_vlans=True,
                multi=False,
            )

        if not updated:
            raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/tags",
    get_api_action_methods(),
    {
        "type": "object",
        "properties": {"tags": schemas.get_tags_schema()},
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def set_project_tags(issuer, project_id, allowed_host_types, request, reason):
    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    action = get_api_action()
    tags = validated_tags(request["tags"] if request else [])

    if action == API_ACTION_ADD and not tags:
        return "", httplib.NO_CONTENT

    update = get_api_action_update_query("tags", action, tags)

    with audit_log.on_update_project(issuer, project_id, {"tags": {action: tags}}, reason):
        with ProjectInterruptableLock(project_id):
            updated = Project.objects(id=project_id).update(multi=False, **update)

        if not updated:
            raise ProjectNotFoundError()

        return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/hbf_project_id",
    ("POST", "PATCH"),
    {
        "type": "object",
        "properties": {
            "hbf_project_id": {
                "type": "string",
                "pattern": schemas.HEXADECIMAL,
                "description": "Hexadecimal HBF project id for HBF-enabled projects",
            },
            "ip_method": {
                "type": "string",
                "enum": [MTN_IP_METHOD_MAC, MTN_IP_METHOD_HOSTNAME],
                "description": "Method for determining host ip address, see docs for explanation. "
                "Default is {}".format(MTN_IP_METHOD_HOSTNAME),
            },
            "use_fastbone": {"type": "boolean", "description": "Use fastbone. Default is true"},
        },
        "required": ["hbf_project_id"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def set_hbf_project_id(issuer, project_id, allowed_host_types, request, reason):
    """Set the HBF/MTN project id for MTN-enabled projects. This method enables mtn vlan scheme for the project."""
    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    mtn_kwargs = set_project_for_mtn_kwargs(
        validated_hbf_project_id(request["hbf_project_id"]),
        request.get("ip_method", MTN_IP_METHOD_MAC),
        request.get("use_fastbone", True),
    )

    with audit_log.on_update_project(issuer, project_id, mtn_kwargs, reason):
        with ProjectInterruptableLock(project_id):
            updated = Project.objects(id=project_id).update(multi=False, **mtn_kwargs)

        if not updated:
            raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/hbf_project_id",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def remove_hbf_project_id(issuer, project_id, allowed_host_types, request, reason):
    """Remove the HBF project id from the project. This also resets vlan scheme to None."""
    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    audit = {"hbf_project_id": None, "vlan_scheme": None, "native_vlan": None, "extra_vlans": None}
    with audit_log.on_update_project(issuer, project_id, audit, reason):
        with ProjectInterruptableLock(project_id):
            Project.objects(
                # check that hbf_project_id is still set on project,
                # because we don't want to just drop any VLAN scheme,
                # we only want to drop hbf-project-id and 'mtn' vlan scheme.
                # With this, we can't be sure if project still exists, though.
                id=project_id,
                hbf_project_id__exists=True,
            ).update(
                multi=False,
                unset__hbf_project_id=True,
                unset__vlan_scheme=True,
                unset__native_vlan=True,
                unset__extra_vlans=True,
            )

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/bot_project_id",
    ("POST", "PATCH"),
    {
        "type": "object",
        "properties": {
            "bot_project_id": {"type": "integer", "minimum": -1, "description": "BOT/OEBS project id"},
            "abc_service_slug": {"type": "string", "description": "ABC service slug"},
        },
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
@admin_request
def set_bot_project_id(issuer, project_id, allowed_host_types, request, reason):
    """Set BOT/OEBS project id for the project."""
    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    abc_service_slug = request.get("abc_service_slug", None)
    req_bot_project_id = request.get("bot_project_id", None)
    if abc_service_slug:
        abc_service = abc.get_service_by_slug(abc_service_slug)
        bot_project_id = bot.get_bot_project_id_by_planner_id(abc_service["id"])
    elif req_bot_project_id:
        bot_project_id = req_bot_project_id
    else:
        raise BadRequestError("ABC service slug or BOT/OEBS project id should be specified")

    with audit_log.on_update_project(issuer, project_id, {"bot_project_id": bot_project_id}, reason):
        with ProjectInterruptableLock(project_id):
            updated = Project.objects(id=project_id).update(
                set__bot_project_id=validated_bot_project_id(bot_project_id), multi=False
            )

        if not updated:
            raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/dns_domain",
    ("POST", "PATCH"),
    {
        "type": "object",
        "properties": {
            "dns_domain": {
                "type": "string",
                "pattern": projects.DNS_DOMAIN_RE,
                "description": "Domain for DNS auto-configuration",
            },
            "yc_dns_zone_id": {"type": "string", "description": "YC DNS zone"},
        },
        "required": ["dns_domain"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
@admin_request
def set_dns_domain(issuer, project_id, request, reason):
    """Set domain for DNS auto-configuration for the project."""
    project = get_by_id(project_id)
    dns_domain = request["dns_domain"]

    if project.certificate_deploy:
        check_dns_domain_allowed_in_certificator(dns_domain)

    log_data = {"dns_domain": dns_domain}
    update_kwargs = {
        "set__dns_domain": dns_domain,
        "multi": False,
    }
    yc_dns_zone_id = request.get("yc_dns_zone_id")
    if yc_dns_zone_id is not None:
        log_data["yc_dns_zone_id"] = yc_dns_zone_id
        update_kwargs["set__yc_dns_zone_id"] = yc_dns_zone_id
    with audit_log.on_update_project(issuer, project_id, log_data, reason):
        with ProjectInterruptableLock(project_id):
            updated = Project.objects(id=project_id).update(**update_kwargs)
        if not updated:
            raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/dns_domain",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
@admin_request
def remove_dns_domain(issuer, project_id, request, reason):
    """Remove domain for DNS auto-configuration from the project."""
    project = get_by_id(project_id)

    if project.certificate_deploy:
        raise BadRequestError("Certificate deploy depends on DNS domain, please disable it first.")

    if project.host_shortname_template:
        raise BadRequestError("Host shortname template is set for the project, please clean it up first.")

    with audit_log.on_update_project(issuer, project_id, {"dns_domain": None}, reason):
        with ProjectInterruptableLock(project_id):
            updated = Project.objects(id=project_id).update(
                unset__dns_domain=True,
                unset__yc_dns_zone_id=True,
                multi=False,
            )

        if not updated:
            raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/host_shortname_template",
    ("PUT", "POST", "PATCH"),
    {
        "type": "object",
        "properties": {
            "host_shortname_template": {
                "type": "string",
                "description": "Template for short names for the project's hosts. "
                "Python format string with mandatory {index} and "
                "optional {location} and {bucket} placeholders.",
            },
        },
        "required": ["host_shortname_template"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def set_hostname_template(issuer, project_id, allowed_host_types, request, reason):
    """Set custom host shortname template for the project."""

    project = projects.get_authorized(issuer, project_id, ["dns_domain"], allowed_host_types=allowed_host_types)
    if not project.dns_domain:
        raise BadRequestError("Can not use custom host shortname template in project without dns settings.")

    host_shortname_template = validated_host_shortname_template(request["host_shortname_template"])

    with audit_log.on_update_project(issuer, project_id, request, reason):
        with ProjectInterruptableLock(project_id):
            updated = Project.objects(id=project_id).update(
                set__host_shortname_template=host_shortname_template, multi=False
            )

        if not updated:
            raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/host_shortname_template",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def remove_hostname_template(issuer, project_id, allowed_host_types, request, reason):
    """Remove custom host shortname template for the project."""

    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    with audit_log.on_update_project(issuer, project_id, {"host_shortname_template": None}, reason):
        with ProjectInterruptableLock(project_id):
            updated = Project.objects(id=project_id).update(unset__host_shortname_template=True, multi=False)

        if not updated:
            raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/automation_plot",
    ("POST", "PATCH"),
    {
        "type": "object",
        "properties": {
            "automation_plot_id": {"type": "string", "description": "Automation Plot unique identifier"},
        },
        "required": ["automation_plot_id"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=[
        iam.UpdateProjectApiIamPermission("project_id"),
        iam.GetAutomationPlotApiIamPermission("automation_plot_id"),
    ],
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def set_automation_plot(issuer, project_id, allowed_host_types, request, reason):
    """Set automation plot for project."""
    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    automation_plot_id = request["automation_plot_id"]

    with audit_log.on_update_project(issuer, project_id, {"automation_plot_id": automation_plot_id}, reason):
        with locked_automation_plot_id(automation_plot_id) as plot_id:
            updated = Project.objects(id=project_id).update(set__automation_plot_id=plot_id, multi=False)

        if not updated:
            raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/automation_plot",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def unset_automation_plot(issuer, project_id, allowed_host_types, request, reason):
    """Remove automation plot from the project."""
    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    with audit_log.on_update_project(issuer, project_id, {"automation_plot_id": None}, reason):
        updated = Project.objects(id=project_id).update(unset__automation_plot_id=True, multi=False)

        if not updated:
            raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/enable_automation",
    "PUT",
    {
        "type": "object",
        "properties": {
            "credit": schemas.get_automation_credit_schema(strict=True),
        },
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def enable_all_automation(issuer, project_id, allowed_host_types, request, reason):
    """
    Enable all automation types for the project.
    This method is half deprecated, only lives for backwards compatibility.
    """

    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    errors = []
    messages = []

    for automation in (PROJECT_HEALING_AUTOMATION, PROJECT_DNS_AUTOMATION):
        try:
            messages.append(_enable_automation(issuer, project_id, request, reason, automation, credits_strict=True))
        except ResourceConflictError as e:
            errors.append(str(e))

    if errors:
        raise ResourceConflictError("\n".join(errors + messages))

    return api_response(messages)


@api_handler(
    "/projects/<project_id>/enable_automation/dns",
    "PUT",
    {
        "type": "object",
        "properties": {
            "credit": schemas.get_automation_credit_schema(PROJECT_DNS_AUTOMATION, strict=True),
        },
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def enable_dns_automation(issuer, project_id, allowed_host_types, request, reason):
    """Enable DNS automation for the project."""
    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    message = _enable_automation(issuer, project_id, request, reason, PROJECT_DNS_AUTOMATION, credits_strict=True)
    return api_response(message)


@api_handler(
    "/projects/<project_id>/enable_automation/healing",
    "PUT",
    {
        "type": "object",
        "properties": {
            "credit": schemas.get_automation_credit_schema(PROJECT_HEALING_AUTOMATION, strict=False),
        },
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def enable_healing_automation(issuer, project_id, allowed_host_types, request, reason):
    """Enable healing automation for the project."""
    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    message = _enable_automation(issuer, project_id, request, reason, PROJECT_HEALING_AUTOMATION, credits_strict=False)
    return api_response(message)


def _enable_automation(issuer, project_id, request, reason, automation_type, credits_strict=False):
    if "credit" in request:
        credits = request["credit"].copy()
        credit_time = credits.pop("time")
    else:
        credits, credit_time = {}, None

    if credits_strict:
        credits = filter_dict_keys(credits, automation_type.get_limit_names())

    if not credits:  # Don't allow empty dicts
        credits, credit_time = None, None

    automation_type.enable_automation(issuer, project_id, credits, credit_time, reason=reason)
    return "Successfully enabled {}.".format(automation_type.get_automation_label())


@api_handler(
    "/projects/<project_id>/enable_automation",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
def disable_all_automation(issuer, project_id, request, reason):
    """Disable automation for the project."""
    return _disable_automation(issuer, project_id, reason, (PROJECT_HEALING_AUTOMATION, PROJECT_DNS_AUTOMATION))


@api_handler(
    "/projects/<project_id>/enable_automation/healing",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
def disable_healing_automation(issuer, project_id, request, reason):
    """Disable automation for the project."""
    return _disable_automation(issuer, project_id, reason, (PROJECT_HEALING_AUTOMATION,))


@api_handler(
    "/projects/<project_id>/enable_automation/dns",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def disable_dns_automation(issuer, project_id, allowed_host_types, request, reason):
    """Disable automation for the project."""
    return _disable_automation(issuer, project_id, reason, (PROJECT_DNS_AUTOMATION,))


def _disable_automation(issuer, project_id, reason, automation_types, allowed_host_types=None):
    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    messages = []
    for automation_type in automation_types:
        automation_type.disable_automation(issuer, project_id, reason)
        messages.append("Successfully disabled {}.".format(automation_type.get_automation_label()))

    return api_response(messages)


@api_handler(
    "/projects/<project_id>/automation_limits",
    ("PATCH", "POST"),
    schemas.get_automation_limits_schema(),
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def set_automation_limits(issuer, project_id, allowed_host_types, request, reason):
    """Set automation limits for the project."""

    automation_limits = request
    if not automation_limits:
        raise BadRequestError("At least one automation limit must be specified for update.")

    old_limits = automation_limits.keys() & schemas.OLD_AUTOMATION_LIMIT_NAMES
    if old_limits:
        raise BadRequestError(
            "Limits {} are not supported anymore. Please, refer to the method documentation"
            " (or upgrade your Wall-E client library/application).",
            ", ".join(old_limits),
        )

    # Just in case. Validate each action limit.
    for action_limits in automation_limits.values():
        util.limits.parse_timed_limits(action_limits)

    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    update = {}
    for name, action_limits in automation_limits.items():
        if action_limits:
            update["set__automation_limits__" + name] = action_limits
        else:
            update["unset__automation_limits__" + name] = True

    with audit_log.on_update_project(issuer, project_id, {"automation_limits": automation_limits}, reason):
        updated_project = Project.objects(id=project_id).only("automation_limits").modify(new=True, **update)
        if updated_project is None:
            raise ProjectNotFoundError()

        result_automation_limits = updated_project.to_api_obj(["automation_limits"])["automation_limits"]
        return api_response(result_automation_limits)


@api_handler(
    "/projects/<project_id>/host_limits",
    ("PATCH", "POST"),
    schemas.get_host_limits_schema(),
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def set_host_limits(issuer, project_id, allowed_host_types, request, reason):
    """Set host limits for the project."""

    host_limits = request
    if not host_limits:
        raise BadRequestError("At least one host limit must be specified for update.")

    # Just in case. Validate each action limit.
    for action_limits in host_limits.values():
        util.limits.parse_timed_limits(action_limits)

    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    update = {}
    for name, action_limits in host_limits.items():
        if action_limits:
            update["set__host_limits__" + name] = action_limits
        else:
            update["unset__host_limits__" + name] = True

    with audit_log.on_update_project(issuer, project_id, {"host_limits": host_limits}, reason):
        updated_project = Project.objects(id=project_id).only("host_limits").modify(new=True, **update)
        if updated_project is None:
            raise ProjectNotFoundError()

        result_host_limits = updated_project.to_api_obj(["host_limits"])["host_limits"]
        return api_response(result_host_limits)


@api_handler(
    "/projects/<project_id>/notifications/recipients",
    "PUT",
    schemas.get_recipients_schema("Recipients to set by severity"),
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def set_notification_recipients(issuer, project_id, allowed_host_types, request, reason):
    """Set recipients for project notifications."""

    if not request:
        raise BadRequestError("At least one severity must be specified for update.")

    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    update = {"set__notifications__recipients__" + severity: emails for severity, emails in request.items()}

    with audit_log.on_update_project(issuer, project_id, {"notifications": {"set_recipients": request}}, reason):
        updated_project = Project.objects(id=project_id).only("notifications.recipients").modify(new=True, **update)
        if updated_project is None:
            raise ProjectNotFoundError()

        new_recipients = updated_project.to_api_obj(["notifications.recipients"])["notifications"]["recipients"]
        return api_response(new_recipients)


@api_handler(
    "/projects/<project_id>/notifications/recipients",
    ("PATCH", "POST"),
    schemas.get_recipients_schema("Recipients to add by severity"),
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def add_notification_recipients(issuer, project_id, allowed_host_types, request, reason):
    """Add recipients for project notifications."""

    if not request:
        raise BadRequestError("At least one severity must be specified for update.")

    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    update = {"add_to_set__notifications__recipients__" + severity: emails for severity, emails in request.items()}

    with audit_log.on_update_project(issuer, project_id, {"notifications": {"add_recipients": request}}, reason):
        updated_project = Project.objects(id=project_id).only("notifications.recipients").modify(new=True, **update)
        if updated_project is None:
            raise ProjectNotFoundError()

        new_recipients = updated_project.to_api_obj(["notifications.recipients"])["notifications"]["recipients"]
        return api_response(new_recipients)


@api_handler(
    "/projects/<project_id>/notifications/recipients",
    "DELETE",
    schemas.get_recipients_schema("Recipients to remove by severity"),
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def remove_notification_recipients(issuer, project_id, allowed_host_types, request, reason):
    """Remove recipients from project notifications."""

    if not request:
        raise BadRequestError("At least one severity must be specified for update.")

    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    recipients_field_raw_name = Project.notifications.db_field + "." + Notifications.recipients.db_field
    update = {recipients_field_raw_name + "." + severity: emails for severity, emails in request.items()}

    with audit_log.on_update_project(issuer, project_id, {"notifications": {"remove_recipients": request}}, reason):
        # MongoEngine doesn't support pullAll for nested fields
        result = Project._get_collection().find_and_modify(
            {Project.id.db_field: project_id}, {"$pullAll": update}, fields=[recipients_field_raw_name], new=True
        )

        if result is None:
            raise ProjectNotFoundError()

        updated_project = Project._from_son(result)
        new_recipients = updated_project.to_api_obj(["notifications.recipients"])["notifications"]["recipients"]
        return api_response(new_recipients)


@api_handler(
    "/projects/<project_id>/reports",
    ("PUT", "POST", "PATCH"),
    {
        "type": "object",
        "properties": {
            "enabled": {"type": "boolean", "description": "Enable/disable reports for the project"},
            "queue": {
                "type": "string",
                "pattern": projects.STARTREK_QUEUE_PATTERN,
                "description": "Create tickets in this queue",
            },
            "summary": {"type": "string", "description": "Optional substring to add to the ticket summary"},
            "extra": {"type": "object", "description": "Other ticket parameters like component or followers"},
        },
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def modify_report_parameters(issuer, project_id, allowed_host_types, request, reason):
    """Set failure host report params for the project."""
    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)
    project = projects.get_by_id(project_id, fields=["reports"])

    optional_fields = ("extra", "summary")
    action = get_api_action()

    if action == API_ACTION_SET or not project.reports:
        if "queue" not in request:
            raise BadRequestError("'queue' is a required property")

        request.setdefault("enabled", True)

        for field in optional_fields:
            if not request.get(field, None):
                request.pop(field, None)

        update_kwargs = dict(set__reports=request)
    else:
        for field in optional_fields:
            if field in request and not request[field]:
                request[field] = None  # allow users to unset fields properly by passing empty dict/list/string

        update_kwargs = {"set__reports__" + field: value for field, value in request.items()}

    if not update_kwargs:
        return "", httplib.NO_CONTENT

    with audit_log.on_update_project(issuer, project_id, {"reports": request}, reason):
        with ProjectInterruptableLock(project_id):
            updated = Project.objects(id=project_id).update(multi=False, **fix_mongo_set_kwargs(**update_kwargs))

        if not updated:
            raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/reports",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
def remove_report_parameters(issuer, project_id, allowed_host_types, request, reason):
    """Remove failure host report params from the project."""
    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    with audit_log.on_update_project(issuer, project_id, {"reports": None}, reason):
        with ProjectInterruptableLock(project_id):
            updated = Project.objects(id=project_id).update(unset__reports=True, multi=False)

        if not updated:
            raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT


def _render_role_member(role):
    return "@" + role.member if role.is_group else role.member


def _render_role_members(roles):
    return [_render_role_member(role) for role in roles]


def _render_role_members_with_id(roles):
    return {_render_role_member(role): role.id for role in roles}


def _render_role_member_info(role):
    return {
        "role_id": role.id,
        "role": role.path[-1],
        "state": role.type,
        "member": _render_role_member(role),
    }


@api_handler(
    "/projects/<project_id>/requested_owners",
    "GET",
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
def get_requested_owners(project_id):
    """Get list of project owners requested in IDM"""
    # TODO remove it after UI has switched to /requested_owners_with_request_id
    project = get_by_id(project_id, fields=("id",))
    requested_roles = project_push.get_project_roles(project, type="requested")
    return api_response(_render_role_members(requested_roles))


@api_handler(
    "/projects/<project_id>/revoking_owners",
    "GET",
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
def get_revoking_owners(project_id):
    """Get list of revoking project owners from IDM"""
    # TODO remove it after UI has switched to /revoking_owners_with_request_id
    project = get_by_id(project_id, fields=("id",))
    revoking_roles = project_push.get_project_roles(project, state=["depriving", "depriving_validation"])
    return api_response(_render_role_members(revoking_roles))


@api_handler(
    "/projects/<project_id>/requested_owners_with_request_id",
    "GET",
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
def get_requested_owners_with_request_id(project_id):
    """Get dict {requested project owner -> IDM request id} from IDM"""
    project = get_by_id(
        project_id,
        fields=(
            "id",
            "type",
        ),
    )
    requested_roles = project_push.get_project_roles(project, type="requested")
    rendered = {"result": _render_role_members_with_id(requested_roles)}
    return api_response(rendered)


@api_handler(
    "/projects/<project_id>/revoking_owners_with_request_id",
    "GET",
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
def get_revoking_owners_with_request_id(project_id):
    """Get dict {revoking project owner -> IDM request id} from IDM"""
    project = get_by_id(
        project_id,
        fields=(
            "id",
            "type",
        ),
    )
    revoking_roles = project_push.get_project_roles(project, state=["depriving", "depriving_validation"])
    rendered = {"result": _render_role_members_with_id(revoking_roles)}
    return api_response(rendered)


@api_handler(
    "/projects/<project_id>/roles_state",
    "GET",
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
def get_roles_status(project_id):
    project = get_by_id(project_id, fields=("id",))

    roles = [
        _render_role_member_info(role_member_info)
        for role_member_info in project_push.get_project_roles(project)
        if role_member_info.type != idm.RoleStateType.INACTIVE
    ]

    rendered = {"result": roles}
    return api_response(rendered)


@api_handler(
    "/projects/<project_id>/rebooting_via_ssh",
    "PUT",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
def enable_ssh_rebooting(issuer, project_id, request, reason):
    project = projects.get_authorized(issuer, project_id)
    role_manager = ProjectRole.get_role_manager(ProjectRole.SSH_REBOOTER, project)

    with role_manager.audit_log_writer.on_request_add_member(ROBOT_WALLE_OWNER, issuer, reason):
        with idm.BatchRequest() as br:
            role_manager.request_add_member(br, issuer)
            br.execute()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/rebooting_via_ssh",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
def disable_ssh_rebooting(issuer, project_id, request, reason):
    project = projects.get_authorized(issuer, project_id)

    role_manager = ProjectRole.get_role_manager(ProjectRole.SSH_REBOOTER, project)

    with role_manager.audit_log_writer.on_request_remove_member(ROBOT_WALLE_OWNER, issuer, reason):
        with idm.BatchRequest() as br:
            role_manager.request_remove_member(br)
            br.execute()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/role/<role>/members",
    "GET",
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
def list_role_members(project_id, role):
    role_manager = _get_role_manager(role, project_id)
    members = role_manager.list_members()
    return api_response({"members": members})


@api_handler(
    "/projects/<project_id>/role/<role>/members",
    ("POST", "PATCH"),
    {
        "type": "object",
        "properties": {
            "member": {"type": "string", "pattern": Project.owners.field.regex.pattern},
        },
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
def request_add_role_member(issuer, project_id, role, request, reason):
    member = request.get("member")
    role_manager = _get_role_manager(role, project_id, issuer)
    with role_manager.audit_log_writer.on_request_add_member(member, issuer, reason):
        with idm.BatchRequest() as br:
            role_manager.request_add_member(br, authorization.get_issuer_login(issuer), member)
            br.execute()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/role/<role>/members",
    "DELETE",
    {
        "type": "object",
        "properties": {
            "member": {"type": "string", "pattern": Project.owners.field.regex.pattern},
        },
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
def request_remove_role_member(issuer, project_id, role, request, reason):
    if request is None:
        raise BadRequestError("Wrong request mimetype, must be 'application/json'")

    member = request.get("member")
    role_manager = _get_role_manager(role, project_id, issuer)
    with role_manager.audit_log_writer.on_request_remove_member(member, issuer, reason):
        with idm.BatchRequest() as br:
            role_manager.request_remove_member(br, member)
            br.execute()

    return "", httplib.NO_CONTENT


def _get_role_manager(role, project_id, issuer=None):
    if role not in ProjectRole.ALL:
        raise BadRequestError("Role parameter value must be one of [{}]".format(", ".join(ProjectRole.ALL)))

    if issuer is not None:
        project = projects.get_authorized(issuer, project_id)
    else:
        project = projects.get_by_id(project_id)

    role_manager = ProjectRole.get_role_manager(role, project)
    return role_manager


@api_handler(
    "/projects/<project_id>/repair_request_severity",
    ("POST", "PATCH"),
    {
        "type": "object",
        "properties": {
            "repair_request_severity": {
                "type": "string",
                "enum": RepairRequestSeverity.ALL,
                "description": "Repair request severity for Eine profile & BOT tickets for hosts in the project",
            },
        },
        "required": ["repair_request_severity"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
@admin_request
def set_repair_request_severity(issuer, project_id, request, reason):
    """Set repair request severity for Eine profile & BOT tickets for hosts in the project."""
    repair_request_severity = request["repair_request_severity"]

    with audit_log.on_update_project(issuer, project_id, {"repair_request_severity": repair_request_severity}, reason):
        with ProjectInterruptableLock(project_id):
            updated = Project.objects(id=project_id).update(
                set__repair_request_severity=repair_request_severity, multi=False
            )
        if not updated:
            raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/tier",
    ("POST", "PATCH"),
    {
        "type": "object",
        "properties": {"tier": {"type": "integer", "description": "Which tier should handle hosts of the project"}},
        "required": ["tier"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
)
@admin_request
def set_project_tier(issuer, project_id, request, reason):
    """Set tier for project."""
    tier = request["tier"]
    if tier not in get_existing_tiers():
        raise BadRequestError(f"Unknown tier '{tier}'")

    with audit_log.on_update_project(issuer, project_id, {"tier": tier}, reason):
        with ProjectInterruptableLock(project_id):
            updated = Project.objects(id=project_id).update(set__tier=tier, multi=False)
        if not updated:
            raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/cms_settings",
    ("POST", "PATCH"),
    {
        "type": "object",
        "properties": {
            "cms_settings": {
                "type": "array",
                "items": schemas.get_cms_schema(),
                "description": "List of project's CMS settings",
            },
        },
        "additionalProperties": False,
        "required": ["cms_settings"],
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateProjectApiIamPermission("project_id"),
    allowed_host_types=HOST_TYPES_WITH_MANUAL_OPERATION,
)
@production_only
def set_cms_settings(issuer, project_id, allowed_host_types, request, reason):
    if not len(request.get("cms_settings", [])):
        raise BadRequestError("You must specify at least one cms settings")

    project = projects.get_authorized(
        issuer,
        project_id,
        additional_fields=[
            "cms",
            "cms_max_busy_hosts",
            "cms_api_version",
            "cms_tvm_app_id",
            "bot_project_id",
            "cms_settings",
        ],
        allowed_host_types=allowed_host_types,
    )
    authorization_enabled = config.get_value("authorization.enabled")

    all_cms_settings = []
    tmp_update_kwargs_for_legacy_fields = {}
    unique_cms_urls = set()

    for request_cms in request["cms_settings"]:
        cms_settings = CmsSettings.from_request(request_cms)
        unique_cms_urls.add(cms_settings.url)

        if cms_settings.url == projects.DEFAULT_CMS_NAME:
            all_cms_settings.append(cms_settings.to_default_cms_kwargs())
            tmp_update_kwargs_for_legacy_fields = dict(
                set__cms=cms_settings.url,
                set__cms_max_busy_hosts=cms_settings.max_busy_hosts,
                unset__cms_api_version=True,
                unset__cms_tvm_app_id=True,
            )
        else:
            if authorization_enabled and not authorization.is_admin(authorization.get_issuer_login(issuer)):
                check_cms_tvm_app_id_requirements(cms_settings, project.bot_project_id)

            all_cms_settings.append(cms_settings.to_user_cms_api_kwargs())
            tmp_update_kwargs_for_legacy_fields = dict(
                set__cms=cms_settings.url,
                set__cms_api_version=cms_settings.api_version,
                set__cms_tvm_app_id=cms_settings.tvm_app_id,
                unset__cms_max_busy_hosts=True,
            )

    if len(request["cms_settings"]) != len(unique_cms_urls):
        raise BadRequestError("All cms settings must be different")

    if projects.DEFAULT_CMS_NAME in unique_cms_urls and len(unique_cms_urls) > 1:
        raise BadRequestError("Default cms can be the only one cms of project")

    with audit_log.on_update_project(issuer, project_id, request, reason):
        updated_project = Project.objects(id=project_id).modify(
            new=True, set__cms_settings=all_cms_settings, **tmp_update_kwargs_for_legacy_fields
        )
        if updated_project is None:
            raise ProjectNotFoundError()

        return api_response(updated_project.to_api_obj())


@api_handler(
    "/projects/<project_id>/maintenance_plot",
    ("POST", "PATCH"),
    {
        "type": "object",
        "properties": {
            "maintenance_plot_id": {"type": "string", "description": "Maintenance Plot unique identifier"},
        },
        "required": ["maintenance_plot_id"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
    allowed_host_types=[HostType.MAC, HostType.SERVER, HostType.VM, HostType.SHADOW_SERVER],
)
def set_maintenance_plot(issuer, project_id, allowed_host_types, request, reason):
    """Set maintenance plot for project."""
    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    plot = get_maintenance_plot(request["maintenance_plot_id"])

    with audit_log.on_update_project(issuer, project_id, {"maintenance_plot_id": plot.id}, reason):
        with MaintenancePlotInterruptableLock(plot.id):
            updated = Project.objects(id=project_id).update(set__maintenance_plot_id=plot.id, multi=False)

            if not updated:
                raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT


@api_handler(
    "/projects/<project_id>/maintenance_plot",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
    allowed_host_types=[HostType.MAC, HostType.SERVER, HostType.VM, HostType.SHADOW_SERVER],
)
def unset_maintenance_plot(issuer, project_id, allowed_host_types, request, reason):
    """Remove maintenance plot from the project."""
    projects.get_authorized(issuer, project_id, allowed_host_types=allowed_host_types)

    with audit_log.on_update_project(issuer, project_id, {"maintenance_plot_id": None}, reason):
        updated = Project.objects(id=project_id).update(unset__maintenance_plot_id=True, multi=False)

        if not updated:
            raise ProjectNotFoundError()

    return "", httplib.NO_CONTENT
