""" Scenario API """

import http.client
import logging

from sepelib.core.constants import DAY_SECONDS
from walle.authorization import iam, blackbox
from walle.scenario.authorization import authorize_scenario
from walle.scenario.constants import (
    ScenarioFsmStatus,
    ScenarioWorkStatus,
    ScriptName,
    ADMIN_PARAM_NAMES_OF_HOST_TRANSFER,
    ScenarioModifyAction,
    WORK_STATUS_API_PARAM,
    RESERVED_LABELS,
)
from walle.scenario.errors import EmptySwitchForNocScenario
from walle.scenario.scenario import Scenario
from walle.scenario.script import ScriptRegistry
from walle.scenario.script_args import (
    HostsTransferParams,
    NocSoftParams,
    NocHardParams,
    ItdcMaintenanceParams,
    ReservedHostsTransferParams,
)
from walle.scenario.validators import validate_scenario_params
from walle.util.api import api_handler, api_response, scenario_id_handler, admin_request
from walle.util.misc import drop_none, filter_dict_keys
from walle.views.api.scenario_api import operations

log = logging.getLogger(__name__)

PAGINATION_PARAMS = {
    "offset": {"type": "integer", "minimum": 0, "description": "Offset to return $limit entries"},
    "limit": {"type": "integer", "description": "Max number of entries per page"},
    "max_limit": 100,
}

LABELS_SCHEME = {
    "properties": {label_name: {"type": "string"} for label_name in RESERVED_LABELS},
    "type": "object",
    "additionalProperties": True,
    "description": "User-defined labels",
}

SCRIPT_ARGS_SCHEME = {
    "anyOf": [
        HostsTransferParams.get_json_schema(),
        NocSoftParams.get_json_schema(),
        NocHardParams.get_json_schema(),
        ItdcMaintenanceParams.get_json_schema(),
        ReservedHostsTransferParams.get_json_schema(),
    ],
}


def get_scenario_general_params_scheme(ignored_params=None):
    params = {
        "name": {"type": "string", "description": "Name of the scenario"},
        "ticket_key": {"type": "string", "description": "Related ticket key"},
        "labels": LABELS_SCHEME,
        "autostart": {"type": "boolean", "description": "Start scenario after creation. Default is False"},
        "hosts": {
            "type": "array",
            "items": {"type": ["integer", "string"]},
            "description": "Inventory numbers, FQDNs or UUIDs of hosts",
            "maxItems": 1000,
        },
    }
    ignored_params = set(ignored_params) if ignored_params else set()
    return filter_dict_keys(params, set(params.keys()) - ignored_params)


@api_handler(
    "/scenarios",
    "POST",
    {
        "type": "object",
        "properties": dict(
            get_scenario_general_params_scheme(),
            **{
                "scenario_type": {
                    "enum": ScriptRegistry.get_keys(),
                    "description": "Type of the scenario",
                },
                "script_args": SCRIPT_ARGS_SCHEME,
            }
        ),
        "required": ["name", "scenario_type", "ticket_key"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def add_scenario(issuer, request, reason):
    """Register a new scenario."""
    scenario_type = request["scenario_type"]
    authorize_scenario(scenario_type, issuer)

    authorize_admin_params(issuer, request.get("script_args", {}))
    if scenario_type == ScriptName.NOC_HARD:
        # NOTE(rocco66): switch validation is actually soft NOC limits check
        validate_scenario_params(request, validate_switch=False, validate_project_id=False)
    else:
        validate_scenario_params(request)

    try:
        scenario = operations.add_scenario(issuer, reason=reason, **request)
        return api_response(scenario.to_api_obj(), code=http.client.CREATED)
    except EmptySwitchForNocScenario as e:
        return api_response({"message": str(e)}, code=http.client.OK)


@api_handler(
    "/scenarios/hosts_add_rtc",
    "POST",
    {
        "type": "object",
        "properties": dict(
            get_scenario_general_params_scheme(),
            **{
                "target_project_id": {"type": "string", "description": "Target Project ID"},
                "target_hardware_segment": {"type": "string", "description": "Target Qloud segment"},
                "abc_service_id": {"type": "integer", "description": "Target abc service for host after deletion"},
                "workdays_only": {
                    "type": "boolean",
                    "description": "Set host's maintenance only at workdays. Default is true",
                },
                "idle_time": {
                    "type": "integer",
                    "description": "How long to wait in maintenance. Default is {}".format(DAY_SECONDS),
                },
            }
        ),
        "required": ["name", "ticket_key", "hosts"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def add_hosts_add_rtc_scenario(issuer, request, reason):

    authorize_scenario(ScriptName.HOSTS_TRANSFER, issuer)
    authorize_admin_params(issuer, request)
    validate_scenario_params(request)

    script_args = drop_none(
        dict(
            target_project_id=request.get('target_project_id', None),
            target_hardware_segment=request.get('target_hardware_segment', None),
            abc_service_id=request.get('abc_service_id', None),
            workdays_only=request.get('workdays_only', True),
            idle_time=request.get('idle_time', DAY_SECONDS),
        )
    )
    prepared_request_data = dict(
        name=request.get('name'),
        scenario_type=ScriptName.HOSTS_TRANSFER,
        hosts=request.get('hosts'),
        ticket_key=request.get('ticket_key'),
        autostart=request.get('autostart'),
        labels=request.get('labels'),
        script_args=script_args,
    )

    scenario = operations.add_scenario(issuer, reason=reason, **prepared_request_data)
    return api_response(scenario.to_api_obj(), code=http.client.CREATED)


def authorize_admin_params(issuer, params):
    if set(params.keys()).intersection(set(ADMIN_PARAM_NAMES_OF_HOST_TRANSFER)):
        blackbox.authorize(
            issuer,
            "You are not change these params: {}".format(ADMIN_PARAM_NAMES_OF_HOST_TRANSFER),
            authorize_admins=True,
        )


def get_scenario_scheme():
    return {
        "scenario_id": {"type": "array", "items": {"type": "integer"}, "description": "Filter by id of scenario"},
        "name": {"type": "array", "items": {"type": "string"}, "description": "Filter by name of scenario"},
        "issuer": {"type": "array", "items": {"type": "string"}, "description": "Filter by issuer"},
        "ticket_key": {"type": "array", "items": {"type": "string"}, "description": "Filter by ticket key"},
        "status": {
            "type": "array",
            "items": {
                "enum": [v.value for v in iter(ScenarioFsmStatus)],
                "description": "Filter by scenario status",
            },
        },
        WORK_STATUS_API_PARAM: {
            "type": "array",
            "items": {
                "enum": [v.value for v in iter(ScenarioWorkStatus)],
                "description": "Filter by scenario work status",
            },
        },
        "reverse": {"type": "boolean", "description": "Scenario entries output order. Default is true"},
        "scenario_type": {
            "type": "array",
            "items": {"type": "string"},
            "description": "Filter by type of scenario",
        },
    }


@api_handler(
    "/scenarios",
    "GET",
    params=get_scenario_scheme(),
    with_fields=Scenario,
    with_paging=PAGINATION_PARAMS,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def get_scenarios(query_args):
    """Query scenarios"""
    return operations.find_scenarios(query_args)


@scenario_id_handler(
    "/scenarios/<scenario_id>",
    "GET",
    with_fields=Scenario,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def get_scenario(scenario_id, query_args):
    """Get the specified scenario."""

    fields = query_args.get("fields")
    scenario = operations.get_scenario(scenario_id, fields=Scenario.api_query_fields(fields))
    return api_response(scenario.to_api_obj(fields), code=http.client.OK)


@api_handler(
    "/scenarios/labels",
    "POST",
    {
        "type": "object",
        "properties": {
            "labels": LABELS_SCHEME,
        },
    },
    params=get_scenario_scheme(),
    with_fields=Scenario,
    with_paging=PAGINATION_PARAMS,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def find_scenarios_by_labels(request, query_args):
    """DEPRECATED. Query scenarios."""
    labels = request.get("labels")
    return operations.find_scenarios(query_args, labels)


@scenario_id_handler(
    "/scenarios/<scenario_id>/cancel",
    ("POST", "PATCH"),
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def cancel_scenario(issuer, scenario_id, request, reason):
    scenario = operations.cancel_scenario(issuer, scenario_id, reason=reason)
    return api_response(scenario.to_api_obj(), code=http.client.OK)


@scenario_id_handler(
    "/scenarios/<scenario_id>/pause",
    ("POST", "PATCH"),
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def pause_scenario(issuer, scenario_id, request, reason):
    scenario = operations.pause_scenario(issuer, scenario_id, reason=reason)
    return api_response(scenario.to_api_obj(), code=http.client.OK)


@scenario_id_handler(
    "/scenarios/<scenario_id>",
    "PATCH",
    {
        "type": "object",
        "properties": dict(
            {
                "name": {"type": "string", "description": "New name of the scenario"},
                # FIXME Noop. Remove after Janitor changes
                "script_args": SCRIPT_ARGS_SCHEME,
                "labels": LABELS_SCHEME,
            }
        ),
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def modify_scenario(issuer, scenario_id, request, reason):
    """Modify the specified scenario."""
    validate_scenario_params(request)
    if "script_args" in request:  # this hack is temporary, to be removed after Janitor changes
        del request["script_args"]
    scenario = operations.modify_scenario(issuer, scenario_id, reason=reason, **request)
    return api_response(scenario.to_api_obj(), code=http.client.ACCEPTED)


@scenario_id_handler(
    "/scenarios/<scenario_id>/start",
    ("POST", "PATCH"),
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def start_scenario(issuer, scenario_id, request, reason):
    scenario = operations.start_scenario(issuer, scenario_id, reason=reason)
    return api_response(scenario.to_api_obj(), code=http.client.ACCEPTED)


@scenario_id_handler(
    "/scenarios/<scenario_id>/snapshot",
    "GET",
    include_to_api_reference=False,
    iam_permissions=iam.NoOneApiIamPermission(),
)
def get_scenario_snapshot(scenario_id):
    """Get the snapshot of the specified scenario"""
    snapshot = operations.get_scenario_state(scenario_id)
    return api_response(snapshot, code=http.client.OK)


@api_handler(
    "/scenarios/<scenario_id>/apply-action",
    "POST",
    {
        "type": "object",
        "properties": dict(
            {
                "action": {
                    "type": "string",
                    "description": "Action to be applied to the selected scenario",
                    "enum": ScenarioModifyAction.ALL,
                },
            }
        ),
        "required": ["action"],
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.NoOneApiIamPermission(),
)
@admin_request
def apply_action_to_scenario(issuer: str, scenario_id: int, request: dict[str, str], reason: str) -> dict:
    action = request["action"]

    operations.apply_action_to_scenario(issuer=issuer, scenario_id=scenario_id, action=action, reason=reason)
    return api_response("", code=http.client.ACCEPTED)
