from collections import Counter, defaultdict

from walle.clients.racktables import shorten_switch_name
from walle.constants import HostType
from walle.errors import BadRequestError, ResourceConflictError
from walle.hosts import Host
from walle.scenario.errors import EmptySwitchForNocScenario
from walle.util.mongo import MongoDocument


def get_hosts_invs_and_most_common_datacenter(switch, project_id):
    query = {
        "location.switch": shorten_switch_name(switch),
        "type": HostType.SERVER,
    }
    if project_id:
        query["project"] = project_id

    hosts = list(
        MongoDocument.for_model(Host).find(
            query, {"inv": 1, "location.short_datacenter_name": 1, "_id": 0, "scenario_id": 1, "name": 1}
        )
    )

    if not hosts:
        error_message = "There are no hosts associated with switch '%s'" % switch
        if project_id:
            error_message += ", or there are no hosts belonging to the project '%s' in this switch" % project_id
            raise BadRequestError(error_message)
        else:
            raise EmptySwitchForNocScenario(switch)
    else:
        scenario_id_to_fqdn_map = defaultdict(list)
        for host in hosts:
            if host.scenario_id:
                scenario_id_to_fqdn_map[host.scenario_id].append(host.name)
        if len(scenario_id_to_fqdn_map):
            error_message = "Can't create scenario, one or more hosts are already acquired by other scenarios: "
            message_parts = [
                "hosts with fqdns {} acquired by scenario with id {}".format(", ".join(fqdns), scenario_id)
                for scenario_id, fqdns in scenario_id_to_fqdn_map.items()
            ]
            error_message += ", ".join(message_parts)
            raise ResourceConflictError(error_message)

    most_common_datacenter = None
    counts = Counter(host.location.short_datacenter_name for host in hosts).most_common(1)
    if counts:
        [(most_common_datacenter, _count)] = counts

    return [host.inv for host in hosts], most_common_datacenter
