"""Operations on scenario"""

import logging

import mongoengine
from mongoengine.errors import DoesNotExist

from walle import audit_log
from walle.errors import ResourceAlreadyExistsError, BadRequestError, ResourceNotFoundError, ResourceConflictError
from walle.hosts import Host
from walle.locks import ScenarioInterruptableLock
from walle.models import timestamp
from walle.scenario.authorization import authorize_scenario, authorize_operation_on_scenario
from walle.scenario.constants import (
    ScenarioFsmStatus,
    ALL_CAN_PAUSE_SCENARIO_STATUSES,
    ALL_CANCELABLE_SCENARIO_STATUSES,
    ALL_CAN_START_SCENARIO_STATUSES,
    WORK_STATUS_LABEL_NAME,
    ScenarioWorkStatus,
    SWITCH_LABEL_NAME,
    DATACENTER_LABEL_NAME,
    ScriptName,
    WORK_STATUS_API_PARAM,
)
from walle.scenario.definitions.base import get_scenario_definition
from walle.scenario.errors import ScenarioValidationError, KeyDoesNotExistsRegistryError, ScenarioDoesNotHaveDefinition
from walle.scenario.handlers import ScenarioModifyActionRegistry
from walle.scenario.host_groups_builders.hosts_list_splitters import get_host_list_splitter
from walle.scenario.host_stage_info import HostStageInfo
from walle.scenario.scenario import Scenario
from walle.scenario.script import ScriptRegistry
from walle.util.api import get_query_result, api_response, FilterQueryParser
from walle.util.misc import drop_none, InvOrUUIDOrName
from walle.views.api.scenario_api import (
    hosts_transfer_scenario,
    noc_scenario,
    full_process_scenario,
)
from walle.views.api.scenario_api.scenario_postprocessor import ScenarioPostprocessor
from walle.views.helpers.hosts import get_hosts_objs_from_unknown_hosts_identifiers

log = logging.getLogger(__name__)


def save_scenario_parameters_to_data_storage(scenario_type: str, script_args: dict, data_storage_dict: dict):
    data_storage = get_scenario_definition(scenario_type).data_storage(data_storage_dict)
    try:
        scenario_parameters = data_storage.scenario_parameters_class(**script_args)
    except Exception as e:
        raise BadRequestError(e)
    data_storage.write_scenario_parameters(scenario_parameters)
    return scenario_parameters


def run_on_scenario_creation_checks(scenario: Scenario):
    try:
        on_scenario_creation_checks = get_scenario_definition(scenario.scenario_type).on_scenario_creation_checks
    except ScenarioDoesNotHaveDefinition:
        return

    checks_results = []
    if on_scenario_creation_checks:
        for check in on_scenario_creation_checks:
            checks_results.append(check(scenario))

    all_checks_passed = True
    error_strings = []
    for result in checks_results:
        if not result.check_passed:
            all_checks_passed = False
            error_strings += result.messages

    if not all_checks_passed:
        raise BadRequestError(", ".join(error_strings))


def add_scenario(
    issuer,
    name,
    scenario_type,
    hosts=None,
    reason=None,
    ticket_key=None,
    script_args=None,
    labels=None,
    autostart=False,
) -> Scenario:
    if not script_args:
        script_args = {}

    host_inv_list = []
    data_storage_dict = {}

    if not labels:
        labels = {}
    labels[WORK_STATUS_LABEL_NAME] = ScenarioWorkStatus.CREATED

    # Get hosts' inventory numbers for NOC scenario.
    if scenario_type == ScriptName.NOC_SOFT:
        save_scenario_parameters_to_data_storage(ScriptName.NOC_SOFT, script_args, data_storage_dict)

        host_inv_list, most_common_datacenter = noc_scenario.get_hosts_invs_and_most_common_datacenter(
            script_args["switch"], script_args.get("project_id")
        )

        labels[DATACENTER_LABEL_NAME] = most_common_datacenter
        labels[SWITCH_LABEL_NAME] = script_args["switch"]

    elif scenario_type == ScriptName.ITDC_MAINTENANCE:
        scenario_parameters = save_scenario_parameters_to_data_storage(
            ScriptName.ITDC_MAINTENANCE, script_args, data_storage_dict
        )
        if scenario_parameters.rack:
            host_inv_list = full_process_scenario.get_hosts_invs_by_location(scenario_parameters.rack)
        else:
            hosts_objs = get_hosts_objs_from_unknown_hosts_identifiers(hosts)
            host_inv_list = full_process_scenario.get_hosts_invs(hosts_objs)

    elif scenario_type == ScriptName.NOC_HARD:
        scenario_parameters = save_scenario_parameters_to_data_storage(
            ScriptName.NOC_HARD, script_args, data_storage_dict
        )
        if scenario_parameters.switch:
            host_inv_list = full_process_scenario.get_hosts_invs_by_switch(scenario_parameters.switch)
        else:
            hosts_objs = get_hosts_objs_from_unknown_hosts_identifiers(hosts)
            host_inv_list = full_process_scenario.get_hosts_invs(hosts_objs)
            if not host_inv_list:
                raise BadRequestError("One of \"hosts\" list or \"switch\" parameter must be specified")

    # Ensure that hosts provided for add/remove hosts scenario are integers (Bot IDs).
    elif scenario_type == ScriptName.HOSTS_TRANSFER:
        save_scenario_parameters_to_data_storage(ScriptName.HOSTS_TRANSFER, script_args, data_storage_dict)
        host_inv_list = hosts_transfer_scenario.get_hosts_invs(hosts)

    elif scenario_type == ScriptName.BENCHMARK:
        host_inv_list = hosts_transfer_scenario.get_hosts_invs(hosts)

    # Parse 'hosts' if no other special cases.
    if not host_inv_list:
        host_inv_list = [host.inv for host in get_hosts_objs_from_unknown_hosts_identifiers(hosts)]

    current_time = timestamp()
    status = ScenarioFsmStatus.STARTED if autostart else ScenarioFsmStatus.CREATED

    try:
        Scenario.validate_name(name)
        script_func = ScriptRegistry.get(scenario_type)
        script = script_func(script_args)
    except ScenarioValidationError as e:
        raise BadRequestError("Invalid params for scenario: {}".format(e))
    root_stage_info = script.serialize()

    # Split hosts to groups if scenario type supports splitting.
    hosts_list_splitter = get_host_list_splitter(scenario_type, default=None)
    if hosts_list_splitter:
        data_storage = get_scenario_definition(scenario_type).data_storage(data_storage_dict)
        hosts_states, host_groups_sources = Scenario.split_scenario_hosts_to_groups(
            host_inv_list, script_func.uses_uuids, hosts_list_splitter
        )
        data_storage.write_host_groups_sources(host_groups_sources)  # This updates `data_storage_dict` created earlier.

        # Generate `MaintenanceApproversGroups` and add them to `data_storage_dict` for backwards compatibility.
        # To be removed after `HostGroupApproveStage` and `MaintenanceApproversWorkflowStage` stops using it,
        # and `MaintenanceApproversScheduler` is removed.
        maintenance_approvers_groups = Scenario.get_maintenance_approvers_groups_from_host_groups_sources(
            host_groups_sources
        )
        data_storage.write(maintenance_approvers_groups)
    else:
        hosts_states = Scenario.create_list_of_host_states(host_inv_list, resolve_uuids=script_func.uses_uuids)

    scenario = Scenario(
        scenario_id=Scenario.next_id(),
        name=name,
        scenario_type=scenario_type,
        hosts=hosts_states,
        issuer=issuer,
        ticket_key=ticket_key,
        script_args=script_args,
        next_check_time=Scenario.get_new_next_check_time(),
        action_time=current_time,
        creation_time=current_time,
        labels=labels,
        stage_info=root_stage_info,
        status=status,
        uses_uuid_keys=script_func.uses_uuids,
        data_storage=data_storage_dict,
    )

    run_on_scenario_creation_checks(scenario)

    with audit_log.on_add_scenario(
        issuer,
        scenario.id,
        scenario.name,
        reason=reason,
        scenario_type=scenario_type,
        ticket_key=ticket_key,
        script_args=script_args,
        status=status,
    ):
        try:
            scenario.save(force_insert=True)
        except mongoengine.NotUniqueError:
            raise ResourceAlreadyExistsError("Scenario with the specified name already exists in Wall-E database.")
        return scenario


def cancel_scenario(issuer, scenario_id, reason=None):
    try:
        scenario = Scenario.objects.get(scenario_id=scenario_id)
    except DoesNotExist:
        raise ResourceNotFoundError("Scenario {} does not exist".format(scenario_id))

    authorize_operation_on_scenario(scenario, issuer)

    if scenario.status not in ALL_CANCELABLE_SCENARIO_STATUSES:
        raise ResourceConflictError(
            "You can cancel only scenarios in next statuses: {}".format(ALL_CANCELABLE_SCENARIO_STATUSES)
        )

    with audit_log.on_cancel_scenario(issuer, scenario_id, reason=reason):
        with ScenarioInterruptableLock(scenario_id):
            scenario.cancel()

        return scenario


def pause_scenario(issuer, scenario_id, reason=None):
    try:
        scenario = Scenario.objects.get(scenario_id=scenario_id)
    except DoesNotExist:
        raise ResourceNotFoundError("Scenario {} does not exist".format(scenario_id))

    authorize_operation_on_scenario(scenario, issuer)

    if scenario.status not in ALL_CAN_PAUSE_SCENARIO_STATUSES:
        raise ResourceConflictError(
            "You can pause only scenarios in next statuses: {}".format(ALL_CAN_PAUSE_SCENARIO_STATUSES)
        )

    with audit_log.on_pause_scenario(issuer, scenario_id, reason=reason):
        with ScenarioInterruptableLock(scenario_id):
            scenario.pause()

        return scenario


def modify_scenario(issuer, scenario_id, reason=None, name=None, labels=None):
    update = drop_none(
        {
            "name": name,
            "labels": labels,
        }
    )

    try:
        scenario = Scenario.objects.get(scenario_id=scenario_id)
    except DoesNotExist:
        raise ResourceNotFoundError("Scenario {} does not exist".format(scenario_id))

    if scenario.status != ScenarioFsmStatus.CREATED and list(update.keys()) != ["labels"]:
        raise ResourceConflictError("You can't modify scenario in this status with selected fields")

    authorize_scenario(scenario.scenario_type, issuer)

    with audit_log.on_modify_scenario(issuer, scenario_id, update, reason=reason):
        modify_dict = {}
        if name:
            modify_dict["set__name"] = name
        if labels:
            modify_dict.update({f"set__labels__{l_name}": l_val for l_name, l_val in labels.items()})
        scenario.modify(**modify_dict)
        return scenario


def get_scenario(scenario_id, fields=None, **query):
    objects = Scenario.objects(scenario_id=scenario_id, **query)

    if fields is not None:
        objects = objects.only(*fields)

    try:
        scenario = objects.get()
    except DoesNotExist:
        raise ResourceNotFoundError("Scenario {} does not exist".format(scenario_id))
    else:
        return scenario


def start_scenario(issuer, scenario_id, reason=None):
    with audit_log.on_start_scenario(issuer, scenario_id, {"status": ScenarioFsmStatus.STARTED}, reason):
        try:
            scenario = Scenario.objects.get(scenario_id=scenario_id)
        except DoesNotExist:
            raise ResourceNotFoundError("Scenario {} does not exist".format(scenario_id))

        authorize_scenario(scenario.scenario_type, issuer)

        if scenario.status not in ALL_CAN_START_SCENARIO_STATUSES:
            raise ResourceConflictError(
                f"You can start only scenarios in next states: {ALL_CAN_START_SCENARIO_STATUSES}"
            )

        scenario.start()
        return scenario


def find_scenarios(query_args, labels=None):
    query_parser = FilterQueryParser(
        Scenario,
        enum_fields=("scenario_id", "status"),
        substring_fields=("name", "scenario_type", "issuer", "ticket_key"),
    )
    label_queries = []
    work_status_filter = query_args.pop(WORK_STATUS_API_PARAM, None)
    if work_status_filter:
        label_queries.append({f"{Scenario.labels.db_field}.{WORK_STATUS_LABEL_NAME}": {"$in": work_status_filter}})

    queries = [query_parser.parse_query_filter(query_args)]

    labels = labels or {}
    if labels:
        label_queries += [{f"{Scenario.labels.db_field}.{key}": value} for key, value in labels.items()]
    if label_queries:
        queries.append({"$and": label_queries})

    postprocessor = ScenarioPostprocessor()
    return api_response(
        get_query_result(
            Scenario,
            queries,
            Scenario.scenario_id,
            query_args,
            postprocessor=postprocessor,
            reverse=query_args.get("reverse", True),
        )
    )


def get_scenario_state(scenario_id):
    objects = Scenario.objects(scenario_id=scenario_id).only(Scenario.stage_info.db_field)

    try:
        scenario = objects.get()
    except DoesNotExist:
        raise ResourceNotFoundError("Scenario {} does not exist".format(scenario_id))
    else:
        return scenario.stage_info.to_mongo().to_dict()


def get_host_scenario_state(scenario_id, host_id):
    host_id_query = InvOrUUIDOrName(host_id)
    host = Host.get_by_host_id_query(host_id_query)
    objects = HostStageInfo.objects(scenario_id=scenario_id, host_uuid=host.uuid).only(
        HostStageInfo.stage_info.db_field
    )

    try:
        hsi = objects.get()
    except DoesNotExist:
        raise ResourceNotFoundError("Scenario {} does not exist".format(scenario_id))
    else:
        return hsi.stage_info.to_mongo().to_dict()


def apply_action_to_scenario(scenario_id: int, issuer: str, action: str, reason: str):
    objects = Scenario.objects(scenario_id=scenario_id)

    try:
        scenario = objects.get()
        modify_action = ScenarioModifyActionRegistry.get(action)
        update = {"applied action to scenario": action}
        with audit_log.on_modify_scenario(issuer, scenario_id, update, reason=reason):
            modify_action(scenario)
    except DoesNotExist:
        raise ResourceNotFoundError("Scenario {} does not exist".format(scenario_id))
    except KeyDoesNotExistsRegistryError:
        raise ResourceNotFoundError("Action {} does not exist".format(action))
