"""Settings"""  # Will be used as category name in API reference
import http.client
import logging

from sepelib.core.constants import DAY_SECONDS, WEEK_SECONDS, MINUTE_SECONDS
from walle import audit_log
from walle.application import app
from walle.errors import BadRequestError, ResourceConflictError, ResourceNotFoundError
from walle.expert import automation
from walle.hosts import DMCRule
from walle.locks import SettingsUpdateInterruptableGlobalLock
from walle.models import timestamp, FsmHandbrake, TimedLimitDocument
from walle.util import cloud_tools
from walle.util.api import api_handler, admin_request, api_response

log = logging.getLogger(__name__)

_MAX_FSM_HANDBRAKE_TIMEOUT = WEEK_SECONDS
_MIN_FSM_HANDBRAKE_TIMEOUT = 10 * MINUTE_SECONDS  # just sanity check
_DEFAULT_FSM_HANDBRAKE_TIMEOUT = DAY_SECONDS


@api_handler("/settings", "GET")
def get_settings():
    """Returns current Wall-E settings."""

    return api_response(app.settings().to_api_obj())


@api_handler(
    "/settings",
    ("PATCH", "POST"),
    {
        "type": "object",
        "properties": {
            "disable_automation": {
                "type": "boolean",
                "description": "Disable all automation that may hurt the cluster",
            },
            "disable_healing": {"type": "boolean", "description": "Disable automated healing"},
            "disable_dns_automation": {"type": "boolean", "description": "Disable DNS automation"},
            "enable_fsm_handbrake": {
                "type": "boolean",
                "description": "Stop all task processing to prevent cluster destruction",
            },
            "fsm_handbrake_timeout": {
                "type": "integer",
                "description": "Timeout for fsm handbrake, default is enable for one day",
            },
            "enable_scenario_fsm_handbrake": {
                "type": "boolean",
                "description": "Stop all scenario processing to prevent cluster destruction",
            },
            "scenario_fsm_handbrake_timeout": {
                "type": "integer",
                "description": "Timeout for scenario fsm handbrake, default is enable for one day",
            },
            "inventory_invalid_hosts_limit": {"type": "integer", "minimum": 0},
        },
        "additionalProperties": False,
    },
    authenticate=True,
    with_reason=True,
)
@admin_request
def update_settings(issuer, request, reason):
    """Change current settings."""

    if not request:
        raise BadRequestError("At least one parameter must be specified for update.")

    if "disable_automation" in request:
        # NB: this branch terminates the handler
        if "disable_healing" in request or "disable_dns" in request:
            raise BadRequestError(
                "You should not use disable_automation and disable_healing/disable_dns at the same time"
            )
        return _toggle_global_automation(
            issuer,
            reason=reason,
            disable_healing=request.get("disable_automation"),
            disable_dns_automation=request.get("disable_automation"),
        )

    elif "disable_healing" in request or "disable_dns_automation" in request:
        # NB: this branch does not terminates the handler, it will continue to other parameters.
        _toggle_global_automation(
            issuer,
            reason=reason,
            disable_healing=request.pop("disable_healing", None),
            disable_dns_automation=request.pop("disable_dns_automation", None),
        )

    with SettingsUpdateInterruptableGlobalLock():

        settings = app.settings()

        if "enable_fsm_handbrake" in request or "fsm_handbrake_timeout" in request:
            enabled = request.pop("enable_fsm_handbrake", None)
            timeout = request.pop("fsm_handbrake_timeout", None)
            changes = _toggle_fsm_handbrake(
                issuer,
                settings.fsm_handbrake,
                reason=reason,
                enabled=enabled,
                timeout=timeout,
            )
            if settings.fsm_handbrake:
                settings.modify({"fsm_handbrake__audit_log_id": settings.fsm_handbrake.audit_log_id}, **changes)
            elif enabled:
                settings.modify({"fsm_handbrake__exists": False}, **changes)

        if "enable_scenario_fsm_handbrake" in request or "scenario_fsm_handbrake_timeout" in request:
            enabled = request.pop("enable_scenario_fsm_handbrake", None)
            timeout = request.pop("scenario_fsm_handbrake_timeout", None)
            changes = _toggle_scenario_fsm_handbrake(
                issuer,
                settings.scenario_fsm_handbrake,
                reason=reason,
                enabled=enabled,
                timeout=timeout,
            )
            if settings.scenario_fsm_handbrake:
                settings.modify(
                    {"scenario_fsm_handbrake__audit_log_id": settings.scenario_fsm_handbrake.audit_log_id}, **changes
                )
            elif enabled:
                settings.modify({"scenario_fsm_handbrake__exists": False}, **changes)

    if request:
        with audit_log.on_change_global_settings(issuer, request, reason=reason):
            settings.validate_obj_fields(request)

            for field, value in request.items():
                setattr(settings, field, value)

            settings.save()

    return api_response(settings.to_api_obj())


def _toggle_fsm_handbrake(issuer, handbrake_attribute, enabled, timeout, reason):
    enabled, timeout = _validate_fsm_handbrake_params(handbrake_attribute, enabled, timeout)
    changes = dict()

    if enabled:
        timeout_time = timestamp() + timeout

        if handbrake_attribute:
            changes = _extend_fsm_handbrake(issuer, handbrake_attribute, timeout_time, reason)
        else:
            changes = _set_fsm_handbrake(issuer, timeout_time, reason)
    elif handbrake_attribute is not None:
        changes = unset_fsm_handbrake(issuer, reason, "fsm_handbrake")

    return changes


def _toggle_scenario_fsm_handbrake(issuer, handbrake_attribute, enabled, timeout, reason):
    enabled, timeout = _validate_fsm_handbrake_params(handbrake_attribute, enabled, timeout)
    changes = {}

    if enabled:
        timeout_time = timestamp() + timeout

        if handbrake_attribute:
            changes = _extend_scenario_fsm_handbrake(issuer, handbrake_attribute, timeout_time, reason)
        else:
            changes = _set_scenario_fsm_handbrake(issuer, timeout_time, reason)
    elif handbrake_attribute is not None:
        changes = unset_fsm_handbrake(issuer, reason, "scenario_fsm_handbrake")

    return changes


def unset_fsm_handbrake(issuer, reason, attr_type):
    return {f"unset__{attr_type}": True}


def _validate_fsm_handbrake_params(handbrake_attribute, enabled, timeout):

    if timeout is not None:
        if enabled is None and handbrake_attribute is None:
            raise BadRequestError("So should be fsm handbrake enabled or disabled?")

        elif handbrake_attribute is not None:
            enabled = True  # extending

    if enabled and timeout is None:
        timeout = _DEFAULT_FSM_HANDBRAKE_TIMEOUT

    if enabled and not (_MIN_FSM_HANDBRAKE_TIMEOUT <= timeout <= _MAX_FSM_HANDBRAKE_TIMEOUT):
        raise BadRequestError(
            "Use timeout between {} and {} seconds.", _MIN_FSM_HANDBRAKE_TIMEOUT, _MAX_FSM_HANDBRAKE_TIMEOUT
        )

    return enabled, timeout


def _extend_fsm_handbrake(issuer, handbrake_attribute, timeout_time, reason):
    log_entry = audit_log.on_extend_fsm_handbrake(
        issuer,
        project_id=None,
        ticket_key=None,
        timeout_time=timeout_time,
        audit_log_id=handbrake_attribute.audit_log_id,
        reason=reason,
    )

    with log_entry:
        return {"set__fsm_handbrake__timeout_time": timeout_time}


def _extend_scenario_fsm_handbrake(issuer, handbrake_attribute, timeout_time, reason):
    log_entry = audit_log.on_extend_fsm_handbrake(
        issuer,
        project_id=None,
        ticket_key=None,
        timeout_time=timeout_time,
        audit_log_id=handbrake_attribute.audit_log_id,
        reason=reason,
    )

    with log_entry:
        return {"set__scenario_fsm_handbrake__timeout_time": timeout_time}


def _set_fsm_handbrake(issuer, timeout_time, reason):
    log_entry = audit_log.on_fsm_handbrake(
        issuer, project_id=None, ticket_key=None, timeout_time=timeout_time, reason=reason
    )
    with log_entry:
        return {"set__fsm_handbrake": FsmHandbrake(timeout_time=timeout_time, audit_log_id=log_entry.id)}


def _set_scenario_fsm_handbrake(issuer, timeout_time, reason):
    log_entry = audit_log.on_scenario_fsm_handbrake(
        issuer, scenario_id=None, ticket_key=None, timeout_time=timeout_time, reason=reason
    )
    with log_entry:
        return {"set__scenario_fsm_handbrake": FsmHandbrake(timeout_time=timeout_time, audit_log_id=log_entry.id)}


def _toggle_global_automation(issuer, reason, disable_healing=None, disable_dns_automation=None):

    messages = []
    errors = []
    automation_types = [
        (automation.GLOBAL_HEALING_AUTOMATION, disable_healing),
        (automation.GLOBAL_DNS_AUTOMATION, disable_dns_automation),
    ]
    for automation_type, disable in automation_types:
        if disable is None:
            continue

        try:
            if disable:
                automation_type.disable(reason, issuer=issuer)
            else:
                automation_type.enable(reason, issuer=issuer)
        except ResourceConflictError as e:
            errors.append(str(e))
        else:
            messages.append(
                "Successfully {} {}.".format(
                    "disabled" if disable else "enabled", automation_type.get_automation_label()
                )
            )

    if errors:
        raise ResourceConflictError("\n".join(errors + messages))

    return api_response(app.settings().reload().to_api_obj())


@api_handler("/config", "GET")
def get_config():
    """Returns some Wall-E config parameters."""

    response_obj = dict(
        # needed for the yasm panel in the UI
        ctype=cloud_tools.get_ctype()
    )
    return api_response(response_obj)


@api_handler("/settings/checks_percentage", "GET")
def get_checks_percentage_index():
    return api_response(
        [{"check": check, "percent": percent} for check, percent in app.settings().checks_percentage_overrides.items()]
    )


@api_handler("/settings/checks_percentage/<check>", "GET")
def get_checks_percentage_check_override(check):
    settings = app.settings()

    if check not in settings.checks_percentage_overrides:
        raise ResourceNotFoundError("Check {} not found".format(check))

    return api_response({"check": check, "percent": settings.checks_percentage_overrides[check]})


@api_handler(
    "/settings/checks_percentage/<check>",
    "PUT",
    {
        "type": "object",
        "properties": {
            "percent": {
                "type": "integer",
                "minimum": 0,
                "maximum": 100,
                "description": "Enable check on percent of hosts",
            },
        },
    },
    authenticate=True,
    with_reason=True,
)
@admin_request
def update_checks_percentage_check_override(request, check, issuer, reason):
    settings = app.settings()

    response = {"check": check, "percent": request["percent"]}

    with audit_log.on_change_checks_percentage_overrides(issuer, response, reason):
        settings.checks_percentage_overrides[check] = request["percent"]
        settings.save()

    return api_response(response)


@api_handler("/settings/checks_percentage/<check>", "DELETE", authenticate=True, with_reason=True)
@admin_request
def delete_checks_percentage_check_override(request, check, issuer, reason):
    settings = app.settings()

    if check not in settings.checks_percentage_overrides:
        raise ResourceNotFoundError("Check {} does not exists".format(check))

    response = {"check": check, "percent": settings.checks_percentage_overrides[check]}

    with audit_log.on_change_checks_percentage_overrides(issuer, response, reason):
        del settings.checks_percentage_overrides[check]
        settings.save()

    return api_response(response)


@api_handler("/settings/global_timed_limits_overrides", "GET")
def get_global_timed_limits_overrides():
    return api_response(
        {
            failure_name: timed_limit.to_mongo()
            for failure_name, timed_limit in app.settings().global_timed_limits_overrides.items()
        }
    )


@api_handler(
    "/settings/global_timed_limits_overrides/<failure_name>",
    "PUT",
    {
        "type": "object",
        "properties": {
            "period": {"type": "string", "description": "Time period (1d, 3h, 15m, ...)"},
            "limit": {"type": "integer", "minimum": 1, "description": "Count of failures in period"},
        },
    },
    authenticate=True,
    with_reason=True,
)
@admin_request
def update_global_timed_limits_override(request, failure_name, issuer, reason):
    settings = app.settings()
    override = {"period": request["period"], "limit": request["limit"]}
    with audit_log.on_change_global_timed_limits_overrides(issuer, override, reason):
        settings.global_timed_limits_overrides[failure_name] = TimedLimitDocument(**override)
        settings.save()
    return api_response(override)


@api_handler("/settings/global_timed_limits_overrides/<failure_name>", "DELETE", authenticate=True, with_reason=True)
@admin_request
def delete_global_timed_limits_override(request, failure_name, issuer, reason):
    settings = app.settings()
    limit_override = settings.global_timed_limits_overrides.get(failure_name)
    if not limit_override:
        raise ResourceNotFoundError("Limit override {} does not exists".format(failure_name))

    response = {"failure_name": failure_name, "period": limit_override["period"], "limit": limit_override["limit"]}
    with audit_log.on_change_checks_percentage_overrides(issuer, response, reason):
        del settings.global_timed_limits_overrides[failure_name]
        settings.save()

    return api_response(response)


@api_handler(
    "/settings/dmc_rules",
    "POST",
    {
        "type": "object",
        "properties": {
            "rule_query": {
                "type": "object",
                "properties": {
                    "project_ids_included": {
                        "type": "array",
                        "items": {"type": "string"},
                        "description": "A list of projects to include in DMC screening",
                    },
                    "project_ids_excluded": {
                        "type": "array",
                        "items": {"type": "string"},
                        "description": "A list of projects to exclude from DMC screening",
                    },
                    "tiers_included": {
                        "type": "array",
                        "items": {"type": "integer"},
                        "description": "A list of tiers to include to DMC screening",
                    },
                    "tiers_excluded": {
                        "type": "array",
                        "items": {"type": "integer"},
                        "description": "A list of tiers to exclude from DMC screening",
                    },
                    "physical_locations_included": {
                        "type": "array",
                        "items": {"type": "string"},
                        "description": "Include by physical location (format: country|city|datacenter|queue|rack)",
                    },
                    "physical_locations_excluded": {
                        "type": "array",
                        "items": {"type": "string"},
                        "description": "Exclude by physical location (format: country|city|datacenter|queue|rack)",
                    },
                },
                "additionalProperties": False,
                "description": "Query to include/exclude hosts from DMC screening",
                "required": [],
            },
        },
        "additionalProperties": False,
        "required": ["rule_query"],
    },
    authenticate=True,
    with_reason=True,
)
@admin_request
def create_dmc_rule(issuer, request, reason):
    rule_query = request["rule_query"]

    log.info("%s", rule_query)

    with audit_log.on_add_dmc_rule(issuer, rule_query, reason):
        rule = DMCRule(id=DMCRule.next_id(), rule_query=rule_query)
        rule.save()
        return api_response(rule.to_api_obj())


@api_handler(
    "/settings/dmc_rules",
    "GET",
    with_fields=DMCRule,
)
def list_all_dmc_rules(query_args):
    """List all dmc rules."""

    def get_all_dmc_rules(fields):
        return DMCRule.with_fields(fields)

    fields = query_args.get("fields")
    return api_response([p.to_api_obj(fields) for p in get_all_dmc_rules(fields)])


def get_dmc_rule(rule_id, fields=None):
    try:
        return DMCRule.with_fields(fields).get(id=rule_id)
    except DMCRule.DoesNotExist:
        raise ResourceNotFoundError("DMC rule does not exists")


@api_handler(
    "/settings/dmc_rules/<rule_id>",
    "GET",
    with_fields=DMCRule,
)
def get_dmc_rule_by_id(rule_id, query_args):
    """Return DMC rule."""
    fields = query_args.get("fields")
    rule = get_dmc_rule(rule_id, fields=fields)
    return api_response(rule.to_api_obj(fields))


@api_handler(
    "/settings/dmc_rules/<rule_id>",
    "DELETE",
    authenticate=True,
    with_reason=True,
)
@admin_request
def delete_dmc_rule(issuer, rule_id, request, reason):
    """Delete specified DMC rule."""

    rule = get_dmc_rule(rule_id)
    with audit_log.on_delete_dmc_rule(issuer, rule_id, reason):
        rule.delete()

    return "", http.client.NO_CONTENT
