import itertools

from walle import constants
from walle.clients import startrek
from walle.clients.utils import retry
from walle.errors import BadRequestError
from walle.util.api import api_handler, api_response
from walle.util.template_loader import JinjaTemplateRenderer
from walle.views.helpers.hosts import get_hosts_objs_from_unknown_hosts_identifiers


def _find_hosts_obj(hosts):
    hosts_obj = get_hosts_objs_from_unknown_hosts_identifiers(
        hosts,
        fields=[
            "name",
            "inv",
            "project",
            "location.city",
            "messages",
        ],
    )
    if not hosts_obj:
        raise BadRequestError("No hosts in request")
    if diff := set(hosts) - {h["name"] for h in hosts_obj} - {h["inv"] for h in hosts_obj}:
        raise BadRequestError(f"Can't find some hosts: {','.join(diff)}")
    return hosts_obj


def _make_description(hosts_obj, additional_description):
    grouped_hosts_obj = []
    sorted_hosts = sorted(hosts_obj, key=lambda h: (h["project"], h["name"]))
    for project, group in itertools.groupby(sorted_hosts, key=lambda h: h["project"]):
        grouped_hosts_obj.append((project, list(group)))
    return JinjaTemplateRenderer().render_template(
        "custom_ticket_description.txt",
        grouped_hosts_obj=grouped_hosts_obj,
        additional_description=additional_description,
    )


def _enrich_summary(summary, hosts):
    return f"{summary} ({len(hosts)} hosts)"


@retry(interval=1, backoff=2, exceptions=(startrek.StartrekClientConnectionError,))
def _create_ticket_attempt(st_client, queue, issuer, summary, description, queue_specific_kwargs):
    kwargs = {
        "queue": queue,
        "summary": summary,
        "description": description,
        "followers": [{"login": constants.ROBOT_WALLE_OWNER}],
        "createdBy": issuer,
    }
    kwargs.update(queue_specific_kwargs)
    return st_client.create_issue(kwargs)


def _get_queue_specific_kwargs(queue: constants.TicketTrackerQueue):
    if queue == constants.TicketTrackerQueue.EXP:
        return {
            "type": constants.INCIDENT_TICKET_TYPE,
            "components": [constants.HARDWARE_AND_FIRMWARE_EXP_COMPONENT],
            "fixVersions": [constants.OTHER_EXP_VERSION],
        }
    else:
        return {}


@api_handler(
    "/create-ticket",
    "POST",
    {
        "type": "object",
        "properties": {
            "hosts": {
                "type": "array",
                "items": {"type": ["integer", "string"]},
                "description": "Inventory numbers, FQDNs or UUIDs of hosts",
                "maxItems": 1000,
            },
            "queue": {
                "type": "string",
                "description": "Queue for ticket",
            },
            "summary": {
                "type": "string",
                "description": "Summary for ticket",
            },
            "description": {
                "type": "string",
                "description": "Some description for ticket",
            },
        },
        "required": ["hosts", "queue", "summary"],
        "additionalProperties": False,
    },
    authenticate=True,
)
def create_ticket_for_hosts(issuer, request):
    """Create custom Tracker ticket for hosts."""
    hosts = request["hosts"]
    description = _make_description(_find_hosts_obj(hosts), request.get("description", ""))
    queue = request["queue"]
    enriched_summary = _enrich_summary(request["summary"], hosts)
    queue_specific_kwargs = _get_queue_specific_kwargs(queue)
    queue_specific_kwargs.setdefault("tags", []).append(constants.CREATED_FROM_WALLE_TRACKER_TAG)
    st_client = startrek.get_client()
    response = _create_ticket_attempt(
        st_client,
        queue,
        issuer[:-1],
        enriched_summary,
        description,
        queue_specific_kwargs,
    )
    return api_response({"link": startrek.get_ticket_url_by_id(response["key"])})
