"""Tokens"""  # Will be used as category name in API reference

import logging

import flask

from sepelib.core import config
from walle.authorization import csrf, blackbox, has_iam
from walle.clients import oauth
from walle.errors import BadRequestError, UnauthenticatedError
from walle.util.api import api_handler, api_response

log = logging.getLogger(__name__)

YC_CSRF_COOKIE_NAME = "yc_session"


@api_handler("/access-token", "GET")
def get_access_token_authorization_url():
    """Returns a URL where user can get an authorization code to obtain an API access token."""

    return api_response({"authorization_url": oauth.get_authorization_url()})


@api_handler(
    "/access-token",
    "POST",
    {
        "type": "object",
        "properties": {"authorization_code": {"type": "string", "description": "Authorization confirmation code"}},
        "required": ["authorization_code"],
        "additionalProperties": False,
    },
)
def obtain_access_token(request):
    """Obtains API access token using the specified authorization confirmation code."""

    authorization_code = request["authorization_code"]

    try:
        access_token = oauth.get_token(authorization_code)
    except oauth.OauthError as e:
        raise BadRequestError("Unable to obtain API access token: {}", e)

    return api_response({"access_token": access_token})


@api_handler("/csrf-token", "GET")
def get_csrf_token():
    """Returns CSRF token."""

    if has_iam():
        session_id = flask.request.cookies.get(YC_CSRF_COOKIE_NAME)
    else:
        session_id = blackbox.authenticate(config.get_value("oauth.client_id")).session_id
    if session_id is None:
        raise UnauthenticatedError("You must be authenticated by Session ID to perform this request.")

    return api_response({"csrf_token": csrf.get_csrf_token(session_id)})
