"""User"""  # Will be used as category name in API reference

import logging
from collections import defaultdict

from walle import authorization
from walle.authorization import iam, has_iam
from walle.clients import staff
from walle.errors import UnauthenticatedError
from walle.expert.automation_plot import AutomationPlot
from walle.idm.project_role_managers import iter_project_roles_memberships, ProjectRole
from walle.util.api import api_handler, api_response

log = logging.getLogger(__name__)


@api_handler("/user", "GET", authenticate=True)
def get_user(issuer):
    """Returns user info."""

    if not authorization.is_user(issuer):
        raise UnauthenticatedError("You must be authenticated to perform this request.")

    login = authorization.get_issuer_login(issuer)

    groups = []
    iam_user_info = ""
    if has_iam():
        my_projects = iam.get_projects_with_access()
        my_automation_plots = []
        project_roles = {}
        iam_user_info = iam.get_iam_user_info()
    else:
        try:
            groups = staff.get_user_groups(login)
        except Exception:
            log.exception("Can't get groups for %r", login)
        owner = [login] + groups

        role_to_projects, project_to_roles = defaultdict(set), defaultdict(set)
        for project_id, role, _ in iter_project_roles_memberships(member__in=owner):
            role_to_projects[role].add(project_id)
            project_to_roles[project_id].add(role)

        my_projects = sorted(role_to_projects[ProjectRole.OWNER])
        project_roles = {project: sorted(roles) for project, roles in project_to_roles.items()}
        my_automation_plots = [ap.id for ap in AutomationPlot.objects(owners__in=owner).only("id")]

    return api_response(
        {
            "login": login,
            "admin": authorization.is_admin(login),
            "projects": my_projects,
            "project_roles": project_roles,
            "automation_plots": my_automation_plots,
            "groups": groups,
            "iam_user_info": iam_user_info,
        }
    )
