"""Default CMS API"""

import functools
import http.client
import logging

from sepelib.core import config
from walle import default_cms as cms
from walle.application import app
from walle.clients.cms import CmsTaskType, CmsTaskAction
from walle.errors import ApiError, ResourceNotFoundError
from walle.util.api import api_response, generic_api_handler, get_api_error_headers, walle_only

log = logging.getLogger(__name__)


def api_handler(*args, **kwargs):
    return generic_api_handler(app.cms_blueprint, *args, oauth_client_id=config.get_value("cms.client_id"), **kwargs)


def log_exceptions(method):
    def _decorator(func):
        @functools.wraps(func)
        def decorated_func(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except ResourceNotFoundError:
                raise
            except Exception:
                kwargs_wo_request = {k: v for k, v in kwargs.items() if k != "request"}
                log.exception(
                    "Got exception when asked for method %s with args %s and kwargs %s", method, args, kwargs_wo_request
                )
                raise

        return decorated_func

    return _decorator


@api_handler(
    "/<project>/tasks",
    "POST",
    {
        "type": "object",
        "properties": {
            "id": {"type": "string", "minLength": 1, "maxLength": 255, "description": "Task ID"},
            "type": {"enum": CmsTaskType.ALL, "description": "Task type"},
            "issuer": {"type": "string", "minLength": 1, "description": "Action issuer"},
            "action": {"enum": CmsTaskAction.ALL, "description": "Requested action"},
            "hosts": {
                "type": "array",
                "minItems": 1,
                "items": {"type": "string", "minLength": 1},
                "description": "Hosts to process the action on",
            },
        },
        "required": ["id", "type", "issuer", "action", "hosts"],
        "additionalProperties": False,
    },
    params={
        "dry_run": {
            "type": "boolean",
            "description": "Don't make any changes, emulate request processing. Default is false",
        },
    },
    authenticate=True,
)
@walle_only
@log_exceptions("add_task")
def add_task(issuer, project, query_args, request):
    """Registers a new task."""
    return api_response(cms.add_task(project, request, dry_run=query_args.get("dry_run", False)))


@api_handler("/<project>/tasks/<task_id>", "GET")
@log_exceptions("get_task")
def get_task(project, task_id):
    """Returns the specified task."""
    return api_response(cms.get_task(project, task_id))


@api_handler("/<project>/tasks", "GET")
@log_exceptions("get_tasks")
def get_tasks(project):
    """Returns all registered tasks."""
    return api_response({"result": cms.get_tasks(project)})


@api_handler("/<project>/tasks/<task_id>", "DELETE", authenticate=True)
@walle_only
@log_exceptions("delete_task")
def delete_task(issuer, project, task_id):
    """Deletes the specified task."""
    cms.delete_task(project, task_id)
    return "", http.client.NO_CONTENT


@app.cms_blueprint.errorhandler(ApiError)
def _api_error_handler(error):
    return api_response({"message": str(error)}, code=error.http_code, headers=get_api_error_headers(error))


@app.cms_blueprint.errorhandler(Exception)
def _internal_error_handler(error):
    return api_response({"message": str(error)}, code=http.client.INTERNAL_SERVER_ERROR)
