from walle.clients.cms import CmsTaskAction
from walle.errors import RequestValidationError, ResourceConflictError
from walle.hosts import HostOperationState, TaskType, Host, HostState, StateExpire
from walle.tasks import schedule_switch_to_maintenance
from walle.views.helpers.validators import prepare_maintenance_params


def set_host_maintenance_for_host(
    issuer,
    host,
    ticket_key,
    timeout_time=None,
    timeout_status=None,
    power_off=False,
    disable_admin_requests=False,
    ignore_cms=False,
    reason=None,
    cms_task_action=CmsTaskAction.PROFILE,
    operation_state=HostOperationState.OPERATION,
):

    maintenance_params = prepare_maintenance_params(timeout_time, timeout_status, ticket_key, operation_state)
    timeout_time, timeout_status, ticket_key, operation_state = maintenance_params

    schedule_switch_to_maintenance(
        issuer,
        TaskType.MANUAL,
        host,
        timeout_time=timeout_time,
        target_status=timeout_status,
        ticket_key=ticket_key,
        reason=reason,
        power_off=power_off,
        ignore_cms=ignore_cms,
        disable_admin_requests=disable_admin_requests,
        cms_task_action=cms_task_action,
        operation_state=operation_state,
    )


def change_host_maintenance(
    issuer,
    host,
    timeout_time=None,
    timeout_status=None,
    ticket_key=None,
    operation_state=None,
    reason=None,
    remove_timeout=False,
):
    if timeout_time is not None and remove_timeout:
        raise RequestValidationError("Cannot set and remove timeout at the same time.")

    maintenance_params = prepare_maintenance_params(
        timeout_time, timeout_status, ticket_key, operation_state, is_new_maintenance=False
    )
    timeout_time, timeout_status, ticket_key, operation_state = maintenance_params

    if not ticket_key:
        ticket_key = host.state_expire.ticket if host.state_expire else host.ticket

    if not ticket_key:
        raise RequestValidationError("Ticket key is required for maintenance.")

    if not timeout_status:
        timeout_status = host.state_expire.status

    if remove_timeout:
        timeout_time = None
    elif not timeout_time:
        timeout_time = host.state_expire.time

    state_expire = StateExpire(
        ticket=ticket_key,
        status=timeout_status,
        time=timeout_time,
        issuer=issuer,
    )

    host_state_kwargs = Host.set_state_kwargs(
        host.state, host.state_author, host.state_audit_log_id, expire=state_expire, reason=reason
    )

    # try to keep host's current ticket key here
    host_state_kwargs["set__ticket"] = host.ticket or ticket_key

    if operation_state:
        host_state_kwargs["set__operation_state"] = operation_state

    if not host.modify(state=HostState.MAINTENANCE, state_audit_log_id=host.state_audit_log_id, **host_state_kwargs):
        raise ResourceConflictError
