import re
from string import Formatter

from sepelib.core.constants import MINUTE_SECONDS
from sepelib.yandex.startrek import StartrekConnectionError, StartrekRequestError
from walle.clients import startrek, racktables, bot, abc
from walle.clients.cauth import CauthFlowType
from walle.clients.utils import retry
from walle.errors import RequestValidationError, ResourceConflictError, ApiError, BadRequestError
from walle.host_status import MIN_STATE_TIMEOUT, MAX_STATE_TIMEOUT
from walle.hosts import HostOperationState, HostStatus
from walle.models import timestamp
from walle.util.text import time_range_message, enumeration_join


def validate_operation_state(operation_state):
    if operation_state is None:
        return HostOperationState.OPERATION

    if operation_state not in [HostOperationState.OPERATION, HostOperationState.DECOMMISSIONED]:
        raise ResourceConflictError(
            "Can't set host operation state to {}. It must be one of the following statuses: {}.",
            operation_state,
            ", ".join([HostOperationState.OPERATION, HostOperationState.DECOMMISSIONED]),
        )

    return operation_state


def validate_state_timeout_time(timeout_time):
    if timeout_time is None:
        return timeout_time

    network_delay_offset = 5 * MINUTE_SECONDS
    minimum = timestamp() + MIN_STATE_TIMEOUT - network_delay_offset
    maximum = timestamp() + MAX_STATE_TIMEOUT + network_delay_offset

    if not minimum <= timeout_time <= maximum:
        raise RequestValidationError(
            "Status timeout must fall into a range {} from current time.",
            time_range_message(MIN_STATE_TIMEOUT, MAX_STATE_TIMEOUT),
        )

    return timeout_time


def validate_timeout_target_status(timeout_status):
    if timeout_status is None:
        return HostStatus.READY

    if timeout_status not in HostStatus.ALL_TIMEOUT:
        raise ResourceConflictError(
            "Can't set host timeout status to {}. It must be one of the following statuses: {}.",
            timeout_status,
            ", ".join(HostStatus.ALL_TIMEOUT),
        )

    return timeout_status


@retry(interval=1, backoff=5, exceptions=(StartrekConnectionError,))
def _get_startrek_ticket(ticket_key):
    client = startrek.get_client()
    return client.get_issue(ticket_key)


def validate_startrek_ticket_key(ticket_key, check_closed=False):
    if ticket_key is None:
        return ticket_key

    # Matches Startrek ticket key or URL in different formats (st/, st.yandex-team.ru, http(s)://st.yandex-team.ru)
    match = re.match(r"((https?://)?st(\.yandex-team\.ru/|//?)?)?(?P<ticket_key>[a-zA-Z]+-\d+)/?", ticket_key)

    if not match:
        raise RequestValidationError("Invalid ticket format. You need to specify the ticket key or URL to ticket.")

    ticket_key = match.group("ticket_key").upper()

    if not check_closed:
        return ticket_key

    try:
        ticket = _get_startrek_ticket(ticket_key)
    except StartrekRequestError as e:
        messages = {
            403: "Wall-E does not have permission to read the ticket. Try adding robot-walle@ to followers",
            404: "specified ticket does not exist",
        }
        message = messages.get(e.response.status_code, str(e))
        raise RequestValidationError("Cannot use the ticket for maintenance: {}.".format(message))
    except StartrekConnectionError:
        # TODO Review the HTTP code that is being returned if ST is failing
        raise RequestValidationError(
            "Cannot use the ticket for maintenance: Startrek is not responding. "
            "Please use timeout if you want to set maintenance immediately."
        )
    except Exception as e:
        raise RequestValidationError(
            "Cannot use the ticket for maintenance, unknown Startrek error: {}.".format(str(e))
        )

    if ticket["status"]["key"] == startrek.TicketStatus.CLOSED:
        raise ResourceConflictError("Cannot use the ticket for maintenance: the ticket is closed.")

    return ticket_key


def prepare_maintenance_params(
    timeout_time=None, timeout_status=None, ticket_key=None, operation_state=None, is_new_maintenance=True, **kwargs
):
    if is_new_maintenance or timeout_time is not None:
        timeout_time = validate_state_timeout_time(timeout_time)

    if is_new_maintenance or timeout_status is not None:
        timeout_status = validate_timeout_target_status(timeout_status)

    if is_new_maintenance or ticket_key is not None:
        # Check the ticket status only if we want to use "closed" status to leave maintenance
        ticket_key = validate_startrek_ticket_key(ticket_key, check_closed=timeout_time is None)

    if is_new_maintenance or operation_state is not None:
        operation_state = validate_operation_state(operation_state)

    if is_new_maintenance and not ticket_key:
        raise RequestValidationError("You must specify ticket key for exiting maintenance.")

    return timeout_time, timeout_status, ticket_key, operation_state


def validated_hbf_project_id(hbf_project_id_string):
    try:
        hbf_project_id = int(hbf_project_id_string, 16)
    except ValueError:
        raise RequestValidationError("Invalid HBF project id: {}", hbf_project_id_string)

    try:
        projects_map = racktables.get_hbf_projects()
    except racktables.RequestException as e:
        raise ApiError("Can not validate HBF project id: {}", e)

    if hbf_project_id in projects_map:
        return hbf_project_id
    else:
        raise RequestValidationError("Unknown HBF project id: {}", hbf_project_id_string)


def validated_bot_project_id(bot_project_id):
    # WALLE-4583
    allowed_bot_project, not_allowed_message = bot.is_allowed_bot_project(bot_project_id)
    if not allowed_bot_project:
        raise BadRequestError(not_allowed_message)

    if not bot.is_valid_oebs_project(bot_project_id):
        raise BadRequestError("Invalid bot project id: {}", bot_project_id)
    return bot_project_id


def check_redundant_tvm_app_id(cms_settings):
    if not cms_settings.supports_tvm() and cms_settings.tvm_app_id is not None:
        raise BadRequestError("CMS doesn't support TVM, but tvm app id was passed")


def check_cms_tvm_app_id_requirements(cms_settings, bot_project_id):
    if cms_settings.supports_tvm():
        if cms_settings.tvm_app_id is None:
            raise BadRequestError(
                "CMS TVM app id must be set for non-default CMSes. If your CMS does not support TVM "
                "yet, please contact Wall-e administrators"
            )
        else:
            service_planner_id = bot.get_planner_id_by_bot_project_id(bot_project_id)
            if cms_settings.tvm_app_id not in abc.get_service_tvm_app_ids(service_planner_id):
                service_slug = abc.get_service_slug(service_planner_id)
                raise BadRequestError(
                    "CMS TVM app id {} is not registered in ABC service {}".format(
                        cms_settings.tvm_app_id, service_slug
                    )
                )


def validated_host_shortname_template(template):
    placeholders = _parse_template_placeholders(template)
    allowed_placeholders = {"index", "bucket", "location"}

    if placeholders.keys() - allowed_placeholders:
        forbidden = sorted(placeholders.keys() - allowed_placeholders)
        raise RequestValidationError(
            "Only '{{index}}', '{{bucket}}' and '{{location}}' placeholders allowed in host shortname template. "
            "{} {} not allowed.",
            enumeration_join(forbidden),
            "are" if len(forbidden) > 1 else "is",
        )

    if "index" not in placeholders:
        raise RequestValidationError("'{index}' is a required placeholder in host shortname template.")

    if "bucket" in placeholders and not placeholders["index"]:
        raise RequestValidationError("'{index}' must be fixed in width when using buckets in host shortname template.")

    # validate domain requirements
    sample = template.format(location="loc", bucket=0, index=0)
    if not re.match(r"^[a-z0-9]+(?:-[a-z0-9]+)*$", sample):
        raise RequestValidationError("host shortname template does not describe a valid host shortname.")

    return template


def _parse_template_placeholders(template):
    res = {}
    for _, field_name, format_spec, _ in Formatter().parse(template):
        if field_name is not None:
            res[field_name] = format_spec
    return res


def check_cauth_settings(flow_type, trusted_sources):
    if flow_type == CauthFlowType.BACKEND_SOURCES and not trusted_sources:
        raise RequestValidationError("CAuth trusted sources must be set for flow_type {}".format(flow_type))
    return flow_type, trusted_sources
