"""Metrics"""  # Will be used as category name in API reference

import functools
import itertools
import logging
import time

import simplejson as json
from flask import request, make_response

import walle.metrics
from sepelib.core.constants import MINUTE_SECONDS
from walle.application import app
from walle.authorization import iam
from walle.expert import automation
from walle.expert.automation_plot import AutomationPlot
from walle.projects import Project
from walle.stats import stats_manager
from walle.util.api import api_handler
from walle.util.mongo import MongoDocument

log = logging.getLogger(__name__)


_YASM_METRICS_OUTDATE_TIMEOUT = 5 * MINUTE_SECONDS
"""Yasm collects metrics every 5 seconds. We collect data for our metrics using mongo aggregated queries,
which is kinda-sorta slow and often takes more the 5 seconds to refresh. In the mean time we respond with old
data but only for a short period of time, and after a certain timeout we start responding with no data."""

_YASM_QUERY_SCHEME = {
    "metrics_limit_exceeded": {"type": "integer", "description": "Golovan stat handle parameter"},
    "invalid_response": {"type": "boolean", "description": "Golovan stat handle parameter"},
    "format": {
        "enum": ["yasm", "solomon"],
        "description": "Output format. "
        "format=yasm enables Golovan stat handle mode "
        "(https://wiki.yandex-team.ru/golovan/userdocs/stat-handle/).",
    },
}


def unistat_handler(path):
    def _decorator(handler):
        @functools.wraps(handler)
        @api_handler(
            path,
            "GET",
            params=_YASM_QUERY_SCHEME,
            blueprint=app.metrics_blueprint,
            # open this permission for Unified Agent
            iam_permissions=iam.AnyoneApiIamPermission(),
        )
        def new_handler(query_args):
            if query_args.get("invalid_response"):
                log.error("Golovan reports an invalid response for %s.", request.path)

            if "metrics_limit_exceeded" in query_args:
                log.error(
                    "Golovan reports activation of metrics limits for %s. Only %s metrics has been read.",
                    request.path,
                    query_args["metrics_limit_exceeded"],
                )

            format = query_args.get("format", "yasm")
            data, update_time = handler(format)

            if update_time is not None and time.time() - update_time > _YASM_METRICS_OUTDATE_TIMEOUT:
                log.error("%s metrics are outdated.", request.path)
                data = []

            response = make_response(json.dumps(data), 200)
            response.mimetype = "application/json"

            return response

        return new_handler

    return _decorator


@unistat_handler("/automation")
def get_automation_metrics(format):
    """Returns automation status metrics."""

    metrics = {
        "healing_enabled": automation.GLOBAL_HEALING_AUTOMATION.is_enabled(),
        "dns_enabled": automation.GLOBAL_DNS_AUTOMATION.is_enabled(),
        "projects": _get_project_automation_status(),
        "automation_plots": _get_automation_plot_checks_status(),
    }

    if format == "yasm":
        metrics = list(
            itertools.chain(
                # Global status must go first to decrease probability of being truncated
                walle.metrics.get_yasm_metrics(
                    {"healing_enabled": metrics["healing_enabled"], "dns_enabled": metrics["dns_enabled"]}
                ),
                itertools.chain.from_iterable(
                    walle.metrics.get_yasm_metrics(project_metrics, prefix="project_" + project)
                    for project, project_metrics in metrics["projects"].items()
                ),
                itertools.chain.from_iterable(
                    itertools.chain.from_iterable(
                        walle.metrics.get_yasm_metrics(
                            check_metrics, prefix="automation_plot_{}_{}_".format(plot, check)
                        )
                        for check, check_metrics in checks.items()
                    )
                    for plot, checks in metrics["automation_plots"].items()
                ),
            )
        )
        return metrics, None

    # TODO: solomon format
    return None, None


@unistat_handler("/hosts")
def get_host_metrics(format):
    """Returns host metrics."""

    return walle.metrics.get("hosts", format)


@unistat_handler("/tasks")
def get_task_metrics(format):
    """Returns task metrics."""

    return walle.metrics.get("tasks", format)


@unistat_handler("/health")
def get_health_metrics(format):
    """Returns task metrics."""

    return walle.metrics.get("health", format)


@unistat_handler("/stats")
def get_stats(format):
    """Returns daemon metrics."""

    if format == "yasm":
        return stats_manager.to_yasm_format(), None
    # TODO: stats_manager.to_solomon_format()
    return None, None


def _get_project_automation_status():
    pd = MongoDocument.for_model(Project)
    projects = pd.find({}, ["healing_automation.enabled", "dns_automation.enabled"])

    return {
        project.id: {
            "healing_automation_enabled": project.healing_automation.enabled,
            "dns_automation_enabled": project.dns_automation.enabled,
        }
        for project in projects
    }


def _get_automation_plot_checks_status():
    apd = MongoDocument.for_model(AutomationPlot)

    return {
        plot.id: {check["name"]: {"healing_automation_enabled": check["enabled"]} for check in plot.checks}
        for plot in apd.find({}, ["checks"])
    }
