import * as cypress from 'cypress';
import * as dateFormat from 'dateformat';
import * as glob from 'glob';
import { merge } from 'mochawesome-merge';
import * as marge from 'mochawesome-report-generator';
import { default as fetch } from 'node-fetch';
import * as os from 'os';
import * as rimraf from 'rimraf';
import * as parallelLimit from 'run-parallel-limit';
import { promisify } from 'util';

import { distributeTasksBetweenProcesses } from './utils/distributeTasksBetweenProcesses';
import CypressFailedRunResult = CypressCommandLine.CypressFailedRunResult;
import CypressRunResult = CypressCommandLine.CypressRunResult;

const parallelLimitPromisified = promisify(parallelLimit);
const globPromisified = promisify(glob);
const rimrafPromisified = promisify(rimraf);

// const TEST_DIR = './integration_check_parallel/';
const TEST_DIR = './integration/';

const LATEST_REPORT = 'https://infracloudui-cdn.s3.mds.yandex.net/deploy-e2e-reports/latest.json';

async function getTestFiles(): Promise<string[]> {
   return await globPromisified(`${TEST_DIR}**/*.ts`);
}

async function cypressRun(spec: string, taskName: string) {
   return await cypress.run({
      reporter: 'mochawesome',
      reporterOptions: {
         reportDir: `./results/json/`,
         reportFilename: `${taskName}.json`,
         overwrite: false,
         html: false,
      },
      spec,
      quiet: true,
      env: {
         ROBOT_LOGIN: process.env.ROBOT_LOGIN,
         ROBOT_PASSWORD: process.env.ROBOT_PASSWORD,
      },
   });
}

type ParallelTaskCallback = (err: CypressFailedRunResult | null, result: CypressRunResult | null) => void;

type ParallelTask = (cb: ParallelTaskCallback) => void;

async function prepare() {
   await rimrafPromisified('./results/json/');
   await rimrafPromisified('./results/html/');
}

async function getTestDistributed(actualSpec: string[], processLimit: number) {
   const response = await fetch(LATEST_REPORT);
   const report = await response.json();

   const runResult = report.results.map(r => {
      const duration = r.suites.reduce((acc, s) => acc + s.duration, 0);

      return { name: './' + r.file, duration };
   });

   return distributeTasksBetweenProcesses({
      processLimit,
      actualTasks: actualSpec,
      preparationDuration: 10000,
      tasks: runResult,
   });
}

async function runTests() {
   const specFiles = await getTestFiles();
   const distributionResult = await getTestDistributed(specFiles, os.cpus().length);

   console.table(distributionResult.map(r => ({ duration: r.duration, tasks: r.tasks.map(t => t.name) })));
   console.log(
      'Estimated duration:',
      distributionResult.reduce((acc, r) => Math.max(acc, r.duration), 0),
   );

   const tasks: ParallelTask[] = distributionResult.map(({ tasks }, i) => cb => {
      const specLine = tasks.map(t => t.name).join(',');
      console.log(`Run tests from ${specLine}`);

      cypressRun(specLine, `run_${i}`).then(
         (results: CypressRunResult) => {
            cb(null, results);
         },
         (err: CypressFailedRunResult) => {
            cb(err, null);
         },
      );
   });

   return parallelLimitPromisified(tasks, tasks.length);
}

async function generateReports() {
   const report = await merge({
      files: ['./results/json/**/*.json'],
      output: './results/all.json',
   });

   const title = `Deploy E2E report from ${dateFormat(new Date())}`;

   await marge.create(report, {
      reportDir: './results/html',
      saveJson: true,
      // showPassed: false,
      charts: true,
      reportTitle: title,
      reportPageTitle: title,
      reportFilename: 'deploy_e2e_report',
   });
}

async function run() {
   await prepare();

   const start = new Date();
   await runTests();
   const duration = new Date().getTime() - start.getTime();
   console.log('Actual duration: ', duration);

   await generateReports();
   // await publishResults();
}

run().then(
   r => {
      console.log(r);
      process.exit(0);
   },
   e => {
      console.log(e);
      process.exit(1);
   },
);

function wait() {
   setTimeout(wait, 1000);
}

wait();
