interface RunResult {
   duration: number;
   name: string;
}

interface DistributionOptions {
   actualTasks?: string[];
   preparationDuration: number;
   processLimit: number;
   tasks: RunResult[];
}

interface DistributionResult {
   duration: number;
   tasks: RunResult[];
}

export function distributeTasksBetweenProcesses({
                                                   actualTasks = [],
                                                   preparationDuration,
                                                   processLimit,
                                                   tasks,
                                                }: DistributionOptions): DistributionResult[] {
   const processes: DistributionResult[] = new Array(processLimit)
      .fill(null)
      .map(() => ({ duration: preparationDuration, tasks: [] }));

   // actualize task list, if actualTasks passed
   if (actualTasks.length > 0) {
      // remove unused tasks
      const toRemove = tasks.filter(r => !actualTasks.includes(r.name)).map(r => r.name);
      tasks = tasks.filter(r => !toRemove.includes(r.name));

      const totalDuration = tasks.reduce((acc, result) => acc + result.duration, 0);
      const averageDuration = tasks.length > 0 ? totalDuration / tasks.length : 0;

      // add new tasks
      for (const actualName of actualTasks) {
         if (!tasks.some(r => r.name === actualName)) {
            tasks.push({ name: actualName, duration: averageDuration });
         }
      }
   }

   // sort before by duration (from long to short)
   tasks.sort((a, b) => b.duration - a.duration);

   // distribution algorithm
   for (const result of tasks) {
      const processWithMinDuration = processes.reduce((acc, p) =>
         (p.duration === acc.duration ? p.tasks.length < acc.tasks.length : p.duration < acc.duration) ? p : acc,
      );

      processWithMinDuration.tasks.push(result);
      processWithMinDuration.duration += result.duration;
   }

   // skip empty processes
   return processes.filter(p => p.tasks.length > 0);
}
