const fs = require('fs');
const fetch = require('node-fetch');
const pathModule = require('path');
const rimraf = require('rimraf');

const API_HOST = 'https://api.wall-e.yandex-team.ru/v1';

const FIELDS = {
   auditLogItem: [
      'automation_plot',
      'error',
      'host_inv',
      'host_name',
      'id',
      'issuer',
      'payload',
      'project',
      'reason',
      'scenario_id',
      'status',
      'status_time',
      'time',
      'type',
   ],
   automationPlot: ['id', 'name', 'owners', 'checks'],
   host: [
      'active_mac',
      'active_mac_source',
      'active_mac_time',
      'config',
      'deploy_config_policy',
      'deploy_network',
      'deploy_tags',
      'extra_vlans',
      'health.check_statuses',
      'health.decision',
      'health.event_time',
      'health.message',
      'health.reasons',
      'health.status',
      'inv',
      'ipmi_mac',
      'ips',
      'juggler_aggregate_name',
      'location.city',
      'location.country',
      'location.datacenter',
      'location.network_source',
      'location.network_timestamp',
      'location.physical_timestamp',
      'location.port',
      'location.queue',
      'location.rack',
      'location.short_datacenter_name',
      'location.short_queue_name',
      'location.switch',
      'macs',
      'messages',
      'name',
      'operation_state',
      'platform',
      'project',
      'provisioner',
      'restrictions',
      'scenario_id',
      'state',
      'state_audit_log_id',
      'state_author',
      'state_reason',
      'state_time',
      'state_timeout',
      'status',
      'status_audit_log_id',
      'status_author',
      'status_reason',
      'status_time',
      'status_timeout',
      'task.audit_log_id',
      'task.error',
      'task.owner',
      'task.status',
      'task.status_message',
      'ticket',
      'uuid',
   ],
   preorder: [
      'acquired_hosts',
      'deploy_config',
      'errors',
      'failed_hosts',
      'id',
      'issuer',
      'messages',
      'owner',
      'prepare',
      'processed',
      'project',
      'provisioner',
      'restrictions',
   ],
   project: [
      'automation_limits',
      'automation_plot_id',
      'bot_project_id',
      'certificate_deploy',
      'cms',
      'cms_api_version',
      'cms_max_busy_hosts',
      'cms_tvm_app_id',
      'default_host_restrictions',
      'deploy_config',
      'deploy_config_policy',
      'deploy_network',
      'deploy_tags',
      'dns_automation.credit',
      'dns_automation.credit_end_time',
      'dns_automation.enabled',
      'dns_automation.status_message',
      'dns_domain',
      'extra_vlans',
      'fsm_handbrake.audit_log_id',
      'fsm_handbrake.issuer',
      'fsm_handbrake.reason',
      'fsm_handbrake.ticket_key',
      'fsm_handbrake.timeout_time',
      'fsm_handbrake.timestamp',
      'hbf_project_id',
      'healing_automation.credit',
      'healing_automation.credit_end_time',
      'healing_automation.enabled',
      'healing_automation.status_message',
      'id',
      'name',
      'native_vlan',
      'notifications.recipients',
      'owned_vlans',
      'owners',
      'profile',
      'profile_tags',
      'provisioner',
      'reboot_via_ssh',
      'reports',
      'roles',
      'tags',
      'vlan_scheme',
   ],
   scenario: [
      'action_time',
      'creation_time',
      'hosts',
      'issuer',
      'labels',
      'message',
      'name',
      'scenario_id',
      'scenario_type',
      'script_args',
      'status',
      'ticket_key',
   ],
};

const Limiter = require('async-limiter');
const tasks = new Limiter({ concurrency: 4 });

function paramsToQuery(params) {
   return Object.keys(params)
      .filter(p => params[p] !== undefined)
      .map(p => {
         if (p === null) {
            return p;
         }

         const v = params[p];
         if (Array.isArray(v)) {
            return `${p}=${v.join(',')}`;
         }

         return `${p}=${encodeURIComponent(v.toString())}`;
      })
      .join('&');
}

function yellow(text) {
   if (process.env.CI) {
      return text;
   }
   return `\x1b[33m${text}\x1b[0m`;
}

function internalDownload(file, path, params) {
   const savePath = `mock_data/${file}`;
   const url = `${API_HOST}${path}?${paramsToQuery(params)}`;

   console.log(`Download ${yellow(file)} from ${url}`);

   return fetch(url).then(resp => {
      const dest = fs.createWriteStream(savePath);
      resp.body.pipe(dest);

      return resp.json();
   });
}

function download(file, path, params = {}) {
   tasks.push(cb => {
      internalDownload(file, path, params).then(cb, cb);
   });
}

function downloadPages({ count, limit }, file, path, params) {
   const fileBase = pathModule.basename(file, '.json');

   function getPage({ page, cursor }) {
      const loaded = page * limit;
      const newFile = `${fileBase}_${loaded}-${loaded + limit - 1}.json`;

      return internalDownload(newFile, path, { ...params, cursor, limit }).then(resp => {
         if (page === count - 1) {
            return null;
         }

         return getPage({ page: page + 1, cursor: resp.next_cursor });
      });
   }

   tasks.push(cb => {
      getPage({ page: 0, cursor: 0 }).then(cb, cb);
   });
}

// clear before
rimraf.sync('mock_data/*.json');
rimraf.sync('mock_data/*.txt');

// download data
download('audit-log.json', '/audit-log', { fields: FIELDS.auditLogItem, limit: 1000 });
download('automation-plot.json', '/automation-plot', { fields: FIELDS.automationPlot });
download('bot-projects.json', '/bot-projects');
download('constants.json', '/constants');
download('deploy-configs.json', '/deploy-configs');
download('deploy-log.txt', '/hosts/100061004/deploy-log');
download('hbf-projects.json', '/hbf-projects');
download('health-checks.json', '/health-checks', { limit: 10000 });
downloadPages({ count: 3, limit: 10000 }, 'hosts.json', '/hosts', { fields: FIELDS.host });
download('physical-location-tree.json', '/physical-location-tree');
download('preorders.json', '/preorders', { fields: FIELDS.preorder });
download('profile-log.txt', '/hosts/100061004/profile-log');
download('projects.json', '/projects', { fields: FIELDS.project });
download('scenarios.json', '/scenarios', { limit: 100, fields: FIELDS.scenario });
download('scenarios_noc.json', '/maintenance/switch', { limit: 100, fields: FIELDS.scenario });

// finish
tasks.onDone(() => console.log('FINISHED'));
