const helpers = require('./helpers');

const {URL} = require('url');

function applyForItem(item, ...features) {
   const input = {result: [item]};

   return helpers.flow(
      ...features,
   )(input).result[0];
}

function _withFiltersFeature(filterValues, filters) {
   const checks = Object.keys(filters)
      // Фильтр в URL не задан
      .filter(filterName => filterValues.hasOwnProperty(filterName))
      .map(filterName => {
         const filterConfig = filters[filterName];

         // Простой фильтр по значению поля
         if (typeof filterConfig === 'string') {
            return item => {
               const values = filterValues[filterConfig];

               // OR
               return values.has(
                  item[filterConfig]
                     ? item[filterConfig].toString()
                     : undefined,
               );
            };
         }

         // Кастомный фильтр
         return item => filterConfig(filterValues[filterName], item);
      });

   return rawBody => ({
      ...rawBody,
      result: rawBody.result
         .filter(item => checks.every(check => check(item))),
   });
}

function withBodyFilters(body, filters) {
   // Объект, где ключи - названия фильтров, а значения - множество значений из body для него
   const filterValues = Object.keys(body)
      .filter(filterName => filters.hasOwnProperty(filterName))
      .reduce((acc, filterName) => {
         acc[filterName] = new Set(body[filterName]);

         return acc;
      }, {});

   return _withFiltersFeature(filterValues, filters);
}

function withCursorPaginationFeature(pathWithSearch, defaultItems, maxItems, predicate) {
   const url = new URL(`http://localhost/${pathWithSearch}`);

   const cursor = url.searchParams.has('cursor')
      ? parseInt(url.searchParams.get('cursor'), 10)
      : 0;

   let limit = url.searchParams.has('limit')
      ? parseInt(url.searchParams.get('limit'), 10)
      : defaultItems;
   limit = Math.min(limit, maxItems);
   limit = Math.max(limit, 0);

   return rawBody => ({
      ...rawBody,
      result: cursor
         // Берем первые N элементов, удовлетворяющих условию
         ? helpers.takeFirstN(rawBody.result, item => predicate(item, cursor), limit)
         // Без курсора деградирует просто до взятия первых limit элементов
         : rawBody.result.slice(0, limit),
      total: rawBody.result.length,
   });
}

function withFieldsFeature(pathWithSearch, defaultFields) {
   const url = new URL(`http://localhost/${pathWithSearch}`);
   const fields = url.searchParams.has('fields')
      ? url.searchParams.get('fields').split(',')
      : defaultFields;

   return rawBody => ({
      ...rawBody,
      result: rawBody.result.map(item => {
         const result = {};
         for (const field of fields) {
            // Запрос субполя
            if (field.includes('.')) {
               // Одного уровня пока хватает
               const [group, subField] = field.split('.');
               if (item.hasOwnProperty(group)) {
                  result[group] = result[group] || {};
                  result[group][subField] = item[group][subField];
               }
            } else {
               result[field] = item[field];
            }
         }

         return result;
      }),
   });
}

function withPaginationFeature(pathWithSearch, defaultItems, maxItems) {
   const url = new URL(`http://localhost/${pathWithSearch}`);

   const offset = url.searchParams.has('offset')
      ? parseInt(url.searchParams.get('offset'), 10)
      : 0;

   let limit = url.searchParams.has('limit')
      ? parseInt(url.searchParams.get('limit'), 10)
      : defaultItems;
   limit = Math.min(limit, maxItems);
   limit = Math.max(limit, 0);

   return rawBody => ({
      ...rawBody,
      result: rawBody.result.slice(offset, offset + limit),
      total: rawBody.result.length,
   });
}

function withPatcher(patcher) {
   return rawBody => ({
      ...rawBody,
      result: rawBody.result.map(patcher),
   });
}

function withTailBytes(pathWithSearch) {
   const url = new URL(`http://localhost/${pathWithSearch}`);
   const tailBytes = url.searchParams.has('tail_bytes')
      ? parseInt(url.searchParams.get('tail_bytes'), 10)
      : null;

   return rawBody => tailBytes
      ? rawBody.substr(-tailBytes)
      : tailBytes;
}

function withUrlFiltersFeature(pathWithSearch, filters) {
   const url = new URL(`http://localhost/${pathWithSearch}`);

   // Объект, где ключи - названия фильтров, а значения - множество значений из URL для него
   const filterValues = Array.from(url.searchParams.keys())
      .filter(filterName => filters.hasOwnProperty(filterName))
      .reduce((acc, filterName) => {
         acc[filterName] = new Set(url.searchParams.get(filterName).split(','));

         return acc;
      }, {});

   return _withFiltersFeature(filterValues, filters);
}

module.exports = {
   applyForItem,
   withBodyFilters,
   withCursorPaginationFeature,
   withFieldsFeature,
   withPaginationFeature,
   withPatcher,
   withTailBytes,
   withUrlFiltersFeature,
};
