const {
   applyForItem,
   withBodyFilters,
   withCursorPaginationFeature,
   withFieldsFeature,
   withPaginationFeature,
   withPatcher,
   withTailBytes,
   withUrlFiltersFeature,
} = require('./features');

const { deepJsonClone, getFileContent, getUrlPath, flow, sample } = require('./helpers');

function _getUrlPartFromEnd(url, shiftFromEnd = 0) {
   const parts = getUrlPath(url).split('/');
   parts.reverse();

   return parts[shiftFromEnd];
}

module.exports = class MocksService {
   constructor(api) {
      this._currentHostTask = new Map();

      this.api = api;
   }

   init() {
      // noinspection HtmlRequiredLangAttribute,HtmlDeprecatedTag
      this.api.whenGET(
         /\/502$/,
         `<html>
<head><title>502 Bad Gateway</title></head>
<body bgcolor="white">
<center><h1>502 Bad Gateway</h1></center>
<hr><center>nginx/1.12.2</center>
</body>
</html>
<!-- a padding to disable MSIE and Chrome friendly error page -->
<!-- a padding to disable MSIE and Chrome friendly error page -->
<!-- a padding to disable MSIE and Chrome friendly error page -->
<!-- a padding to disable MSIE and Chrome friendly error page -->
<!-- a padding to disable MSIE and Chrome friendly error page -->
<!-- a padding to disable MSIE and Chrome friendly error page -->
`,
         502,
      );

      this._setAuth();
      this._setConstants();
      this._setHosts();
      this._setHostActions();
      this._setProjects();
      this._setProjectActions();
      this._setAuditLog();
      this._setPreorders();
      this._setAutomationPlots();
      this._setScenarios();
   }

   _setAuth() {
      this._userProjects = ['test', 'sandbox-mtn', 'yp-man'];
      this._userPlots = ['wall-e-hw-checks', 'rtc'];
      this._userProjectRoles = { 'yp-man': ['user'], 'sandbox-mtn': ['user'], 'test': ['owner', 'user'] };

      this.api.whenGET(
         /\/user$/,
         {
            admin: false,
            _MOCK_API: true,
            login: 'khoden',
            projects: this._userProjects,
            automation_plots: this._userPlots,
            groups: ['@svc_yt_administration', '@allstaff', '@svc_wall-e', '@svc_wall-e_development', '@yandex'],
            project_roles: this._userProjectRoles,
         },
         200,
         500,
      );

      this.api.whenGET(/\/csrf-token$/, { csrf_token: 'fake-csrf-token' });
   }

   _setConstants() {
      this.constants = require('./mock_data/constants');
      this.physicalLocationTree = require('./mock_data/physical-location-tree');
      this.deployConfigs = require('./mock_data/deploy-configs');
      this.botProjects = require('./mock_data/bot-projects');
      this.hbfProjects = require('./mock_data/hbf-projects');
      this.health = require('./mock_data/health-checks');

      this.api.whenGET(/\/constants/, this.constants);
      this.api.whenGET(/\/config$/, { ctype: 'test' });
      this.api.whenGET(/\/bot-projects$/, this.botProjects);
      this.api.whenGET(/\/hbf-projects$/, this.hbfProjects);
      this.api.whenGET(/\/deploy-configs$/, this.deployConfigs);
      this.api.whenGET(/\/settings$/, {
         disable_dns_automation: true,
         disable_healing_automation: false,
      });

      this.api.whenGET(/\/physical-location-tree/, url => {
         const data = url.includes('?project')
            ? { result: this.physicalLocationTree.result.slice(0, 2) }
            : this.physicalLocationTree;

         return [200, data];
      });

      this.api.whenGET(/\/health-checks$/, url => {
         return [
            200,
            flow(withUrlFiltersFeature(url, { fqdn: 'fqdn' }), withPaginationFeature(url, 100, 10000))(this.health),
         ];
      });
   }

   _setHosts() {
      this.hosts = {
         result: [
            ...require('./mock_data/hosts_0-9999').result,
            ...require('./mock_data/hosts_10000-19999').result,
            ...require('./mock_data/hosts_20000-29999').result,
         ],
      };
      this.deployLog = getFileContent('./mock_data/deploy-log.txt');
      this.profileLog = getFileContent('./mock_data/profile-log.txt');

      const withHostUrlFilters = url => {
         return withUrlFiltersFeature(url, {
            health: (urlValues, item) => {
               const firstValue = Array.from(urlValues)[0]; // TODO several

               return item.health && item.health.check_statuses.hasOwnProperty(firstValue)
                  ? item.health.check_statuses[firstValue] !== 'passed'
                  : false;
            },
            physical_location: (urlValues, item) => {
               const firstValue = Array.from(urlValues)[0]; // TODO several

               if (!item.location) {
                  console.warn('NO LOCATION FOR ', item.name);

                  return false;
               }

               const l = item.location;
               const hostLocation = [l.country, l.city, l.datacenter].join('|');

               return hostLocation.startsWith(firstValue);
            },
            project: 'project',
            restrictions: (urlValues, item) => {
               const firstValue = Array.from(urlValues)[0]; // TODO several

               return item.restrictions ? item.restrictions.some(r => r === firstValue) : false;
            },
            state: 'state',
            status: 'status',
         });
      };

      const defaultHostFields = ['config', 'inv', 'name', 'state', 'status', 'status_author'];

      this.api.whenGET(/\/hosts$/, url => [
         200,
         flow(
            this._withHostTasks(),
            withHostUrlFilters(url),
            withPaginationFeature(url, 100, 10000),
            withFieldsFeature(url, defaultHostFields),
         )(this.hosts),
      ]);

      this.api.whenPOST(/\/get-hosts$/, (url, body) => [
         200,
         flow(
            withBodyFilters(body, {
               invs: (values, item) => (values.size === 0 ? true : values.has(item.inv)),
               uuids: (values, item) => (values.size === 0 ? true : values.has(item.uuid)),
            }),
            withHostUrlFilters(url),
            withPaginationFeature(url, 100, 10000),
            withFieldsFeature(url, defaultHostFields),
            this._withHostTasks(),
         )(this.hosts),
      ]);

      this.api.whenPOST(/\/hosts$/, (url, body) => [201, body]);

      this.api.whenGET(/\/hosts\/([\w\d.-]+?)$/, url => {
         const id = _getUrlPartFromEnd(url);
         if (id === '900906771') {
            return [
               200,
               {
                  'inv': 900906771,
                  // 'juggler_aggregate_name': 'wall-e',
                  // 'location': {
                  //    'network_source': 'eine',
                  //    'network_timestamp': 1516698086,
                  //    'port': 'ge1/0/13',
                  //    'switch': 'sas1-s578',
                  // },
               },
            ];
         }

         const item = this._getHostById(id);
         if (item) {
            return [200, applyForItem(item, this._withHostTasks(), withFieldsFeature(url, defaultHostFields))];
         } else {
            return [404, { errorMessage: `Host #${id} not found` }];
         }
      });

      this.api.whenGET(/\/hosts\/([\w\d.-]+?)\/current-configuration$/, {
         expected_hbf_project_id: '0x604',
         expected_native_vlan: 333,
         expected_vlans: [333, 688, 700, 788],
         hbf_project_id: '0x604',
         hbf_project_id_synced: true,
         native_vlan: 333,
         vlans: [333, 688, 700, 788],
         vlans_synced: true,
      });

      this.api.whenGET(/\/hosts\/([\w\d.-]+?)\/power-status$/, () => [
         200,
         {
            is_powered_on: Math.random() > 0.5,
         },
      ]);

      this.api.whenGET(/\/hosts\/([\w\d.-]+?)\/deploy-log$/, url => [200, withTailBytes(url)(this.deployLog)]);

      this.api.whenGET(/\/hosts\/([\w\d.-]+?)\/profile-log$/, url => [200, withTailBytes(url)(this.profileLog)]);
   }

   _getHostById(id) {
      return this.hosts.result.find(h => h.uuid === id || h.inv.toString() === id || h.name === id);
   }

   _setHostActions() {
      this.api.whenPOST(/\/hosts\/([\w\d.-]+?)$/, (url, body) => {
         const id = _getUrlPartFromEnd(url);

         const item = this._getHostById(id);
         if (!item) {
            return [404, { errorMessage: `Host #${id} not found` }];
         }

         delete body.reason;

         Object.assign(item, body);

         return [200, item];
      });

      this.api.whenDELETE(/\/hosts\/([\w\d.-]+?)$/, url => {
         const id = _getUrlPartFromEnd(url);

         const item = this._getHostById(id);
         if (!item) {
            return [404, { errorMessage: `Host #${id} not found` }];
         }

         this.hosts.result = this.hosts.result.filter(h => h.name !== id && h.inv.toString() !== id);

         return [204, {}];
      });

      // noinspection LongLine
      this.api.whenPOST(
         /\/hosts\/[\da-f]+\/(reboot|redeploy|force-status|profile|switch-project|power-on|power-off|check-dns|prepare|switch-vlans|set-maintenance|set-assigned)$/,
         (url, body) => {
            const action = _getUrlPartFromEnd(url);
            const id = _getUrlPartFromEnd(url, 1);

            const host = this._getHostById(id);
            if (!host) {
               return [404, {}];
            }

            const hasError = Math.random() > 0.9;
            if (hasError) {
               return [
                  400,
                  {
                     errorMessage: 'Error message from mocks',
                  },
               ];
            }

            this._startHostAction(host, action, body.reason);

            return [
               200,
               {
                  inv: host.inv,
                  name: host.name,
                  state: host.state,
                  status: `${action}ing`, // просто хак
                  status_author: 'khoden@',
                  status_reason: body.reason,
                  task: {
                     status: 'pending',
                  },
               },
            ];
         },
      );

      this.api.whenPOST(/\/hosts\/[\da-f]+\/cancel-task$/, (url, body) => {
         const id = _getUrlPartFromEnd(url, 1);

         const host = this._getHostById(id);
         if (!host) {
            return [404, {}];
         }

         this._endHostAction(host, 'cancel-task', body.reason);

         return [
            200,
            {
               inv: host.inv,
               name: host.name,
               state: host.state,
               status: host.status,
               status_author: 'khoden@',
               status_reason: body.reason,
               task: {
                  status: 'cancelled',
               },
            },
         ];
      });

      this.api.whenPOST(/\/hosts\/[\da-f]+\/release-host$/, (url, body) => {
         const id = _getUrlPartFromEnd(url, 1);

         const host = this._getHostById(id);
         if (!host) {
            return [404, {}];
         }

         this._endHostAction(host, 'release-host', body.reason);

         return [
            200,
            {
               inv: host.inv,
               name: host.name,
               state: host.state,
               status: host.status,
               status_author: 'khoden@',
               status_reason: body.reason,
               task: {
                  status: 'cancelled',
               },
            },
         ];
      });

      this.api.whenPUT(
         /\/hosts\/\d+\/deploy_config$/,
         {
            inv: 444,
         },
         200,
      );

      this.api.whenDELETE(
         /\/hosts\/\d+\/deploy_config$/,
         {
            inv: 444,
         },
         200,
      );

      this.api.whenPUT(
         /\/hosts\/\d+\/extra_vlans$/,
         {
            inv: 444,
         },
         200,
      );
   }

   _setProjects() {
      this.projects = require('./mock_data/projects');

      const defaultFields = ['id', 'name', 'tags'];

      const patcher = withPatcher(project => {
         if (project.id === 'test') {
            return {
               ...project,
               // dns_automation: {enabled: true},
               // healing_automation: {enabled: true},
               // reboot_via_ssh: false,
            };
         }

         return project;
      });

      this.api.whenGET(/\/projects$/, url => [
         200,
         flow(patcher, withFieldsFeature(url, defaultFields))(this.projects),
      ]);

      this.api.whenGET(/\/projects\/([\w\d.-]+?)\/requested_owners_with_request_id$/, {
         result: { ddubrava: 14079306 },
      });

      this.api.whenGET(/\/projects\/([\w\d.-]+?)\/revoking_owners_with_request_id$/, {
         result: { khoden: 13850549 },
      });

      // добавление овнера
      this.api.whenPOST(/\/projects\/([\w\d.-]+?)\/owners$/, (url, body) => {
         const id = _getUrlPartFromEnd(url, 1);

         const project = this.projects.result.find(p => p.id === id);
         if (!project) {
            return [404, { errorMessage: `Project ${body.id} not found` }];
         }

         project.owners = Array.from(new Set([...project.owners, ...body.owners]));

         return [200, project];
      });

      // удаление овнера
      this.api.whenDELETE(/\/projects\/([\w\d.-]+?)\/owners$/, (url, body) => {
         const id = _getUrlPartFromEnd(url, 1);

         const project = this.projects.result.find(p => p.id === id);
         if (!project) {
            return [404, { errorMessage: `Project ${body.id} not found` }];
         }

         project.owners = project.owners.filter(o => !body.owners.includes(o));

         return [200, project];
      });

      this.api.whenGET(/\/projects\/([\w\d.-]+?)$/, url => {
         const id = _getUrlPartFromEnd(url);

         const item = this.projects.result.find(p => p.id === id);
         if (item) {
            return [200, applyForItem(item, patcher, withFieldsFeature(url, defaultFields))];
         } else {
            return [404, { errorMessage: `Project '${id}' not found` }];
         }
      });

      // update
      this.api.whenPOST(/\/projects\/([\w\d.-]+?)$/, (url, body) => {
         const project = this.projects.result.find(p => p.id === body.id);
         if (!project) {
            return [404, { errorMessage: `Project ${body.id} not found` }];
         }

         if (body.cms) {
            body.cms_api_version = body.cms.api_version;
            body.cms_max_busy_hosts = body.cms.max_busy_hosts;
            body.cms_tvm_app_id = body.cms.tvm_app_id;
            body.cms = body.cms.url || 'default';
         }

         Object.assign(project, body);

         return [200, project];
      });

      this.api.whenPUT(/\/projects\/([\w\d.-]+?)\/notifications\/recipients$/, (url, body) => {
         const id = _getUrlPartFromEnd(url, 2);
         const project = this.projects.result.find(p => p.id === id);

         if (!project) {
            return [404, { errorMessage: `Project ${id} not found` }];
         }

         const notifs = {
            audit: body.audit,
            bot: body.bot,
            critical: body.critical,
            error: body.error,
            info: body.info,
            warning: body.warning,
         };

         project.notifications.recipients = notifs;

         return [200, notifs];
      });

      // vlan_scheme update
      this.api.whenPUT(/\/projects\/([\w\d.-]+?)\/vlan_scheme$/, (url, body) => {
         const projectId = _getUrlPartFromEnd(url, 1);
         const project = this.projects.result.find(p => p.id === projectId);
         if (!project) {
            return [404, { errorMessage: `Project ${projectId} not found` }];
         }

         const payload = {
            extra_vlans: body.extra_vlans,
            native_vlan: body.native_vlan,
            reason: body.reason,
            scheme: body.scheme,
         };

         Object.assign(project, payload);

         return [200, payload];
      });

      this.api.whenPOST(/\/projects\/([\w\d.-]+?)\/role\/([\w\d.-]+?)\/members$/, (url, body) => {
         const role = _getUrlPartFromEnd(url, 1);
         const projectId = _getUrlPartFromEnd(url, 3);

         const project = this.projects.result.find(p => p.id === projectId);
         if (!project) {
            return [404, { errorMessage: `Project ${projectId} not found` }];
         }

         const roles = project.roles || {};
         roles[role] = [...roles[role], body.member];

         project.roles = roles;

         return [204, {}];
      });

      this.api.whenDELETE(/\/projects\/([\w\d.-]+?)\/role\/([\w\d.-]+?)\/members$/, (url, body) => {
         const role = _getUrlPartFromEnd(url, 1);
         const projectId = _getUrlPartFromEnd(url, 3);

         const project = this.projects.result.find(p => p.id === projectId);
         if (!project) {
            return [404, { errorMessage: `Project ${projectId} not found` }];
         }

         const roles = project.roles || {};
         roles[role] = (roles[role] || []).filter(i => i !== body.member);

         project.roles = roles;

         return [204, {}];
      });
   }

   _setProjectActions() {
      const justCheckProjectExistingHandler = (url, body) => {
         const projectId = _getUrlPartFromEnd(url, 1);
         const project = this.projects.result.find(p => p.id === projectId);
         if (!project) {
            return [404, { errorMessage: `Project ${projectId} not found` }];
         }

         return [200, body];
      };

      // create
      this.api.whenPOST(/\/projects$/, (url, body) => {
         const existProject = this.projects.result.find(p => p.id === body.id);
         if (existProject) {
            return [409, { errorMessage: `Project ${body.id} has already exist` }];
         }

         const newProject = {
            ...body,
            automation_limits: {},
            dns_automation: {
               enabled: body.enable_dns_automation || false,
            },
            healing_automation: {
               enabled: body.enable_healing_automation || false,
            },
         };

         if (body.cms) {
            Object.assign(newProject, {
               cms: body.cms.url,
               cms_api_version: body.cms.api_version,
               cms_max_busy_hosts: body.cms.max_busy_hosts,
               cms_tvm_app_id: body.cms.tvm_app_id,
            });
         } else {
            Object.assign(newProject, {
               cms: 'default',
               cms_max_busy_hosts: 10,
            });
         }

         delete newProject.enable_dns_automation;
         delete newProject.enable_healing_automation;

         if (body.owners && !body.owners.includes('khoden')) {
            newProject.owners.push('khoden');
         }

         if (!body.owners) {
            newProject.owners = ['khoden'];
         }

         this._userProjects.push(newProject.id);
         this.projects.result.push(newProject);

         return [201, newProject];
      });

      // edit
      this.api.whenPATCH(
         /\/projects\/([\w\d.-]+?)$/,
         (url, { enable_dns_automation, enable_healing_automation, cms, ...patch }) => {
            const projectId = _getUrlPartFromEnd(url);
            const existProject = this.projects.result.find(p => p.id === projectId);
            if (!existProject) {
               return [404, { errorMessage: `Project ${projectId} not found` }];
            }

            patch.dns_automation = { enabled: enable_dns_automation };
            patch.healing_automation = { enabled: enable_healing_automation };

            if (cms) {
               patch.cms = cms.url;
               patch.cms_api_version = cms.api_version;
               patch.cms_max_busy_hosts = cms.max_busy_hosts;
               patch.cms_tvm_app_id = cms.tvm_app_id;
            }

            Object.assign(existProject, patch);

            return [200, existProject];
         },
      );

      // clone
      this.api.whenPOST(/\/projects\/clone\/([\w\d.-]+?)$/, (url, body) => {
         const fromProjectId = _getUrlPartFromEnd(url);

         const fromProject = this.projects.result.find(p => p.id === fromProjectId);
         if (!fromProject) {
            return [404, { errorMessage: `Project ${fromProjectId} not found` }];
         }

         const existProject = this.projects.result.find(p => p.id === body.id);
         if (existProject) {
            return [409, { errorMessage: `Project ${body.id} has already exist` }];
         }

         const newProject = {
            ...deepJsonClone(fromProject),
            id: body.id,
            name: body.id,
            owners: [...fromProject.owners, 'khoden'],
         };

         this._userProjects.push(newProject.id);
         this.projects.result.push(newProject);

         return [201, newProject];
      });

      this.api.whenPOST(/\/projects\/([\w\d.-]+?)\/fsm-handbrake$/, justCheckProjectExistingHandler);

      this.api.whenPUT(/\/projects\/([\w\d.-]+?)\/host-profiling-config$/, justCheckProjectExistingHandler);

      this.api.whenPUT(/\/projects\/([\w\d.-]+?)\/host-provisioner-config$/, (url, body) => {
         const projectId = _getUrlPartFromEnd(url, 1);
         const project = this.projects.result.find(p => p.id === projectId);
         if (!project) {
            return [404, { errorMessage: `Project ${projectId} not found` }];
         }

         Object.assign(project, body);

         return [200, body];
      });

      this.api.whenPOST(/\/projects\/([\w\d.-]+?)\/dns_domain$/, (url, body) => {
         const fromProjectId = _getUrlPartFromEnd(url, 1);

         const existProject = this.projects.result.find(p => p.id === fromProjectId);
         if (!existProject) {
            return [404, { errorMessage: `Project ${fromProjectId} not found` }];
         }

         if (Math.random() > 0.5) {
            return [
               400,
               {
                  errors: [],
                  message: `DNS domain ${body.dns_domain} is not in certficator white list`,
                  result: 'FAIL',
               },
            ];
         }

         existProject.dns_domain = body.dns_domain;

         return [200, body];
      });

      this.api.whenDELETE(/\/projects\/([\w\d.-]+?)\/dns_domain$/, (url, body) => {
         const fromProjectId = _getUrlPartFromEnd(url, 1);

         const existProject = this.projects.result.find(p => p.id === fromProjectId);
         if (!existProject) {
            return [404, { errorMessage: `Project ${fromProjectId} not found` }];
         }

         delete existProject.dns_domain;

         return [200, body];
      });

      this.api.whenPUT(/\/projects\/([\w\d.-]+?)\/enable_automation\/(dns|healing)$/, (url, body) => {
         const projectId = _getUrlPartFromEnd(url, 2);
         const project = this.projects.result.find(p => p.id === projectId);
         if (!project) {
            return [404, { errorMessage: `Project ${projectId} not found` }];
         }

         const type = _getUrlPartFromEnd(url);

         delete body.credit.time;

         project[`${type}_automation`] = {
            credit: body.credit,
            credit_end_time: 1586171505,
            enabled: true,
            status_message: body.reason,
         };

         return [200, body];
      });

      this.api.whenDELETE(/\/projects\/([\w\d.-]+?)\/enable_automation\/(dns|healing)$/, (url, body) => {
         const projectId = _getUrlPartFromEnd(url, 2);
         const project = this.projects.result.find(p => p.id === projectId);
         if (!project) {
            return [404, { errorMessage: `Project ${projectId} not found` }];
         }

         const type = _getUrlPartFromEnd(url);

         project[`${type}_automation`] = { 'enabled': false };

         return [200, body];
      });

      this.api.whenDELETE(/\/projects\/([\w\d.-]+?)\/fsm-handbrake$/, justCheckProjectExistingHandler);

      this.api.whenPOST(/\/projects\/([\w\d.-]+?)\/reports$/, (url, body) => {
         const projectId = _getUrlPartFromEnd(url, 1);
         const projectInd = this.projects.result.findIndex(p => p.id === projectId);
         if (projectInd === -1) {
            return [404, { errorMessage: `Project ${projectId} not found` }];
         }

         this.projects.result[projectInd] = { ...this.projects.result[projectInd], reports: body };

         return [200, {}];
      });

      this.api.whenPUT(/\/projects\/([\w\d.-]+?)\/owners$/, (url, body) => {
         const projectId = _getUrlPartFromEnd(url, 1);
         const projectInd = this.projects.result.findIndex(p => p.id === projectId);
         if (projectInd === -1) {
            return [404, { errorMessage: `Project ${projectId} not found` }];
         }

         this.projects.result[projectInd] = {
            ...this.projects.result[projectInd],
            owners: body.owners,
         };

         return [200, {}];
      });

      this.api.whenPUT(/\/projects\/([\w\d.-]+?)\/rebooting_via_ssh$/, (url, body) => {
         const projectId = _getUrlPartFromEnd(url, 1);
         const project = this.projects.result.find(p => p.id === projectId);
         if (!project) {
            return [404, { errorMessage: `Project ${projectId} not found` }];
         }

         project.reboot_via_ssh = true;

         return [200, body];
      });

      this.api.whenDELETE(/\/projects\/([\w\d.-]+?)\/rebooting_via_ssh$/, (url, body) => {
         const projectId = _getUrlPartFromEnd(url, 1);
         const project = this.projects.result.find(p => p.id === projectId);
         if (!project) {
            return [404, { errorMessage: `Project ${projectId} not found` }];
         }

         project.reboot_via_ssh = false;

         return [200, body];
      });

      // remove
      this.api.whenDELETE(/\/projects\/([\w\d.-]+?)$/, url => {
         const projectId = _getUrlPartFromEnd(url);

         const project = this.projects.result.find(p => p.id === projectId);
         if (!project) {
            return [404, { errorMessage: `Project ${projectId} not found` }];
         }

         this.projects.result = this.projects.result.filter(p => p.id !== projectId);

         return [204, {}];
      });
   }

   _setAuditLog() {
      this.auditLog = require('./mock_data/audit-log');

      const defaultAuditLogFields = [
         'host_inv',
         'host_name',
         'issuer',
         'reason',
         'status',
         'status_time',
         'time',
         'type',
      ];

      this.api.whenGET(/\/audit-log$/, url => [
         200,
         flow(
            withUrlFiltersFeature(url, {
               host_inv: 'host_inv',
               host_name: 'host_name',
               status: 'status',
               type: 'type',
               issuer: 'issuer',
               end_time: (urlValues, item) => {
                  const endTime = parseInt(Array.from(urlValues)[0]);

                  return item.time < endTime;
               },
               start_time: (urlValues, item) => {
                  const startTime = parseInt(Array.from(urlValues)[0]);

                  return item.time > startTime;
               },
            }),
            withCursorPaginationFeature(url, 100, 1000, (item, cursor) => item.time < cursor),
            withFieldsFeature(url, defaultAuditLogFields),
         )(this.auditLog),
      ]);

      this.api.whenGET(/\/audit-log\/([\w\d.-]+?)$/, url => {
         const id = _getUrlPartFromEnd(url);

         const item = this.auditLog.result.find(h => h.id === id);
         if (!item) {
            return [404, { errorMessage: `Audit log '${id}' not found` }];
         }

         return [200, item];
      });
   }

   _setPreorders() {
      this.preorders = require('./mock_data/preorders');

      const defaultFields = ['id', 'issuer', 'owner', 'prepare', 'processed', 'project'];

      this.api.whenGET(/\/preorders$/, url => [200, flow(withFieldsFeature(url, defaultFields))(this.preorders)]);

      this.api.whenGET(/\/preorders\/([\w\d.-]+?)$/, url => {
         const id = _getUrlPartFromEnd(url);

         const item = this.preorders.result.find(h => h.id.toString() === id);
         if (item) {
            return [200, applyForItem(item, withFieldsFeature(url, defaultFields))];
         } else {
            return [404, { errorMessage: `Preorder '${id}' not found` }];
         }
      });

      this.api.whenPOST(/\/preorders\/\d+\/restart$/, {});

      this.api.whenPOST(/\/preorders$/, (url, body) => {
         const existPreorder = this.preorders.result.find(p => p.id === body.id);
         if (existPreorder) {
            return [409, { errorMessage: `Preorder ${body.id} has already exist` }];
         }

         this.preorders.result.push(body);

         return [201, body];
      });
   }

   _startHostAction(host, action, reason) {
      console.log('start host action', host.name || host.inv, action);

      this._currentHostTask.set(host.uuid, {
         status: `${action}ing`, // просто хак
         status_author: 'khoden@',
         status_reason: reason,
         task: {
            status: 'pending',
         },
      });

      setTimeout(() => this._endHostAction(host, action), 10 * 1000);
   }

   _endHostAction(host, action) {
      this._currentHostTask.delete(host.uuid);

      switch (action) {
         case 'profile':
            host.name += ':profiled';
            break;

         case 'switch-project':
            host.project = 'test';
            break;

         case 'release-host':
            host.project = 'release-host';
            break;

         default:
            break;
      }

      host.status = host.status.replace(/ing$/, '');

      console.log('end host action', host.name || host.inv, action);
   }

   _withHostTasks() {
      return rawBody => ({
         ...rawBody,
         result: rawBody.result.map(host => {
            const patch = this._currentHostTask.get(host.uuid);

            return patch ? { ...host, ...patch } : host;
         }),
      });
   }

   _setAutomationPlots() {
      this.automationPlots = require('./mock_data/automation-plot');

      const defaultFields = ['id', 'name'];

      this.api.whenGET(/\/automation-plot\/$/, url => [
         200,
         flow(withFieldsFeature(url, defaultFields))(this.automationPlots),
      ]);

      this.api.whenGET(/\/automation-plot\/([\w\d.-]+?)$/, url => {
         const id = _getUrlPartFromEnd(url);

         const item = this.automationPlots.result.find(h => h.id === id);
         if (item) {
            return [200, applyForItem(item, withFieldsFeature(url, defaultFields))];
         } else {
            return [404, { errorMessage: `Automation plot '${id}' not found` }];
         }
      });

      // create
      this.api.whenPOST(/\/automation-plot\/$/, (url, body) => {
         const item = this.automationPlots.result.find(p => p.id === body.id);
         if (item) {
            return [409, { errorMessage: `Automation plot '${body.id}' has already exist` }];
         }

         if (body.owners && !body.owners.includes('khoden')) {
            body.owners.push('khoden');
         }

         this.automationPlots.result.push(body);

         return [201, body];
      });

      this.api.whenPOST(/\/automation-plot\/([\w\d.-]+?)$/, (url, body) => {
         const id = _getUrlPartFromEnd(url);

         const item = this.automationPlots.result.find(h => h.id === id);
         if (item) {
            Object.assign(item, body);

            return [200, applyForItem(item, withFieldsFeature(url, defaultFields))];
         } else {
            return [404, { errorMessage: `Automation plot '${id}' not found` }];
         }
      });

      // remove
      this.api.whenDELETE(/\/automation-plot\/([\w\d.-]+?)$/, url => {
         const id = _getUrlPartFromEnd(url);

         const item = this.automationPlots.result.find(p => p.id === id);
         if (!item) {
            return [404, { errorMessage: `Automation plot ${id} not found` }];
         }

         this.automationPlots.result = this.automationPlots.result.filter(p => p.id !== id);

         return [204, {}];
      });
   }

   _setScenarios() {
      const scenarios = require('./mock_data/scenarios');
      this.scenarios_noc = require('./mock_data/scenarios_noc');

      this.scenarios = scenarios.result
         ? { result: scenarios.result.sort((a, b) => b.creation_time - a.creation_time) }
         : scenarios;

      const withScenarioUrlFilters = url => {
         return withUrlFiltersFeature(url, {
            name: (urlValues, item) => {
               const name = Array.from(urlValues)[0];

               return item.name.toLowerCase().includes(name.toLowerCase());
            },
            issuer: (urlValues, item) => {
               const issuer = Array.from(urlValues)[0];

               // TODO магия с @
               return item.issuer.replace(/@$/, '') === issuer.replace(/@$/, '');
            },
            status: (urlValues, item) => {
               const status = Array.from(urlValues)[0];

               return item.status === status;
            },
            scenario_type: (urlValues, item) => {
               const type = Array.from(urlValues)[0];

               return item.scenario_type === type;
            },
         });
      };

      this.api.whenGET(/\/scenarios$/, url => [
         200,
         flow(withScenarioUrlFilters(url), withPaginationFeature(url, 100, 100))(this.scenarios),
      ]);

      this.api.whenGET(/\/maintenance\/switch$/, url => [
         200,
         flow(withScenarioUrlFilters(url), withPaginationFeature(url, 100, 100))(this.scenarios_noc),
      ]);

      const withHosts = count => resp => {
         const hosts = this.hosts.result.slice(0, count).map(item => ({
            inv: item.inv,
            status: sample(['processing', 'finished', 'done', 'queue']),
         }));

         return {
            ...resp,
            result: resp.result.map(item => ({
               ...item,
               hosts: hosts,
            })),
         };
      };

      this.api.whenGET(/\/scenarios\/([\w\d.-]+?)$/, url => {
         const id = parseInt(_getUrlPartFromEnd(url), 10);
         const item = this.scenarios.result.find(item => item.scenario_id === id);

         if (!item) {
            return [404, { errorMessage: `Scenario ${id} not found` }];
         }

         item.message = 'my super-duper message\nsecond line of the message';

         return [200, applyForItem(item, withHosts(200))];
      });

      this.api.whenGET(/\/maintenance\/switch\/([\w\d.-]+?)$/, url => {
         const id = parseInt(_getUrlPartFromEnd(url), 10);
         const item = this.scenarios_noc.result.find(item => item.id === id);

         if (!item) {
            return [404, { errorMessage: `Scenario ${id} not found` }];
         }

         return [200, applyForItem(item, withHosts(200))];
      });

      const createScenario = (url, body) => {
         // TODO hosts
         const newScenario = {
            ...body,
            creation_time: Math.floor(Date.now() / 1000),
            issuer: 'khoden',
            scenario_id: Math.max(...this.scenarios.result.map(i => i.scenario_id)) + 1,
         };

         this.scenarios.result.unshift(newScenario);

         return [201, newScenario];
      };

      // create
      this.api.whenPOST(/\/scenarios$/, createScenario);

      this.api.whenPOST(/\/maintenance\/switch$/, (url, body) => {
         const scenario = {
            ...body,
            action_time: Math.floor(Date.now() / 1000),
            id: Math.max(...this.scenarios.result.map(i => i.scenario_id)) + 1,
            issuer: 'khoden',
         };

         this.scenarios.result.unshift({ ...scenario, scenario_id: scenario.id });
         this.scenarios_noc.result.unshift(scenario);

         return [201, scenario];
      });

      // create, hosts add rtc
      this.api.whenPOST(/\/scenarios\/hosts_add_rtc$/, createScenario);

      // cancel
      this.api.whenPATCH(/\/scenarios\/([\w\d.-]+?)\/start$/, url => {
         const id = parseInt(_getUrlPartFromEnd(url, 1), 10);
         const item = this.scenarios.result.find(p => p.scenario_id === id);
         if (!item) {
            return [404, { errorMessage: `Scenario ${id} not found` }];
         }

         item.status = 'started';

         return [204];
      });

      // cancel
      this.api.whenPATCH(/\/scenarios\/([\w\d.-]+?)\/cancel$/, url => {
         const id = parseInt(_getUrlPartFromEnd(url, 1), 10);
         const item = this.scenarios.result.find(p => p.scenario_id === id);
         if (!item) {
            return [404, { errorMessage: `Scenario ${id} not found` }];
         }

         this.scenarios.result = this.scenarios.result.filter(p => p.scenario_id !== id);

         return [204];
      });
   }
};
