import { ThemeProvider } from '@yandex-data-ui/common';
import {
   ActionContainer,
   IActionStore,
   ModalContainer,
   SystemUsersContext,
   ToastContainer,
   useBehaviourSubject,
} from '@yandex-infracloud-ui/libs';
import { Pointerfocus } from 'lego-on-react';
import * as React from 'react';
import { PropsWithChildren, useEffect } from 'react';
import { useDispatch, useSelector } from 'react-redux';

import {
   automationPlotActions,
   automationPlotActionService,
   AutomationPlotSubject,
} from '../actions/automation_plot_actions';
import {
   maintenancePlotActions,
   maintenancePlotActionService,
   MaintenancePlotSubject,
} from '../actions/maintenance_plot_actions';
import { hostActions, hostActionService, HostSubject } from '../actions/host_actions';
import { globalState, IAutomationPlot, IHost, IMaintenancePlot, SYSTEM_USERS } from '../models';
import { config } from '../services';
import { UserContext } from '../shared';
import { Banner } from '../shared/Banner/Banner';
import { loadUser } from '../state/globals';
import { RootState } from '../state/store';

import styles from './App.module.css';
import { Header } from './Header';

const actionStore: IActionStore = {
   [HostSubject]: {
      actions: hostActions,
      resolveNameFromContext: (hosts: IHost[]) => hosts.map(h => h.name),
      service: hostActionService,
   },
   [AutomationPlotSubject]: {
      actions: automationPlotActions,
      resolveNameFromContext: (plot: IAutomationPlot) => [plot.id],
      service: automationPlotActionService,
   },
   [MaintenancePlotSubject]: {
      actions: maintenancePlotActions,
      resolveNameFromContext: (plot: IMaintenancePlot) => [plot.id],
      service: maintenancePlotActionService,
   },
};

interface IProps {
   isTesting?: boolean;
}

export const App = ({ isTesting = false, children }: PropsWithChildren<IProps>) => {
   const oldUser = useBehaviourSubject(globalState.user);
   const newUser = useSelector((s: RootState) => s.globals.user);
   const dispatch = useDispatch();

   useEffect(() => {
      if (!newUser) {
         dispatch(loadUser());
      }
   }, [dispatch, newUser]);

   return (
      <UserContext.Provider value={newUser || oldUser}>
         <SystemUsersContext.Provider value={SYSTEM_USERS}>
            <ThemeProvider theme={config.theme}>
               <Pointerfocus />

               <div className={styles.app}>
                  <Banner
                     buttonText={'Complete the survey'}
                     id={'survey1'}
                     title={'Help us to make Y.Wall-e better!'}
                     text={'The survey takes only 10 minutes'}
                     url={'https://forms.yandex-team.ru/surveys/25874/'}
                     endDate={new Date('2019-12-01T00:00:00.000Z')}
                  />

                  <Header addInfraBuzzer={!isTesting && config.useInfraBuzzer} user={oldUser} />
                  {oldUser || config.isExternal ? (
                     <div className={styles.pageContent}>{children}</div>
                  ) : (
                     <h1 className={styles.redirect}>Redirection to login...</h1>
                  )}
               </div>

               <ToastContainer />

               <ActionContainer actionStore={actionStore} />

               <ModalContainer />
            </ThemeProvider>
         </SystemUsersContext.Provider>
      </UserContext.Provider>
   );
};
