import { InfraBuzzer, PRODUCTION, store as infraStore } from '@yandex-data-ui/infra-buzzer';
import {
   Avatar,
   FeedbackButton,
   globalHotKeys,
   HotKeysHelpModal,
   styleHelpers,
   useBehaviourSubject,
} from '@yandex-infracloud-ui/libs';
import { Location } from 'history';
import { Button, Icon, Menu, Popup } from 'lego-on-react';
import React, { RefObject, useCallback, useRef, useState } from 'react';
import { Provider } from 'react-redux';
import { match, NavLink, useLocation } from 'react-router-dom';
import { excludedOperationStatuses } from 'rich_shared/scenario_list/ScenarioList/ScenarioList.state';
import { useConstants } from 'shared';

import { EXTERNAL_LINKS, ROUTE_LINKS, UserModel } from '../models';
import { config } from '../services';
import { ApiPresetSwitcher } from './ApiPresetSwitcher';
import { InstallationTypeSwitcher } from './InstallationTypeSwitcher';

import styles from './Header.module.css';
import keyboardSvg from './images/keyboard.svg';
import { MainMenu } from './MainMenu';
import { ThemeSwitcher } from './ThemeSwitcher';

interface IProps {
   addInfraBuzzer?: boolean;
   user: UserModel | null;
}

const _isProjectTab = (match1: match<any>, location: Location): boolean => {
   const path = location.pathname;

   return (
      path.startsWith('/new_project/') ||
      path.startsWith('/projects/') ||
      path.startsWith('/host/') ||
      path.startsWith('/project/')
   );
};

const _getPlotNavLink = (location: Location): JSX.Element => {
   const path = location.pathname;

   if (path.startsWith('/automation_plots')) {
      return (
         <NavLink to={ROUTE_LINKS.automationPlots()} activeClassName={styles.active}>
            Automation Plots
         </NavLink>
      );
   }

   if (path.startsWith('/maintenance_plots')) {
      return (
         <NavLink to={ROUTE_LINKS.maintenancePlots()} activeClassName={styles.active}>
            Maintenance Plots
         </NavLink>
      );
   }

   return (
      <NavLink to={ROUTE_LINKS.automationPlots()} activeClassName={styles.active}>
         Configure
      </NavLink>
   );
};

export const Header = React.memo(({ addInfraBuzzer = true, user }: IProps) => {
   // hooks
   const { constants } = useConstants();
   const location = useLocation();
   const hotkeys = useBehaviourSubject(globalHotKeys.configs);

   const [visible, setVisible] = useState(false);
   const switcherRef = useRef<HTMLSpanElement>();

   const defaultOperationStatuses = React.useMemo(
      () => new Set(constants?.scenario_work_statuses.filter(e => !excludedOperationStatuses.has(e))),
      [constants],
   );

   // handlers
   const toggle = useCallback(() => setVisible(v => !v), []);
   const close = useCallback(() => setVisible(false), []);

   // Первый захардкожен (вызов справки по горячим клавишам), он не считается
   const hasGlobalHotKeys = hotkeys.length > 1;

   // render
   return (
      <header data-e2e={'Header'} className={styles.header}>
         <MainMenu />

         <nav>
            <ul>
               <li>
                  <NavLink to={ROUTE_LINKS.home()} isActive={_isProjectTab} activeClassName={styles.active}>
                     Projects
                  </NavLink>
               </li>

               {config.isExternal ? null : (
                  <li>
                     <NavLink to={ROUTE_LINKS.preorders()} activeClassName={styles.active}>
                        Pre-orders
                     </NavLink>
                  </li>
               )}

               {config.isExternal ? null : (
                  <li>
                     <NavLink to={ROUTE_LINKS.scenarios(defaultOperationStatuses)} activeClassName={styles.active}>
                        Scenarios
                     </NavLink>
                  </li>
               )}

               {config.isExternal ? (
                  <li>
                     <NavLink to={ROUTE_LINKS.automationPlots()} activeClassName={styles.active}>
                        Automation Plots
                     </NavLink>
                  </li>
               ) : (
                  <li>
                     {_getPlotNavLink(location)}

                     {/* TODO: fix button height */}
                     <span ref={switcherRef as RefObject<HTMLSpanElement>}>
                        <Button theme={'link'} size={'xs'} onClick={toggle}>
                           <Icon type={'arrow'} direction={visible ? 'top' : 'bottom'} />
                        </Button>
                     </span>

                     <Popup
                        theme={'normal'}
                        target={'anchor'}
                        directions={['bottom-right']}
                        visible={visible}
                        onOutsideClick={close}
                        anchor={switcherRef.current}
                     >
                        <nav>
                           <Menu theme={'normal'} size={'m'} cls={styles.submenu} onClick={close}>
                              <Menu.Item cls={styles.submenuItem}>
                                 <NavLink to={ROUTE_LINKS.automationPlots()}>Automation Plots</NavLink>
                              </Menu.Item>

                              <Menu.Item cls={styles.submenuItem}>
                                 <NavLink to={ROUTE_LINKS.maintenancePlots()}>Maintenance Plots</NavLink>
                              </Menu.Item>
                           </Menu>
                        </nav>
                     </Popup>
                  </li>
               )}
            </ul>
         </nav>

         {config.isLocalhost ? <ApiPresetSwitcher /> : null}

         {config.isLocalhost ? <InstallationTypeSwitcher /> : null}

         <div className={styles.spacer} />

         <a
            href={EXTERNAL_LINKS.walleDocs()}
            target={'_blank'}
            rel={'noopener noreferrer'}
            className={styleHelpers.externalLink}
         >
            Documentation
         </a>

         {hasGlobalHotKeys ? (
            <Button
               theme={'clear'}
               size={'s'}
               cls={styles.icon}
               icon={<Icon url={keyboardSvg} />}
               onClick={globalHotKeys.toggleHelp}
            />
         ) : null}

         {config.feedbackFormId ? (
            <FeedbackButton formId={config.feedbackFormId} title={'Wall-E bug report or feedback'} />
         ) : null}

         {addInfraBuzzer ? (
            <Provider store={infraStore}>
               <div>
                  <InfraBuzzer subscribeTo={config.infraBuzzer} environment={PRODUCTION} />
               </div>
            </Provider>
         ) : null}

         {config.isLocalhost ? <ThemeSwitcher /> : null}

         {user ? (
            <span className={styles.avatar}>
               <Avatar login={user.login} big={true} />
            </span>
         ) : null}

         <HotKeysHelpModal />
      </header>
   );
});

Header.displayName = 'Header';
