import { ActionButtons, IResult, useDismounted } from '@yandex-infracloud-ui/libs-next';
import * as React from 'react';
import { SyntheticEvent, useCallback, useMemo } from 'react';
import { Observable } from 'rxjs';
import { switchMap, takeUntil } from 'rxjs/operators';
import { IHost } from '../../models';

import { canCallAction, hostActions } from './actions';
import { hostActionService } from './host_action_service';
import { HostAction } from './models';

interface IProps {
   compact: boolean;

   /**
    * Выбранные в списке (галочками) хосты
    *
    * Отличие от getSelectedHosts в том, что при состоянии "выбрать все"
    * тут будет лишь те хосты, которые выбранны чекбоксами, а не те, что будут подгружены позднее.
    */
   currentContext: IHost[];
   disabled?: boolean;
   leftText?: React.ReactNode;
   isShadow?: boolean;

   getSelectedHosts(): Observable<IHost[]>;

   onActionFinished?(action: HostAction, result: IResult[]): void;
}

export const HostActionButtons = React.memo(
   ({
      compact,
      currentContext: hosts,
      disabled = false,
      getSelectedHosts,
      leftText,
      onActionFinished,
      isShadow = false,
   }: IProps) => {
      // hooks
      const dismounted = useDismounted();
      const disabledActions = useMemo(
         () =>
            new Set(
               hostActions.filter(ha => hosts.some(h => !canCallAction(ha.action as HostAction, h))).map(a => a.action),
            ),
         [hosts],
      );

      // handlers
      const _onAction = useCallback(
         (e: SyntheticEvent, action: HostAction) => {
            getSelectedHosts()
               .pipe(
                  switchMap(selected => hostActionService.do(action, selected)),
                  takeUntil(dismounted),
               )
               .subscribe(resultWithAction => {
                  if (onActionFinished) {
                     onActionFinished(action, resultWithAction.results);
                  }
               });
         },
         [dismounted, getSelectedHosts, onActionFinished],
      );

      // render
      return (
         <ActionButtons
            compact={compact}
            disabled={disabled || isShadow}
            leftText={leftText}
            actions={hostActions}
            onAction={_onAction}
            disabledActions={disabledActions}
         />
      );
   },
);

HostActionButtons.displayName = 'HostActionButtons';
