import { FormFieldType, NullableDate, toTimestamp } from '@yandex-infracloud-ui/libs';

import { IHostActionParams, OperationState } from '../../../models';
import { HostAction, HostActionMeta } from '../models';
import { operationStateField, reasonField, ticketKeyField, timeoutStatusField, timeoutTimeField } from './commonFields';
import { BooleanSwitcherField } from '../../../rich_shared/form_fields';

interface IChangeMaintenanceHostParams extends IHostActionParams {
   _timeoutTime: Date | null;

   /**
    * Operation state of the host after switching to maintenance
    */
   operation_state: OperationState;

   /**
    * Clear maintenance state timeout to allow using ticket for leaving maintenance
    */
   remove_timeout: boolean;

   /**
    * Attach ticket to host's manual status.
    */
   ticket_key: string;

   /**
    * Status the host will be switched to at timeout_time
    */
   timeout_status: 'ready' | 'dead';

   /**
    * Time until which the specified status will be active for the host.
    *
    * When the time comes, the host will be automatically switched to the status
    * specified by timeout_status parameter.
    * It's guaranteed that status will be switched not earlier than the specified time,
    * but it's not guaranteed when exactly it will happen.
    * In practice, it will be in minutes since the specified time.
    */
   timeout_time: NullableDate | number;
}

export class ChangeMaintenanceHostAction extends HostActionMeta<IChangeMaintenanceHostParams> {
   public readonly important: boolean = false;

   constructor() {
      super({
         action: HostAction.ChangeMaintenance,
         fields: [
            ticketKeyField,
            timeoutTimeField,
            timeoutStatusField,
            {
               getHelp: () => 'Clear maintenance state timeout to allow using ticket for leaving maintenance',
               label: 'Remove timeout',
               name: 'remove_timeout',
               type: FormFieldType.Custom,
               component: BooleanSwitcherField,
            },
            operationStateField,
            reasonField,

            // Hidden
            { name: 'timeout_time', type: FormFieldType.Hidden },
         ],
         initialValue: {
            _timeoutTime: null,
            ignore_maintenance: false,
            operation_state: OperationState.Operation,
            reason: '',
            remove_timeout: false,
            ticket_key: '',
            timeout_status: 'ready',
            timeout_time: null,
         },
         name: 'Change maintenance',
         titles: ['Change host maintenance', 'Change hosts maintenance'],
      });
   }

   public transform(params: IChangeMaintenanceHostParams): IChangeMaintenanceHostParams {
      params.timeout_time = params._timeoutTime instanceof Date ? toTimestamp(params._timeoutTime) : null;

      return params;
   }
}
