import { clearParams, FormFieldType, IFormField } from '@yandex-infracloud-ui/libs';
import { concat, Observable } from 'rxjs';
import { map, pairwise } from 'rxjs/operators';

import { IEditExtraVlansHostParams, IHost } from '../../../models';
import { createVLANsSelectField } from '../../../rich_shared/form_fields';
import { hostApi } from '../../../services';

import { HostAction, HostActionMeta } from '../models';
import { ignoreMaintenanceField, reasonField } from './commonFields';
import { ISwitchVlansHostParams } from './SwitchVlansHostAction';

export class EditExtraVlansHostAction extends HostActionMeta<IEditExtraVlansHostParams> {
   public readonly important: boolean = false;

   /**
    * Т.к. может отправлятся дополнительный запрос переключения VLAN,
    * то для очистки параметров запроса нужен список полей.
    *
    * Эта форма "виртуальная", она не представлена на странице, формируется в момент отправки запроса.
    */
   private _virtualSwitchFormFields: IFormField<any>[] = [
      ignoreMaintenanceField,
      reasonField,
      { name: 'vlans', type: FormFieldType.Hidden },
   ];

   constructor() {
      super({
         action: HostAction.EditExtraVlans,
         fields: [
            {
               component: createVLANsSelectField({ canSelect: false }),
               label: 'Extra VLANs',
               name: '_vlanConfig',
               type: FormFieldType.Custom,
            },
            {
               checkboxLabel: 'Switch VLANs after completion',
               name: '_switchVlan',
               type: FormFieldType.Boolean,
            },

            reasonField,

            // Options
            ignoreMaintenanceField,

            // Hidden
            { name: 'vlans', type: FormFieldType.Hidden },
         ],
         initialValue: {
            _switchVlan: false,
            ignore_maintenance: false,
            reason: '',
            vlans: new Set(),
         },
         name: 'Edit Extra VLANs',
         titles: ['Edit Extra VLANs for the host', 'Edit Extra VLANs for hosts'],
      });
   }

   public transform(params: IEditExtraVlansHostParams): IEditExtraVlansHostParams {
      if (params._vlanConfig) {
         params.vlans = params._vlanConfig.items;
      }

      return params;
   }

   public do(host: IHost, params: IEditExtraVlansHostParams, context: IHost[]): Observable<any> {
      const cleaned = this.clean(params, context);

      if (!params._switchVlan) {
         return hostApi.changeExtraVlans(host.uuid, cleaned);
      }

      const switchParams = clearParams(
         this._virtualSwitchFormFields,
         {
            ignore_maintenance: params.ignore_maintenance,
            reason: params.reason,
            vlans: params.vlans,
         } as ISwitchVlansHostParams,
         context,
      );

      return concat(
         hostApi.changeExtraVlans(host.uuid, cleaned),
         hostApi.doSimpleAction(host.uuid, HostAction.SwitchVlans, switchParams),
      ).pipe(
         // TODO придумать способ поизящнее для возврата значения только первого Observable
         pairwise(),
         map(r => r[0] as IHost),
      );
   }
}
