import { FormFieldType, NullableDate, toTimestamp } from '@yandex-infracloud-ui/libs';
import {
   CmsTaskAction,
   DefaultCmsTaskAction,
   DefaultOperationState,
   IHostActionParams,
   OperationState,
} from '../../../models';
import { HostAction, HostActionMeta } from '../models';
import {
   cmsTaskActionField,
   disableAdminRequestsField,
   ignoreCmsField,
   ignoreMaintenanceField,
   operationStateField,
   reasonField,
   ticketKeyField,
   timeoutStatusField,
   timeoutTimeField,
} from './commonFields';

interface IPowerOffHostParams extends IHostActionParams {
   _timeoutTime: Date | null;

   /**
    * CMS action name for switching to maintenance.
    */
   cms_task_action: CmsTaskAction;

   /**
    * Don't issue any admin requests if something is broken - just fail the task
    */
   disable_admin_requests: boolean;

   /**
    * Don't acquire permission from CMS
    */
   ignore_cms: boolean; // false by default

   /**
    * Operation state of the host after switching to maintenance
    */
   operation_state: OperationState;

   /**
    * Attach ticket to host's manual status.
    */
   ticket_key: string;

   /**
    * Status the host will be switched to at timeout_time
    */
   timeout_status: 'ready' | 'dead';

   /**
    * Time until which the specified status will be active for the host.
    *
    * When the time comes, the host will be automatically switched to the status
    * specified by timeout_status parameter.
    * It's guaranteed that status will be switched not earlier than the specified time,
    * but it's not guaranteed when exactly it will happen.
    * In practice, it will be in minutes since the specified time.
    */
   timeout_time: NullableDate | number;
}

export class PowerOffHostAction extends HostActionMeta<IPowerOffHostParams> {
   public readonly important: boolean = false;

   constructor() {
      super({
         action: HostAction.PowerOff,
         fields: [
            ticketKeyField,
            timeoutTimeField,
            timeoutStatusField,
            cmsTaskActionField,
            operationStateField,
            reasonField,

            // Options
            disableAdminRequestsField,
            ignoreCmsField,
            ignoreMaintenanceField,

            // Hidden
            { name: 'timeout_time', type: FormFieldType.Hidden },
         ],
         initialValue: {
            _timeoutTime: null,
            cms_task_action: DefaultCmsTaskAction,
            disable_admin_requests: false,
            ignore_cms: false,
            ignore_maintenance: false,
            operation_state: DefaultOperationState,
            reason: '',
            ticket_key: '',
            timeout_status: 'ready',
            timeout_time: null,
         },
         name: 'Power off',
         titles: ['Power off the host', 'Power off the hosts'],
      });
   }

   public transform(params: IPowerOffHostParams): IPowerOffHostParams {
      params.timeout_time = params._timeoutTime instanceof Date ? toTimestamp(params._timeoutTime) : null;

      return params;
   }
}
