import { FormFieldType } from '@yandex-infracloud-ui/libs';

import {
   DefaultDeployConfigPolicy,
   DefaultDeployNetwork,
   DefaultProvisioner,
   DeployNetwork,
   IDeployConfig,
   IHost,
   IHostActionParams,
   Provisioner,
} from '../../../models';
import { ProfileConfigField, RestrictionEditorField, TagInputField, EnumField } from '../../../rich_shared/form_fields';
import { HostAction, HostActionMeta } from '../models';
import {
   checkField,
   deployConfigField,
   deployConfigPolicyField,
   deployNetworkField,
   deployTagsField,
   disableAdminRequestsField,
   ignoreCmsField,
   ignoreMaintenanceField,
   reasonField,
   withAutoHealingField,
} from './commonFields';

interface IPrepareHostParams extends IHostActionParams {
   _deployConfig?: IDeployConfig;
   _keepName: 'keep' | 'notKeep';
   _profileMode: 'profile' | 'skip';
   _restrictionMode: 'default' | 'manual';
   /**
    * Check host after task completion and consider it as failed if host is not healthy
    */
   check: boolean; // true by default

   /**
    * Deploy config
    */
   config: string;

   /**
    * Deploy host either in service or in project VLAN's
    */
   deploy_network: DeployNetwork;

   /**
    * Modifying the configuration of the redeploying host action
    */
   deploy_config_policy: string;

   /**
    * Einstellung tags to apply to host before deployment
    *
    * "pattern": "^[a-zA-Z0-9_:-]+$"
    * "maxItems": 20,
    */
   deploy_tags: Set<string>;

   /**
    * Don't issue any admin requests if something is broken - just fail the task
    */
   disable_admin_requests: boolean; // false by default

   /**
    * Extra VLAN IDs to be assigned to the host
    *
    * "maximum": 4094,
    * "minimum": 1,
    * "type": "integer"
    */
   extra_vlans: Set<number>;

   /**
    * Don't acquire permission from CMS
    */
   ignore_cms: boolean; // false by default

   /**
    * Keep host's current name
    */
   keep_name: boolean; // false by default

   /**
    * Einstellung profile
    */
   profile: string;

   /**
    * Einstellung tags to apply to host before profile assigning
    *
    * "pattern": "^[a-zA-Z0-9_:-]+$"
    * "maxItems": 20,
    */
   profile_tags: Set<string>;

   /**
    * Provisioning system
    */
   provisioner: Provisioner;

   /**
    * Host restrictions
    */
   restrictions: Set<string>;

   /**
    * Skip profiling
    */
   skip_profile: boolean; // false by default

   /**
    * Try to automatically repair the host if task will fail
    */
   with_auto_healing: boolean; // false by default (for API), true for UI
}

export class PrepareHostAction extends HostActionMeta<IPrepareHostParams> {
   public readonly important: boolean = false;

   constructor() {
      super({
         action: HostAction.Prepare,
         fields: [
            // Profile
            {
               label: 'Profiling',
               name: '_profileMode',
               options: [
                  { value: 'profile', title: 'Profile' },
                  { value: 'skip', title: 'Skip profile' },
               ],
               type: FormFieldType.Custom,
               component: EnumField,
            },
            {
               component: ProfileConfigField,
               label: 'Profile config',
               name: 'profile',
               type: FormFieldType.Custom,
            },
            {
               component: TagInputField,
               isDisabled: p => !p.profile,
               isHidden: p => p._profileMode === 'skip',
               name: 'profile_tags',
               type: FormFieldType.Custom,
            },

            // Deploy
            deployConfigField,
            deployNetworkField,
            deployConfigPolicyField,
            deployTagsField,

            {
               label: "Keep host's name",
               name: '_keepName',
               options: [
                  { value: 'keep', title: 'Keep' },
                  { value: 'notKeep', title: 'Not keep' },
               ],
               type: FormFieldType.Custom,
               component: EnumField,
            },

            // Restrictions
            {
               label: 'Restrictions',
               name: '_restrictionMode',
               options: [
                  { value: 'default', title: 'Take from project' },
                  { value: 'manual', title: 'Manual' },
               ],
               type: FormFieldType.Custom,
               component: EnumField,
            },
            {
               component: RestrictionEditorField,
               isHidden: (params: IPrepareHostParams) => params._restrictionMode === 'default',
               name: 'restrictions',
               type: FormFieldType.Custom,
            },

            reasonField,

            // Options
            checkField,
            withAutoHealingField,
            disableAdminRequestsField,
            ignoreCmsField,
            ignoreMaintenanceField,

            // Hidden
            { name: 'config', type: FormFieldType.Hidden },
            { name: 'provisioner', type: FormFieldType.Hidden, isHidden: p => !p.config },
            { name: 'skip_profile', type: FormFieldType.Hidden },
            { name: 'keep_name', type: FormFieldType.Hidden, isHidden: p => !p.keep_name },
         ],
         initialValue: {
            _keepName: 'notKeep',
            _profileMode: 'profile',
            _restrictionMode: 'default',
            check: true,
            config: '',
            deploy_network: DefaultDeployNetwork,
            deploy_config_policy: DefaultDeployConfigPolicy,
            deploy_tags: new Set(),
            disable_admin_requests: false,
            extra_vlans: new Set(), // TODO
            ignore_cms: false,
            ignore_maintenance: false,
            keep_name: false,
            profile: '',
            profile_tags: new Set(),
            provisioner: DefaultProvisioner,
            reason: '',
            restrictions: new Set(),
            skip_profile: false,
            with_auto_healing: true,
         },
         name: 'Prepare',
         titles: ['Prepare the host', 'Prepare hosts'],
      });
   }

   public transform(params: IPrepareHostParams): IPrepareHostParams {
      params.config = params._deployConfig ? params._deployConfig.config : '';
      params.provisioner = params._deployConfig ? params._deployConfig.provisioner : DefaultProvisioner;
      params.skip_profile = params._profileMode === 'skip';
      params.keep_name = params._keepName === 'keep';

      return params;
   }

   /**
    * Очищает параметры перед отправкой на сервер.
    *
    * Увы, clearParams очищает все пустые значения. В большинстве случаев это нормально.
    * Но конкретно в этом случае есть разница между задаванием пустого списка ограничений
    *    и не задаванием ограничений вообще.
    * Поэтому приходится возвращать значение обратно.
    */
   public clean(params: IPrepareHostParams, context: IHost[]): Partial<IPrepareHostParams> {
      const cleaned = super.clean(params, context);
      if (params._restrictionMode === 'manual') {
         cleaned.restrictions = params.restrictions;
      }

      return cleaned;
   }
}
