import { FormFieldType } from '@yandex-infracloud-ui/libs';

import {
   DefaultDeployConfigPolicy,
   DefaultDeployNetwork,
   DefaultProvisioner,
   DeployNetwork,
   IDeployConfig,
   IHostActionParams,
   Provisioner,
} from '../../../models';
import { EnumSelectField, ProfileConfigField, TagInputField, EnumField } from '../../../rich_shared/form_fields';
import { HostAction, HostActionMeta } from '../models';
import {
   advancedField,
   checkField,
   deployConfigField,
   deployConfigPolicyField,
   deployNetworkField,
   deployTagsField,
   disableAdminRequestsField,
   ignoreCmsField,
   ignoreMaintenanceField,
   reasonField,
   withAutoHealingField,
} from './commonFields';

enum ProfileMode {
   FirmwareUpdate = 'firmware-update',
   HighloadTest = 'highload-test',
   ExtraHighloadTest = 'extra-highload-test',
   DangerousHighloadTest = 'dangerous-highload-test',
   DiskRwTest = 'disk-rw-test',
   SwpUp = 'swp-up',
   Default = 'default',
   Custom = 'custom',
}

enum RepairMode {
   High = 'high',
   Medium = 'medium',
   Low = 'low',
}

interface IProfileHostParams extends IHostActionParams {
   _advanced: boolean;
   _deployConfig?: IDeployConfig;
   _mode: 'preset' | 'custom';

   /**
    * Check host after task completion and consider it as failed if host is not healthy
    */

   check: boolean; // true by default
   /**
    * Deploy config
    */

   config: string;

   /**
    * Modifying the configuration of the redeploying host action
    */
   deploy_config_policy: string;

   /**
    * Deploy host either in service or in project VLAN's
    */

   deploy_network: DeployNetwork;

   /**
    * Einstellung tags to apply to host before deployment
    * pattern is ^[a-zA-Z0-9_:-]+$
    * maxItems 20
    */
   deploy_tags: Set<string>;

   /**
    * Don't issue any admin requests if something is broken - just fail the task
    */
   disable_admin_requests: boolean; // false by default

   /**
    * Don't acquire permission from CMS
    */
   ignore_cms: boolean; // false by default

   /**
    * Einstellung profile
    */
   profile: string;

   /**
    * Profile mode
    */
   profile_mode: ProfileMode;

   /**
    * Einstellung tags to apply to host before profile assigning
    *
    * pattern is ^[a-zA-Z0-9_:-]+$
    * maxItems 20
    */
   profile_tags: Set<string>;

   /**
    * Provisioning system
    */
   provisioner: Provisioner;

   /**
    * Redeploy host after profiling
    */
   redeploy: boolean; // false by default

   /**
    * Repair Request Severity for Eine
    */
   repair_request_severity: RepairMode | null; // null by default

   /**
    * Try to automatically repair the host if task will fail
    */
   with_auto_healing: boolean; // false by default (for API), true for UI
}

export class ProfileHostAction extends HostActionMeta<IProfileHostParams> {
   constructor() {
      super({
         action: HostAction.Profile,
         fields: [
            // Profile
            {
               label: 'Profile',
               name: '_mode',
               options: [
                  { value: 'preset', title: 'Preset' },
                  { value: 'custom', title: 'Custom' },
               ],
               type: FormFieldType.Custom,
               component: EnumField,
            },
            {
               component: EnumSelectField,
               isHidden: p => p._mode !== 'preset',
               label: 'Profile preset',
               name: 'profile_mode',
               options: [
                  { value: ProfileMode.Default, title: 'default' },
                  { value: ProfileMode.FirmwareUpdate, title: 'firmware-update' },
                  { value: ProfileMode.HighloadTest, title: 'highload-test' },
                  { value: ProfileMode.ExtraHighloadTest, title: 'extra-highload-test' },
                  { value: ProfileMode.DangerousHighloadTest, title: 'dangerous-highload-test' },
                  { value: ProfileMode.DiskRwTest, title: 'disk-rw-test' },
                  { value: ProfileMode.SwpUp, title: 'swp-up' },
               ],
               type: FormFieldType.Custom,
            },
            {
               component: ProfileConfigField,
               isHidden: p => p._mode !== 'custom',
               label: 'Profile config',
               name: 'profile',
               type: FormFieldType.Custom,
            },
            {
               component: TagInputField,
               isDisabled: p => !p.profile,
               isHidden: p => p._mode !== 'custom',
               label: 'Profile tags',
               name: 'profile_tags',
               type: FormFieldType.Custom,
            },

            // Redeploy
            {
               checkboxLabel: 'Redeploy host after successful profiling',
               name: 'redeploy',
               type: FormFieldType.Boolean,
            },
            {
               ...deployConfigField,
               isHidden: p => !p.redeploy,
            },
            {
               ...deployNetworkField,
               isHidden: p => !p.redeploy || deployNetworkField.isHidden!(p),
            },
            {
               ...deployConfigPolicyField,
               isHidden: p => !p.redeploy,
            },
            {
               ...deployTagsField,
               isHidden: p => !p.redeploy || deployTagsField.isHidden!(p),
            },

            reasonField,

            // Options
            checkField,
            withAutoHealingField,
            disableAdminRequestsField,
            ignoreMaintenanceField,

            // Advanced
            advancedField,
            {
               component: EnumSelectField,
               isHidden: p => !p._advanced,
               label: 'Repair Request Severity',
               name: 'repair_request_severity',
               options: [
                  { value: RepairMode.High, title: 'high' },
                  { value: RepairMode.Medium, title: 'medium' },
                  { value: RepairMode.Low, title: 'low' },
               ],
               type: FormFieldType.Custom,
            },
            {
               ...ignoreCmsField,
               isHidden: p => !p._advanced,
            },

            // Hidden
            { name: 'config', type: FormFieldType.Hidden },
            { name: 'provisioner', type: FormFieldType.Hidden, isHidden: p => !p.config },
         ],
         initialValue: {
            _advanced: false,
            _mode: 'preset',
            profile_mode: ProfileMode.Default,
            check: true,
            config: '',
            deploy_network: DefaultDeployNetwork,
            deploy_config_policy: DefaultDeployConfigPolicy,
            deploy_tags: new Set(),
            disable_admin_requests: false,
            ignore_cms: false,
            ignore_maintenance: false,
            profile: '',
            profile_tags: new Set(),
            provisioner: DefaultProvisioner,
            reason: '',
            redeploy: false,
            repair_request_severity: null,
            with_auto_healing: true,
         },
         name: 'Profile',
         titles: ['Profile the host', 'Profile hosts'],
      });
   }

   public transform(params: IProfileHostParams): IProfileHostParams {
      params.config = params._deployConfig ? params._deployConfig.config : '';
      params.provisioner = params._deployConfig ? params._deployConfig.provisioner : DefaultProvisioner;

      return params;
   }
}
