import { store } from './../../../state/store';
import { FormFieldType } from '@yandex-infracloud-ui/libs';

import { IHost, IHostActionOptions, IHostActionParams } from '../../../models';
import { HostAction, HostActionMeta } from '../models';
import {
   advancedField,
   checkField,
   disableAdminRequestsField,
   ignoreCmsField,
   ignoreMaintenanceField,
   reasonField,
   withAutoHealingField,
} from './commonFields';

import { EnumField } from '../../../rich_shared/form_fields';

interface IRebootHostParams extends IHostActionParams {
   _advanced: boolean;
   /**
    * Check host after task completion and consider it as failed if host is not healthy
    */
   check: boolean; // true by default

   /**
    * Don't issue any admin requests if something is broken - just fail the task
    */
   disable_admin_requests: boolean; // false by default

   /**
    * Don't acquire permission from CMS
    */
   ignore_cms: boolean; // false by default

   /**
    * Allow using ssh for the operation.
    */
   ssh: 'forbid' | 'only' | 'fallback'; // forbid by default

   /**
    * Try to automatically repair the host if task will fail
    */
   with_auto_healing: boolean; // false by default (for API), true for UI
}

function hasDisabledSsh(hosts: IHost[]): boolean {
   const projects = store.getState().projects.allItems.byIds;

   return hosts.some(host => {
      const hostProject = projects[host.project];

      return hostProject && !hostProject.reboot_via_ssh;
   });
}

export class RebootHostAction extends HostActionMeta<IRebootHostParams> {
   constructor() {
      super({
         action: HostAction.Reboot,
         fields: [
            {
               getHelp(params: IHostActionOptions, context: IHost[]): string | null {
                  return hasDisabledSsh(context) ? 'Rebooting via SSH is not enabled in project' : null;
               },
               isDisabled(params: IHostActionOptions, context: IHost[]): boolean {
                  return hasDisabledSsh(context);
               },
               label: 'Use SSH for operation',
               name: 'ssh',
               options: [
                  { value: 'forbid', title: 'Forbidden' },
                  { value: 'only', title: 'Only' },
                  { value: 'fallback', title: 'Fallback' },
               ],
               type: FormFieldType.Custom,
               component: EnumField,
            },

            reasonField,

            // Options
            checkField,
            withAutoHealingField,
            disableAdminRequestsField,
            ignoreMaintenanceField,

            // Advanced
            advancedField,
            {
               ...ignoreCmsField,
               isHidden: p => !p._advanced,
            },
         ],
         initialValue: {
            _advanced: false,
            check: true,
            disable_admin_requests: false,
            ignore_cms: false,
            ignore_maintenance: false,
            reason: '',
            ssh: 'forbid',
            with_auto_healing: true,
         },
         name: 'Reboot',
         titles: ['Reboot the host', 'Reboot hosts'],
      });
   }
}
