import { FormFieldType } from '@yandex-infracloud-ui/libs';

import {
   DefaultDeployConfigPolicy,
   DefaultDeployNetwork,
   DefaultProvisioner,
   DeployNetwork,
   IDeployConfig,
   IHostActionParams,
   Provisioner,
} from '../../../models';
import { HostAction, HostActionMeta } from '../models';
import {
   advancedField,
   checkField,
   deployConfigField,
   deployConfigPolicyField,
   deployNetworkField,
   deployTagsField,
   disableAdminRequestsField,
   ignoreCmsField,
   ignoreMaintenanceField,
   reasonField,
   withAutoHealingField,
} from './commonFields';

interface IRedeployHostParams extends IHostActionParams {
   _advanced: boolean;
   _deployConfig?: IDeployConfig;

   /**
    * Modifying the configuration of the redeploying host action
    */
   deploy_config_policy: string;

   /**
    * Check host after task completion and consider it as failed if host is not healthy
    */
   check: boolean; // true by default

   /**
    * Deploy config
    */
   config: string;

   /**
    * Don't issue any admin requests if something is broken - just fail the task
    */
   disable_admin_requests: boolean; // false by default

   /**
    * Don't acquire permission from CMS
    */
   ignore_cms: boolean; // false by default

   /**
    * Deploy host either in service or in project VLAN's
    */
   network: DeployNetwork;

   /**
    * Provisioning system
    */
   provisioner: Provisioner;

   /**
    * Einstellung tags to apply to host before deployment
    * "pattern": "^[a-zA-Z0-9_:-]+$",
    */
   tags: Set<string>;

   /**
    * Try to automatically repair the host if task will fail
    */
   with_auto_healing: boolean; // false by default
}

export class RedeployHostAction extends HostActionMeta<IRedeployHostParams> {
   constructor() {
      super({
         action: HostAction.Redeploy,
         fields: [
            deployConfigField,
            {
               ...deployNetworkField,
               name: 'network',
            },
            deployConfigPolicyField,
            {
               ...deployTagsField,
               name: 'tags',
            },

            reasonField,

            // Options
            checkField,
            withAutoHealingField,
            disableAdminRequestsField,
            ignoreMaintenanceField,

            // Advanced
            advancedField,
            {
               ...ignoreCmsField,
               isHidden: p => !p._advanced,
            },

            // Hidden
            { name: 'config', type: FormFieldType.Hidden },
            { name: 'provisioner', type: FormFieldType.Hidden, isHidden: p => !p.config },
         ],
         initialValue: {
            _advanced: false,
            deploy_config_policy: DefaultDeployConfigPolicy,
            check: true,
            config: '',
            disable_admin_requests: false,
            ignore_cms: false,
            ignore_maintenance: false,
            network: DefaultDeployNetwork,
            provisioner: DefaultProvisioner,
            reason: '',
            tags: new Set(),
            with_auto_healing: true,
         },
         name: 'Redeploy',
         titles: ['Redeploy the host', 'Redeploy hosts'],
      });
   }

   public transform(params: IRedeployHostParams): IRedeployHostParams {
      params.config = params._deployConfig ? params._deployConfig.config : '';
      params.provisioner = params._deployConfig ? params._deployConfig.provisioner : DefaultProvisioner;

      return params;
   }
}
