import { FormFieldType, NullableDate, toTimestamp } from '@yandex-infracloud-ui/libs';

import {
   CmsTaskAction,
   DefaultCmsTaskAction,
   DefaultOperationState,
   IHostActionParams,
   OperationState,
} from '../../../models';
import { HostAction, HostActionMeta } from '../models';
import {
   advancedField,
   cmsTaskActionField,
   disableAdminRequestsField,
   ignoreCmsField,
   operationStateField,
   reasonField,
   ticketKeyField,
   timeoutStatusField,
   timeoutTimeField,
} from './commonFields';

interface ISetMaintenanceHostParams extends IHostActionParams {
   _advanced: boolean;
   _timeoutTime: Date | null;

   /**
    * CMS action name for switching to maintenance.
    */
   cms_task_action: CmsTaskAction;

   /**
    * Don't issue any admin requests if something is broken - just fail the task
    */
   disable_admin_requests: boolean;

   /**
    * Don't acquire permission from CMS
    */
   ignore_cms: boolean;

   /**
    * Operation state of the host after switching to maintenance
    */
   operation_state: OperationState;

   /**
    * Power off the host
    */
   power_off: boolean;

   /**
    * Attach ticket to host's manual status.
    */
   ticket_key: string;

   /**
    * Status the host will be switched to at timeout_time
    */
   timeout_status: 'ready' | 'dead';

   /**
    * Time until which the specified status will be active for the host.
    *
    * When the time comes, the host will be automatically switched to the status
    * specified by timeout_status parameter.
    * It's guaranteed that status will be switched not earlier than the specified time,
    * but it's not guaranteed when exactly it will happen.
    * In practice, it will be in minutes since the specified time.
    */
   timeout_time: NullableDate | number;
}

export class SetMaintenanceHostAction extends HostActionMeta<ISetMaintenanceHostParams> {
   public readonly important: boolean = true;

   constructor() {
      super({
         action: HostAction.SetMaintenance,
         fields: [
            ticketKeyField,
            reasonField,
            timeoutTimeField,
            timeoutStatusField,
            operationStateField,
            {
               /**
                * #WALLEUI-705
                * Т.к. cmsTaskActionField используется в других actions,
                * а action TemporaryUnreachable нужен только для SetMaintenance,
                * то расширяем options только здесь
                * '...он нужен только для set-maintenance' - Alexander Presnyakov
                */
               ...cmsTaskActionField,
               options: [
                  { value: CmsTaskAction.Profile, title: 'Profile' },
                  { value: CmsTaskAction.Reboot, title: 'Reboot' },
                  { value: CmsTaskAction.TemporaryUnreachable, title: 'Temporary unreachable' },
               ],
               isHidden: p => p.ignore_cms,
            },
            {
               checkboxLabel: 'Power off the host',
               name: 'power_off',
               type: FormFieldType.Boolean,
            },
            {
               ...disableAdminRequestsField,
               isHidden: p => !p.power_off,
            },

            // Advanced
            advancedField,
            {
               ...ignoreCmsField,
               isHidden: p => !p._advanced,
            },

            // Hidden
            {
               name: 'timeout_time',
               type: FormFieldType.Hidden,
            },
         ],
         initialValue: {
            _advanced: false,
            _timeoutTime: null,
            cms_task_action: DefaultCmsTaskAction,
            disable_admin_requests: false,
            ignore_cms: false,
            ignore_maintenance: false, // Не используется
            operation_state: DefaultOperationState,
            power_off: false,
            reason: '',
            ticket_key: '',
            timeout_status: 'ready',
            timeout_time: null,
         },
         name: 'Set maintenance',
         titles: ['Set host maintenance', 'Set hosts maintenance'],
      });
   }

   public transform(params: ISetMaintenanceHostParams): ISetMaintenanceHostParams {
      params.timeout_time = params._timeoutTime instanceof Date ? toTimestamp(params._timeoutTime) : null;

      return params;
   }
}
