import { FormFieldType } from '@yandex-infracloud-ui/libs';

import { IHost, IHostActionParams } from '../../../models';
import { ProjectSelectField, RestrictionEditorField, EnumField } from '../../../rich_shared/form_fields';
import { HostAction, HostActionMeta } from '../models';
import { disableAdminRequestsField, ignoreCmsField, ignoreMaintenanceField, reasonField } from './commonFields';

interface ISwitchProjectHostParams extends IHostActionParams {
   _releaseMode: 'leave' | 'release';
   _restrictionMode: 'default' | 'manual';

   /**
    * Don't issue any admin requests if something is broken - just fail the task
    */
   disable_admin_requests: boolean; // false by default

   /**
    * Erase disks while releasing host
    */
   erase_disks: boolean; // true by default

   /**
    * Force switching without checking the projects for compatibility
    */
   force: boolean; // false by default

   /**
    * Don't acquire permission from CMS
    */
   ignore_cms: boolean; // false by default

   /**
    * Project ID to switch the host to
    */
   project: string;

   /**
    * Release the host before switching to the project
    */
   release: boolean;

   /**
    * Restrictions to assign to the host after switching
    */
   restrictions: Set<string>;
}

export class SwitchProjectHostAction extends HostActionMeta<ISwitchProjectHostParams> {
   public readonly important: boolean = false;

   constructor() {
      super({
         action: HostAction.SwitchProject,
         fields: [
            {
               component: ProjectSelectField,
               label: 'Project',
               name: 'project',
               type: FormFieldType.Custom,
            },
            {
               getError: p =>
                  p._releaseMode === 'release'
                     ? `The host will be removed from DNS${p.erase_disks ? ' and all disks will be erased' : ''}.`
                     : null,
               label: 'Before action',
               name: '_releaseMode',
               options: [
                  { value: 'leave', title: 'Leave as is' },
                  { value: 'release', title: 'Release' },
               ],
               type: FormFieldType.Custom,
               component: EnumField,
            },
            {
               isHidden: p => p._releaseMode === 'release',
               label: 'Restrictions',
               name: '_restrictionMode',
               options: [
                  { value: 'default', title: 'Default' },
                  { value: 'manual', title: 'Manual' },
               ],
               type: FormFieldType.Custom,
               component: EnumField,
            },
            {
               component: RestrictionEditorField,
               isHidden: p => p._releaseMode === 'release' || p._restrictionMode === 'default',
               name: 'restrictions',
               type: FormFieldType.Custom,
            },
            {
               checkboxLabel: 'Erase disks',
               isHidden: p => p._releaseMode !== 'release',
               name: 'erase_disks',
               type: FormFieldType.Boolean,
            },
            {
               checkboxLabel: 'Force movement',
               isHidden: p => p._releaseMode === 'release',
               name: 'force',
               type: FormFieldType.Boolean,
            },

            reasonField,

            // Options
            disableAdminRequestsField,
            ignoreCmsField,
            ignoreMaintenanceField,

            // Hidden
            { name: 'release', type: FormFieldType.Hidden },
         ],
         initialValue: {
            _releaseMode: 'leave',
            _restrictionMode: 'default',
            disable_admin_requests: false,
            erase_disks: true,
            force: false,
            ignore_cms: false,
            ignore_maintenance: false,
            project: '',
            reason: '',
            release: true,
            restrictions: new Set(),
         },
         labels: { run: 'Move' },
         name: 'Switch project',
         titles: ['Move the host to another project', 'Move the hosts to another project'],
      });
   }

   public transform(params: ISwitchProjectHostParams): ISwitchProjectHostParams {
      params.release = params._releaseMode === 'release';

      return params;
   }

   /**
    * Очищает параметры перед отправкой на сервер.
    *
    * Увы, clearParams очищает все пустые значения. В большинстве случаев это нормально.
    * Но конкретно в этом случае есть разница между задаванием пустого списка ограничений
    *    и не задаванием ограничений вообще.
    * Поэтому приходится возвращать значение обратно.
    */
   public clean(params: ISwitchProjectHostParams, context: IHost[]): Partial<ISwitchProjectHostParams> {
      const cleaned = super.clean(params, context);
      if (params._releaseMode === 'leave' && params._restrictionMode === 'manual') {
         cleaned.restrictions = params.restrictions;
      }

      return cleaned;
   }
}
