import { FormFieldType } from '@yandex-infracloud-ui/libs';

import { IHostActionParams, IVlanValue } from '../../../models';

import { createVLANsSelectField, EnumField } from '../../../rich_shared/form_fields';
import { HostAction, HostActionMeta } from '../models';
import { ignoreMaintenanceField, reasonField } from './commonFields';

enum NetworkTarget {
   Parking = 'parking',
   Project = 'project',
   Service = 'service',
}

export interface ISwitchVlansHostParams extends IHostActionParams {
   _networkTarget: NetworkTarget | 'manual';

   _vlanConfig?: IVlanValue;

   /**
    * Native VLAN ID
    */
   native_vlan?: number;

   /**
    * Target network
    */
   network_target?: NetworkTarget;

   /**
    * Allowed VLAN IDs
    */
   vlans?: Set<number>;
}

export class SwitchVlansHostAction extends HostActionMeta<ISwitchVlansHostParams> {
   public readonly important: boolean = false;

   constructor() {
      super({
         action: HostAction.SwitchVlans,
         fields: [
            {
               label: 'Target network',
               name: '_networkTarget',
               options: [
                  { value: NetworkTarget.Project, title: 'Project' },
                  { value: NetworkTarget.Service, title: 'Service' },
                  { value: NetworkTarget.Parking, title: 'Parking' },
                  { value: 'manual', title: 'Manual' },
               ],
               type: FormFieldType.Custom,
               component: EnumField,
            },
            {
               component: createVLANsSelectField({ canSelect: true }),
               getHelp: p => (!p.native_vlan ? "Are you sure you don't want to set native VLAN?" : null),
               isHidden: p => p._networkTarget !== 'manual',
               label: 'VLANs',
               name: '_vlanConfig',
               type: FormFieldType.Custom,
            },

            reasonField,

            // Options
            ignoreMaintenanceField,

            // Hidden
            { name: 'native_vlan', type: FormFieldType.Hidden, isHidden: p => p._networkTarget !== 'manual' },
            { name: 'vlans', type: FormFieldType.Hidden, isHidden: p => p._networkTarget !== 'manual' },
            { name: 'network_target', type: FormFieldType.Hidden, isHidden: p => p._networkTarget === 'manual' },
         ],
         initialValue: {
            _networkTarget: NetworkTarget.Project,
            ignore_maintenance: false,
            network_target: NetworkTarget.Project,
            reason: '',
            vlans: new Set(),
         },
         name: 'Switch VLANs',
         titles: ['Switch VLANs for the host', 'Switch VLANs for hosts'],
      });
   }

   public transform(params: ISwitchVlansHostParams): ISwitchVlansHostParams {
      if (params._networkTarget === 'manual') {
         if (params._vlanConfig) {
            params.native_vlan = params._vlanConfig.selected;
            params.vlans = params._vlanConfig.items;
         }

         delete params.network_target;

         return params;
      }

      delete params.native_vlan;
      delete params._vlanConfig;
      params.vlans = new Set();
      params.network_target = params._networkTarget;

      return params;
   }
}
