import { FormFieldType, IFormField, isEmpty } from '@yandex-infracloud-ui/libs';

import { CmsTaskAction, DeployNetwork, IHost, IHostActionOptions, OperationState, Provisioner } from '../../../models';
import {
   DeployConfigField,
   DeployConfigPolicyField,
   DeployTagsInputField,
   EnumField,
   AdvancedField,
} from '../../../rich_shared/form_fields';

// region Host action options (boolean)

export const advancedField: IFormField<any> = {
   component: AdvancedField,
   label: 'Advanced options',
   name: '_advanced',
   type: FormFieldType.Custom,
};

export const checkField: IFormField<IHostActionOptions> = {
   checkboxLabel: 'Check after completion',
   isHidden: () => false,
   name: 'check',
   type: FormFieldType.Boolean,
};

export const disableAdminRequestsField: IFormField<IHostActionOptions> = {
   checkboxLabel: "Don't create tickets for DC engineers",
   getHelp: () => "Don't issue any admin requests if something is broken - just fail the task",
   // hidden if !check || !with_auto_healing
   isHidden: (p: IHostActionOptions) => {
      const hiddenIfNotCheck = withAutoHealingField.isHidden!(p);

      const hiddenIfNotAutoHealing = 'with_auto_healing' in p ? !p.with_auto_healing : false;

      return hiddenIfNotAutoHealing || hiddenIfNotCheck;
   },
   name: 'disable_admin_requests',
   type: FormFieldType.Boolean,
};

export const ignoreCmsField: IFormField<IHostActionOptions> = {
   checkboxLabel: 'Ignore CMS',
   getHelp: () => "Don't acquire permission from CMS",
   isHidden: () => false,
   name: 'ignore_cms',
   type: FormFieldType.Boolean,
};

export const ignoreMaintenanceField: IFormField<IHostActionOptions> = {
   checkboxLabel: 'Ignore maintenance',
   isHidden: (p: IHostActionOptions, context?: IHost[]) =>
      context ? !context.some(h => h.state === 'maintenance') : false,
   name: 'ignore_maintenance',
   type: FormFieldType.Boolean,
};

export const withAutoHealingField: IFormField<IHostActionOptions> = {
   checkboxLabel: 'Enable Wall-E automated host healing for this task',
   isHidden: p => ('check' in p ? !p.check : false),
   name: 'with_auto_healing',
   type: FormFieldType.Boolean,
};

export const eraseDiskField: IFormField<IHostActionOptions> = {
   checkboxLabel: 'Erase disks',
   isHidden: p => ('check' in p ? !p.check : false),
   name: 'erase_disks',
   type: FormFieldType.Boolean,
};

// endregion

function hasDeployConfig(p: any): boolean {
   if ('deploy_config' in p && !isEmpty(p.deploy_config)) {
      return true;
   }

   // noinspection RedundantIfStatementJS
   if ('config' in p && !isEmpty(p.config)) {
      return true;
   }

   return false;
}

export const deployConfigField: IFormField<any> = {
   component: DeployConfigField,
   getHelp: (params: any) =>
      params._deployConfig ? null : "If empty, the project's default deploy config will be used",
   label: 'Deploy config',
   name: '_deployConfig',
   placeholder: 'Deploy config',
   type: FormFieldType.Custom,
   required: true,
};

export const deployConfigPolicyField: IFormField<any> = {
   component: DeployConfigPolicyField,
   isHidden: p => !hasDeployConfig(p),
   label: 'Deploy config policy',
   name: 'deploy_config_policy',
   type: FormFieldType.Custom,
};

export const deployNetworkField: IFormField<any> = {
   getHelp: () => "Deploy host either in service or in project VLAN's",
   // deploy_network – часть настроек наливки, надо показывать только если выбран provisioner=lui
   isHidden: p => !hasDeployConfig(p) || p.provisioner !== Provisioner.LUI,
   label: 'Network',
   name: 'deploy_network',
   options: [
      { value: DeployNetwork.Project, title: 'Project' },
      { value: DeployNetwork.Service, title: 'Service' },
   ],
   type: FormFieldType.Custom,
   component: EnumField,
};

export const deployTagsField: IFormField<any> = {
   component: DeployTagsInputField,
   getHelp: () => 'Einstellung tags to apply to host before deployment',
   // deploy_tags – часть настроек наливки, надо показывать только если выбран provisioner=eine
   isHidden: p => !hasDeployConfig(p) || p.provisioner !== Provisioner.EINE,
   label: 'Deploy tags',
   name: 'deploy_tags',
   placeholder: 'Enter deploy tags',
   type: FormFieldType.Custom,
};

export const reasonField: IFormField<any> = {
   label: 'Reason',
   name: 'reason',
   placeholder: 'Reason (optional)',
   type: FormFieldType.Text,
};

export const operationStateField: IFormField<any> = {
   getHelp: () => `Host operation state for this maintenance`,
   label: 'Operation state',
   name: 'operation_state',
   options: [
      { value: OperationState.Operation, title: 'Operation' },
      { value: OperationState.Decommissioned, title: 'Decommissioned' },
   ],
   type: FormFieldType.Custom,
   component: EnumField,
};

export const cmsTaskActionField: IFormField<any> = {
   getHelp: () => `CMS action for this maintenance`,
   label: 'CMS task action',
   name: 'cms_task_action',
   options: [
      { value: CmsTaskAction.Profile, title: 'Profile' },
      { value: CmsTaskAction.Reboot, title: 'Reboot' },
   ],
   type: FormFieldType.Custom,
   component: EnumField,
};

export const timeoutTimeField: IFormField<any> = {
   getHelp: () =>
      `If specified, host will be switched to 'assigned' state at requested time regardless of the status of the ticket`,
   hint: `Time until which the specified status will be active for the host.
When the time comes, the host will be automatically switched to the status specified by timeout_status parameter.
It's guaranteed that status will be switched not earlier than the specified time, but it's not guaranteed when exactly it will happen.
In practice, it will be in minutes since the specified time.`,
   label: 'Timeout time',
   name: '_timeoutTime',
   type: FormFieldType.DateTime,
};

export const timeoutStatusField: IFormField<any> = {
   getHelp: () => `Assign specific status to host after this maintenance`,
   label: 'Timeout status',
   name: 'timeout_status',
   options: [
      { value: 'ready', title: 'Ready' },
      { value: 'dead', title: 'Dead' },
   ],
   type: FormFieldType.Custom,
   component: EnumField,
};

export const ticketKeyField: IFormField<any> = {
   getHelp: () => `Host will be switched to 'assigned' state when this ticket is closed, unless timeout is specified`,
   label: 'Ticket key',
   name: 'ticket_key',
   required: true,
   type: FormFieldType.String,
};
