import { toggleSetItem } from '@yandex-infracloud-ui/libs';

import { HostState, IHost } from '../../../models';
import { HostAction, HostActionMeta } from '../models';

import { CancelTaskHostAction } from './CancelTaskHostAction';
import { ChangeMaintenanceHostAction } from './ChangeMaintenanceHostAction';
import { CheckDNSHostAction } from './CheckDNSHostAction';
import { CreateScenarioHostAction } from './CreateScenarioHostAction';
import { DeployConfigHostAction } from './DeployConfigHostAction';
import { EditExtraVlansHostAction } from './EditExtraVlansHostAction';
import { ForceStatusHostAction } from './ForceStatusHostAction';
import { FqdnDeinvalidationHost } from './FqdnDeinvalidationHost';
import { PowerOffHostAction } from './PowerOffHostAction';
import { PowerOnHostAction } from './PowerOnHostAction';
import { PrepareHostAction } from './PrepareHostAction';
import { ProfileHostAction } from './ProfileHostAction';
import { RebootHostAction } from './RebootHostAction';
import { RedeployHostAction } from './RedeployHostAction';
import { RemoveHostAction } from './RemoveHostAction';
import { SetAssignedHostAction } from './SetAssignedHostAction';
import { SetMaintenanceHostAction } from './SetMaintenanceHostAction';
import { SwitchProjectHostAction } from './SwitchProjectHostAction';
import { SwitchVlansHostAction } from './SwitchVlansHostAction';
import { ReleaseHostAction } from './ReleaseHostAction';
import { CreateTicketHostAction } from './CreateTicketHostAction';

export const hostActions: HostActionMeta<any>[] = [
   new CreateTicketHostAction(),

   new RebootHostAction(),
   new ProfileHostAction(),
   new SetAssignedHostAction(),
   new SetMaintenanceHostAction(),
   new RedeployHostAction(),
   new CancelTaskHostAction(),

   // -------- unimportant ----------
   new PowerOffHostAction(),
   new PowerOnHostAction(),
   new DeployConfigHostAction(),
   new PrepareHostAction(),
   new ReleaseHostAction(),
   new CheckDNSHostAction(),
   new ChangeMaintenanceHostAction(),
   new RemoveHostAction(),
   new SwitchProjectHostAction(),
   new SwitchVlansHostAction(),
   new EditExtraVlansHostAction(),
   new FqdnDeinvalidationHost(),
   new ForceStatusHostAction(),
   new CreateScenarioHostAction(),
];

const allActions = new Set(hostActions.map(a => a.action as HostAction));

/**
 * Возвращает список доступных действий над хостом
 *
 * Описание алгоритма (после он был чутка доработан):
 * Если есть таск
 *    и хост не в free, то можно только
 *       force status,
 *       set maintenance,
 *       change deploy config
 *       configure vlans.
 *    Если хост во free, то только
 *       force-status.
 *
 * Если таска нет,
 *    то для free только
 *       prepare и delete,
 *    для assigned
 *       все кроме set assigned,
 *    для maintenance
 *       все,
 *       но set maintenance заменить на change maintenance
 */
export function getAvailableHostActions(hasTask: boolean, state: HostState): Set<HostAction> {
   switch (state) {
      case HostState.Free:
         return new Set(
            hasTask
               ? [HostAction.CancelTask, HostAction.ForceStatus]
               : [HostAction.Prepare, HostAction.Remove, HostAction.SwitchProject],
         );

      case HostState.Assigned:
         if (hasTask) {
            return new Set([
               HostAction.CancelTask,
               HostAction.DeployConfig,
               HostAction.EditExtraVlans,
               HostAction.SetMaintenance,
               HostAction.ForceStatus,
            ]);
         } else {
            let actions = allActions;
            actions = toggleSetItem(actions, HostAction.SetAssigned);
            actions = toggleSetItem(actions, HostAction.ChangeMaintenance);
            actions = toggleSetItem(actions, HostAction.CancelTask);
            actions = toggleSetItem(actions, HostAction.Prepare);

            return actions;
         }

      case HostState.Maintenance:
         if (hasTask) {
            return new Set([
               HostAction.CancelTask,
               HostAction.ChangeMaintenance,
               HostAction.DeployConfig,
               HostAction.EditExtraVlans,
               HostAction.ForceStatus,
            ]);
         } else {
            let actions = allActions;
            actions = toggleSetItem(actions, HostAction.SetMaintenance);
            actions = toggleSetItem(actions, HostAction.CancelTask);
            actions = toggleSetItem(actions, HostAction.Prepare);

            return actions;
         }

      default:
         // Return all actions in case new state has been added on backend
         // we do not want to restrict actions that are allowed, so just show all.
         return allActions;
   }
}

/**
 * Проверяет, можно ли для хоста выполнить конкретное действие
 *
 * Неоптимальный вариант, но не хотелось бы нарушать логику в getAvailableHostAction,
 * т.к. она в таком виде лучше кладётся в голову.
 *
 * Альтернативный вариант с предикатом для каждого HostAction будет сложнее читать и править.
 *
 * TODO добавить мемоизацию
 */
export function canCallAction(action: HostAction, host: IHost): boolean {
   if (action === HostAction.FqdnDeinvalidation) {
      return host.status === 'invalid';
   }

   // TODO проверять права пользователя и может что-нибудь еще
   if (action === HostAction.CreateScenario) {
      return true;
   }

   return getAvailableHostActions(Boolean(host.task), host.state).has(action);
}
