import {
   DeclarativeFields,
   DISMISS_REASON,
   ExtendedFieldConfig,
   FormHooks,
   FormHooksContext,
   IModalProps,
   ModalLayout,
   setAsArray,
   SetField,
   toasts,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { Button } from 'lego-on-react';
import React, { PropsWithChildren, useCallback, useMemo, useRef, useState } from 'react';
import { Form, Formik, FormikProps } from 'formik';
import classes from '../../../design/commonModals.module.css';

import { IHostActionProps } from '../models';
import { DevJson, TrackerLink } from 'shared';
import { TextInputField, QueueConstantField } from 'rich_shared/form_fields2';
import { object, string } from 'yup';
import { hostApi } from 'services';
import { HostAction } from '..';
import { takeUntil } from 'rxjs/operators';
import { useOpenModal } from 'rich_shared/scenario_list/components/config';

interface TicketFormFields {
   summary: string;
   queue: string;
   description?: string;
   hosts?: string[];
}

interface Props extends IModalProps<TicketFormFields> {
   initialValue?: Partial<TicketFormFields>;
}

interface TicketLinkModalProps extends IModalProps<{}> {
   link: string;
}

const shape = {
   summary: string().required('This is a required field').nullable(),
   queue: string().required('This is a required field').nullable(),
   description: string().nullable(),
   hosts: setAsArray().of(string()),
};

const schema = object().shape(shape);

const fields: ExtendedFieldConfig[] = [
   {
      as: TextInputField,
      label: 'Summary',
      name: 'summary',
      placeholder: 'Enter ticket summary',
      required: true,
   },
   {
      as: QueueConstantField,
      label: 'Queue',
      name: 'queue',
      placeholder: 'Enter queue for ticket',
      required: true,
   },
   {
      as: SetField,
      label: 'Hosts',
      name: 'hosts',
      placeholder: 'Enter host name',
      required: true,
   },
   {
      as: TextInputField,
      label: 'Description',
      name: 'description',
      placeholder: 'Some description for ticket (Optinal)',
      controlProps: {
         multiline: true,
         rows: 4,
      },
   },
];

const InternalForm: React.FC<PropsWithChildren<{}>> = ({ children }) => (
   <Form className={classes.form}>
      <DeclarativeFields configs={fields} />

      {children}
   </Form>
);

const initialValueBase: TicketFormFields = {
   summary: '',
   queue: '',
   description: '',
   hosts: [],
};

const TicketLinkModal = ({ link, ok, cancel }: TicketLinkModalProps) => {
   const onDismiss = () => cancel(DISMISS_REASON);

   return (
      <ModalLayout onDismiss={onDismiss} title={'Ticket has been created'} showFooter={false}>
         <TrackerLink ticket={link} />
      </ModalLayout>
   );
};

export const CreateTicketModal = React.memo(({ initialValue, ok, cancel }: Props) => {
   const [apiError, setApiError] = useState('');
   const dismounted = useDismounted();
   const { openModal } = useOpenModal();

   const initialValuePatched = useMemo(() => ({ ...initialValueBase, ...initialValue }), [initialValue]);

   const formRef = useRef<FormikProps<TicketFormFields>>();

   const formHooks = useMemo(
      () =>
         ({
            onFieldChange: (name, value) => {
               setApiError('');

               if (!formRef.current) {
                  return;
               }
            },
         } as FormHooks<TicketFormFields>),
      [formRef],
   );

   const onDismiss = () => cancel(DISMISS_REASON);

   const onSubmit = useCallback(
      (form: TicketFormFields) => {
         hostApi
            .actionWithMultiHosts(HostAction.CreateTicket, form)
            .pipe(takeUntil(dismounted))
            .subscribe(resp => {
               const { link } = resp;

               ok(form);
               openModal(TicketLinkModal, { link });
            }, toasts.handleApiError('Ticket creating'));
      },
      [dismounted, ok, openModal],
   );

   return (
      <ModalLayout title={'Create ticket'} showFooter={false} onDismiss={onDismiss}>
         <Formik initialValues={initialValuePatched} onSubmit={onSubmit} validationSchema={schema}>
            {form => {
               formRef.current = form;

               return (
                  <FormHooksContext.Provider value={formHooks}>
                     <InternalForm>
                        <DevJson>{{ form }}</DevJson>

                        <footer className={classes.footer}>
                           {apiError && !form.isSubmitting ? <span className={classes.error}>{apiError}</span> : null}

                           <Button
                              theme={'clear'}
                              size={'s'}
                              cls={classes.button}
                              onClick={onDismiss}
                              disabled={form.isSubmitting}
                           >
                              Cancel
                           </Button>

                           <Button
                              theme={'action'}
                              size={'s'}
                              cls={classes.button}
                              type={'submit'}
                              disabled={!form.isValid || form.isSubmitting || !form.dirty}
                           >
                              Create ticket
                           </Button>
                        </footer>
                     </InternalForm>
                  </FormHooksContext.Provider>
               );
            }}
         </Formik>
      </ModalLayout>
   );
});

export const CreateTicketHostModal: React.FC<IHostActionProps> = React.memo(({ onRun, onCancel, context, meta }) => {
   const initialValue = useMemo(
      () => ({
         hosts: context.map(item => item.name),
      }),
      [context],
   );

   const handleOk = useCallback(() => onRun(null, []), [onRun]);

   const handleCancel = useCallback(() => onCancel(null), [onCancel]);

   return <CreateTicketModal initialValue={initialValue} ok={handleOk} cancel={handleCancel} />;
});
