import { IClonable, IEqualable, IJsonable } from '@yandex-infracloud-ui/libs';

export type ShortPeriod = 's' | 'm' | 'h' | 'd' | 'w';

interface IPeriod {
   letter: ShortPeriod;
   one: string;
   several: string;
}

export const PERIOD_LIST: IPeriod[] = [
   { letter: 's', one: 'second', several: 'seconds' },
   { letter: 'm', one: 'minute', several: 'minutes' },
   { letter: 'h', one: 'hour', several: 'hours' },
   { letter: 'd', one: 'day', several: 'days' },
   { letter: 'w', one: 'week', several: 'weeks' },
];

const PERIODS: Map<string, IPeriod> = new Map(PERIOD_LIST.map(i => [i.letter, i] as [string, IPeriod]));

/**
 * Простенькая обёртка для хранения значения лимита автоматизации. Иммутабельный класс.
 *
 * Для каждого вида лимита можно задать несколько пар период-количество,
 * для которых затем будут проверяться лимиты.
 *
 * В качестве периода используется строка вида $number(s|m|h|d|w),
 * суффикс которой указывает на единицу измерения:
 *    s - секунды
 *    m - минуты
 *    h - часы
 *    d - дни
 *    w - недели
 */
// tslint:disable-next-line
export class AutomationLimit implements IClonable<AutomationLimit>, IEqualable<AutomationLimit>, IJsonable {
   public readonly period: ShortPeriod;

   public readonly periodValue: number;

   public readonly times: number;

   constructor(times: number, fullPeriod: string) {
      this.times = times;
      if (!(_isInteger(this.times) && this.times >= 0)) {
         throw new Error('Limit must be not negative integer');
      }

      this.period = fullPeriod.slice(-1) as ShortPeriod;
      if (!PERIODS.has(this.period)) {
         throw new Error('Period must be in $number(s|m|h|d|w) format');
      }

      this.periodValue = parseInt(fullPeriod.slice(0, -1), 10);
      if (!(_isInteger(this.periodValue) && this.periodValue > 0)) {
         throw new Error('Period value must be positive integer');
      }
   }

   public updateTimes(value: number): AutomationLimit {
      return new AutomationLimit(value, this.getFullPeriod());
   }

   public updatePeriodValue(value: number): AutomationLimit {
      return new AutomationLimit(this.times, `${value}${this.period}`);
   }

   public updatePeriodPeriod(value: ShortPeriod): AutomationLimit {
      return new AutomationLimit(this.times, `${this.periodValue}${value}`);
   }

   public updateFullPeriod(value: string): AutomationLimit {
      return new AutomationLimit(this.times, value);
   }

   public preposition(skipArticle = false): string {
      return !skipArticle && this.periodValue === 1 ? 'in a' : 'in';
   }

   public getFullPeriod(): string {
      return `${this.periodValue}${this.period}`;
   }

   public prepositionWithTimes(skipArticle = false): string {
      return `${this.times === 1 ? 'time' : 'times'} ${this.preposition(skipArticle)}`;
   }

   public toString() {
      const period = PERIODS.get(this.period)!;
      const periodStr = this.periodValue === 1 ? period.one : `${this.periodValue} ${period.several}`;

      return `${this.times} ${this.preposition()} ${periodStr}`;
   }

   public toJSON() {
      return { limit: this.times, period: this.getFullPeriod() };
   }

   public clone(): AutomationLimit {
      return new AutomationLimit(this.times, this.getFullPeriod());
   }

   public isEqual(obj: AutomationLimit): boolean {
      return this.times === obj.times && this.getFullPeriod() === obj.getFullPeriod();
   }
}

function _isInteger(value: number): boolean {
   return !isNaN(value) && parseInt(value.toString(), 10) === value;
}
